/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;


import com.sapportals.wcm.util.jdbc.escape.Escape;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * Merge ACLs from all ACL managers in the tables of one connection pool (merge
 * framework and service ACLs). The implementation copes with framework ACLs and
 * service ACLs which reside in the same pool (which is the case for most oracle
 * installations).
 */
public final class JDBCMergeAclsNoConfig {
  /**
   * hard-wired destination connection pool
   */
  public final static String DEFAULT_DESTINATION_CONNECTION_POOL = "dbcon_acl";

  /**
   * hard-wired ACL managers
   */
  private final static String FRAMEWORK_ACL_MANAGER_ID = "acl_jdbc_rep_mgr";
  private final static String SERVICE_ACL_MANAGER_ID = "acl_jdbc_service_mgr";

  /**
   * old table names
   */
  private final static String OLD_ACL_ID_MAPPING_TABLE_NAME = "wcm_acl_id_mapping";
  private final static String OLD_ACL_TABLE_NAME = "wcm_acl_acl";
  private final static String OLD_ACL_ENTRY_TABLE_NAME = "wcm_acl_acl_entry";
  private final static String OLD_ACL_ENTRY_ID_TABLE_NAME = "wcm_acl_acl_entry_id";
  private final static String OLD_OWNER_TABLE_NAME = "wcm_acl_owner";
  private final static String OLD_PERMISSION_TABLE_NAME = "wcm_acl_perm";
  private final static String OLD_SUPPORTED_PERMISSION_TABLE_NAME = "wcm_acl_supp_perm";
  private final static String OLD_PERMISSION_MEMBER_TABLE_NAME = "wcm_acl_perm_member";

  /**
   * new table names
   */
  private final static String NEW_ACL_TABLE_NAME = "wcm_app_acl_acl";
  private final static String NEW_ACL_ENTRY_TABLE_NAME = "wcm_app_acl_acl_entry";
  private final static String NEW_OWNER_TABLE_NAME = "wcm_app_acl_owner";
  private final static String NEW_PERMISSION_TABLE_NAME = "wcm_app_acl_perm";
  private final static String NEW_SUPPORTED_PERMISSION_TABLE_NAME = "wcm_app_acl_supp_perm";
  private final static String NEW_PERMISSION_MEMBER_TABLE_NAME = "wcm_app_acl_perm_member";

  /**
   * column names
   */
  private final static String EXTERNAL_ACL_ID_COL = "external";
  private final static String INTERNAL_ACL_ID_COL = "internal";
  private final static String ACL_ID_COL = "acl_id";
  private final static String OBJECT_TYPE_COL = "object_type";
  private final static String PRINCIPAL_NAME_COL = "principal_name";
  private final static String PRINCIPAL_TYPE_COL = "principal_type";
  private final static String LOCKING_USER_COL = "locking_user";
  private final static String SORT_INDEX_COL = "sort_index";
  private final static String IS_NEGATIVE_COL = "is_negative";
  private final static String IS_PROPAGATED_COL = "is_propagated";
  private final static String PERMISSION_NAME_COL = "permission_name";
  private final static String IS_PREDEFINED_COL = "is_predefined";
  private final static String PERMISSION_MEMBER_COL = "member_name";

  /**
   * column sizes
   */
  private final static int APP_ID_LENGTH = 127;
  private final static int MANAGER_ID_LENGTH = 127;

  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.acl.jdbc.JDBCMergeAclsNoConfig.class);

  private String m_appID;
  private int m_ownerID = -2;


  public JDBCMergeAclsNoConfig() { }

  public void merge(String appID, String aclManagerID, Connection oldAclManagerConnection, Connection destinationConnection)
    throws Exception {
    m_appID = appID;

    // if the old ACL tables exist, merge with the new tables
    if (existOldTables(oldAclManagerConnection)) {

      // copy the ACLs
      m_ownerID = populateManagerApplication(destinationConnection, aclManagerID, m_appID);
      copyTables(aclManagerID, oldAclManagerConnection, destinationConnection, m_ownerID);

      // remove the old ACL manager's tables
      removeOldTables(oldAclManagerConnection);
    }
  }


  public static boolean existOldTables(Connection oldAclManagerConnection)
    throws Exception {
    boolean result = false;
    try {
      // check only one table
      String sql = "SELECT * FROM " + OLD_ACL_ENTRY_ID_TABLE_NAME;
      SqlResult queryResult = query(oldAclManagerConnection, sql);
      if (queryResult != null) {
        result = true;
        queryResult.close();
      }
    }
    catch (Exception e) {
                  //$JL-EXC$
      // ignore
      s_log.debugT(e.getMessage());
    }
    return result;
  }

  public static boolean existNewTables(Connection newAclManagerConnection)
    throws Exception {
    boolean result = false;
    try {
      // check only one table
      String sql = "SELECT * FROM " + NEW_ACL_TABLE_NAME;
      SqlResult queryResult = query(newAclManagerConnection, sql);
      if (queryResult != null) {
        result = true;
        queryResult.close();
      }
    }
    catch (Exception e) {
            //$JL-EXC$      
      // ignore
      s_log.debugT(e.getMessage());      
    }
    return result;
  }

  public static void removeOldTables(Connection oldAclManagerConnection)
    throws Exception {

    removeTable(oldAclManagerConnection, OLD_ACL_ID_MAPPING_TABLE_NAME);
    removeTable(oldAclManagerConnection, OLD_ACL_TABLE_NAME);
    removeTable(oldAclManagerConnection, OLD_ACL_ENTRY_TABLE_NAME);
    removeTable(oldAclManagerConnection, OLD_ACL_ENTRY_ID_TABLE_NAME);
    removeTable(oldAclManagerConnection, OLD_OWNER_TABLE_NAME);
    removeTable(oldAclManagerConnection, OLD_PERMISSION_TABLE_NAME);
    removeTable(oldAclManagerConnection, OLD_SUPPORTED_PERMISSION_TABLE_NAME);
    removeTable(oldAclManagerConnection, OLD_PERMISSION_MEMBER_TABLE_NAME);
  }

  private static void removeTable(Connection oldAclManagerConnection, String tableName)
    throws Exception {
    String sql = "DROP TABLE " + tableName;
    update(oldAclManagerConnection, sql);
  }

  public static void copyTables(String aclManagerID, Connection oldAclManagerConnection,
    Connection destinationConnection, int m_ownerID)
    throws Exception {
    copyAcls(aclManagerID, oldAclManagerConnection, destinationConnection, m_ownerID);
    copyAclEntries(aclManagerID, oldAclManagerConnection, destinationConnection, m_ownerID);
    copyAclOwners(aclManagerID, oldAclManagerConnection, destinationConnection, m_ownerID);
    copyPermissions(aclManagerID, oldAclManagerConnection, destinationConnection, m_ownerID);
    copySupportedPermissions(aclManagerID, oldAclManagerConnection, destinationConnection, m_ownerID);
    copyPermissionMembers(aclManagerID, oldAclManagerConnection, destinationConnection, m_ownerID);
    changeAclIdMappings(aclManagerID, oldAclManagerConnection, destinationConnection);
  }

  public static int populateManagerApplication(Connection destinationConnection, String aclManagerID, String applID)
    throws Exception {
    applID = trim(Escape.escapeSqlString(applID, APP_ID_LENGTH));
    aclManagerID = trim(Escape.escapeSqlString(aclManagerID, MANAGER_ID_LENGTH));

    SqlResult queryResult = null;
    try {
      PreparedStatement st = destinationConnection.prepareCall("SELECT ID FROM wcm_app_acl_manager_appl WHERE app=? AND manager=?");
      st.setString(1, applID);
      st.setString(2, aclManagerID);
      queryResult = query(destinationConnection, st);
      if (queryResult != null) {
        if (queryResult.m_result.next()) {
          return queryResult.m_result.getInt(1);
        }
        queryResult.close();
      }

      st = destinationConnection.prepareCall("INSERT INTO wcm_app_acl_manager_appl (manager, app) VALUES (?,?)");
      st.setString(1, aclManagerID);
      st.setString(2, applID);
      update(destinationConnection, st);

      st = destinationConnection.prepareCall("SELECT ID FROM wcm_app_acl_manager_appl WHERE app=? AND manager=?");
      st.setString(1, applID);
      st.setString(2, aclManagerID);
      queryResult = query(destinationConnection, st);
      if (queryResult != null) {
        if (queryResult.m_result.next()) {
          return queryResult.m_result.getInt(1);
        }
        queryResult.close();
      }
      return -1;
    }
    finally {
      if (queryResult != null) {
        queryResult.close();
      }
    }
  }

  private static void changeAclIdMappings(String aclManagerID, Connection oldAclManagerConnection, Connection destinationConnection)
    throws Exception {

    String querySQL = "SELECT * FROM " + OLD_ACL_ID_MAPPING_TABLE_NAME;
    SqlResult queryResult = query(oldAclManagerConnection, querySQL);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          String externalAclID = trim(queryResult.m_result.getString(EXTERNAL_ACL_ID_COL));
          String internalAclID = trim(queryResult.m_result.getString(INTERNAL_ACL_ID_COL));

          updateRIDs(destinationConnection, "wcm_app_acl_acl", internalAclID, externalAclID);
          updateRIDs(destinationConnection, "wcm_app_acl_acl_entry", internalAclID, externalAclID);
          updateRIDs(destinationConnection, "wcm_app_acl_owner", internalAclID, externalAclID);
        }
      }
      finally {
        queryResult.close();
      }
    }
  }

  private static void updateRIDs(Connection con, String table, String oldRID, String newRID)
    throws Exception {
    String updateSQL = "UPDATE " + table + " SET Hash=?, RID=?, RIDStart=? WHERE RIDStart=?";
    PreparedStatement st = con.prepareCall(updateSQL);
    st.setInt(1, newRID.hashCode());
    st.setString(2, newRID);
    st.setString(3, (newRID.length() > 900) ? newRID.substring(0, 900) : newRID);
    st.setString(4, oldRID);
    update(con, st);
  }

  private static void copyAcls(String aclManagerID, Connection oldAclManagerConnection,
    Connection destinationConnection, int m_ownerID)
    throws Exception {

    String querySQL = "SELECT * FROM " + OLD_ACL_TABLE_NAME;
    SqlResult queryResult = query(oldAclManagerConnection, querySQL);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          String aclID = trim(queryResult.m_result.getString(ACL_ID_COL));
          if (aclID.startsWith("=")) {
            aclID = aclID.substring(1);
          }
          String objectType = trim(queryResult.m_result.getString(OBJECT_TYPE_COL));
          String lockingUser = trim(queryResult.m_result.getString(LOCKING_USER_COL));

          String updateSQL = "INSERT INTO " + NEW_ACL_TABLE_NAME
             + " (OwnerID, Hash, RID, RIDStart, object_type, locking_user, change_level) VALUES (?,?,?,?,?,?,?)";
          PreparedStatement st = destinationConnection.prepareCall(updateSQL);
          st.setInt(1, m_ownerID);
          st.setInt(2, aclID.hashCode());
          st.setString(3, aclID);
          st.setString(4, aclID);
          st.setString(5, objectType);
          st.setString(6, lockingUser);
          st.setInt(7, 0);
          update(destinationConnection, st);
        }
      }
      finally {
        queryResult.close();
      }
    }
  }

  private static void copyAclEntries(String aclManagerID, Connection oldAclManagerConnection,
    Connection destinationConnection, int m_ownerID)
    throws Exception {

    String querySQL = "SELECT * FROM " + OLD_ACL_ENTRY_TABLE_NAME;
    SqlResult queryResult = query(oldAclManagerConnection, querySQL);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          String aclID = trim(queryResult.m_result.getString(ACL_ID_COL));
          if (aclID.startsWith("=")) {
            aclID = aclID.substring(1);
          }
          String principalName = trim(queryResult.m_result.getString(PRINCIPAL_NAME_COL));
          int principalType = queryResult.m_result.getInt(PRINCIPAL_TYPE_COL);
          int sortIndex = queryResult.m_result.getInt(SORT_INDEX_COL);
          int isNegative = queryResult.m_result.getInt(IS_NEGATIVE_COL);
          int isPropagated = queryResult.m_result.getInt(IS_PROPAGATED_COL);
          String permissionName = trim(queryResult.m_result.getString(PERMISSION_NAME_COL));

          // filter unsupported permissions
          if (isPermissionSupportedForManager(aclManagerID, permissionName)) {
            String updateSQL = "INSERT INTO " + NEW_ACL_ENTRY_TABLE_NAME
               + " (OwnerID, Hash, RID, RIDStart, principal_name, principal_type, sort_index, is_negative, is_propagated, permission_name) "
               + "VALUES (?,?,?,?,?,?,?,?,?,?)";
            PreparedStatement st = destinationConnection.prepareCall(updateSQL);
            st.setInt(1, m_ownerID);
            st.setInt(2, aclID.hashCode());
            st.setString(3, aclID);
            st.setString(4, aclID);
            st.setString(5, principalName);
            st.setInt(6, principalType);
            st.setInt(7, sortIndex);
            st.setInt(8, isNegative);
            st.setInt(9, isPropagated);
            st.setString(10, permissionName);
            update(destinationConnection, st);
          }
        }
      }
      finally {
        queryResult.close();
      }
    }
  }


  private static void copyAclOwners(String aclManagerID, Connection oldAclManagerConnection,
    Connection destinationConnection, int m_ownerID)
    throws Exception {

    String querySQL = "SELECT * FROM " + OLD_OWNER_TABLE_NAME;
    SqlResult queryResult = query(oldAclManagerConnection, querySQL);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          String aclID = trim(queryResult.m_result.getString(ACL_ID_COL));
          if (aclID.startsWith("=")) {
            aclID = aclID.substring(1);
          }
          String principalName = trim(queryResult.m_result.getString(PRINCIPAL_NAME_COL));
          int principalType = queryResult.m_result.getInt(PRINCIPAL_TYPE_COL);

          String updateSQL = "INSERT INTO " + NEW_OWNER_TABLE_NAME
             + " (OwnerID, Hash, RID, RIDStart, principal_name, principal_type)"
             + " VALUES (?,?,?,?,?,?)";
          PreparedStatement st = destinationConnection.prepareCall(updateSQL);
          st.setInt(1, m_ownerID);
          st.setInt(2, aclID.hashCode());
          st.setString(3, aclID);
          st.setString(4, aclID);
          st.setString(5, principalName);
          st.setInt(6, principalType);
          update(destinationConnection, st);
        }
      }
      finally {
        queryResult.close();
      }
    }
  }

  private static void copyPermissions(String aclManagerID, Connection oldAclManagerConnection,
    Connection destinationConnection, int m_ownerID)
    throws Exception {

    String querySQL = "SELECT * FROM " + OLD_PERMISSION_TABLE_NAME;
    SqlResult queryResult = query(oldAclManagerConnection, querySQL);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          String permissionName = trim(queryResult.m_result.getString(PERMISSION_NAME_COL));
          int isPredefined = queryResult.m_result.getInt(IS_PREDEFINED_COL);

          // filter unsupported permissions
          if (isPermissionSupportedForManager(aclManagerID, permissionName)) {
            String updateSQL = "INSERT INTO " + NEW_PERMISSION_TABLE_NAME
               + " (OwnerID, permission_name, is_predefined) VALUES (?,?,?)";
            PreparedStatement st = destinationConnection.prepareCall(updateSQL);
            st.setInt(1, m_ownerID);
            st.setString(2, permissionName);
            st.setInt(3, isPredefined);
            update(destinationConnection, st);
          }
        }
      }
      finally {
        queryResult.close();
      }
    }
  }

  private static void copySupportedPermissions(String aclManagerID, Connection oldAclManagerConnection,
    Connection destinationConnection,
    int m_ownerID)
    throws Exception {

    String querySQL = "SELECT * FROM " + OLD_SUPPORTED_PERMISSION_TABLE_NAME;
    SqlResult queryResult = query(oldAclManagerConnection, querySQL);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          String permissionName = trim(queryResult.m_result.getString(PERMISSION_NAME_COL));
          String objectType = trim(queryResult.m_result.getString(OBJECT_TYPE_COL));

          // filter unsupported permissions
          if (isPermissionSupportedForManager(aclManagerID, permissionName)) {
            String updateSQL = "INSERT INTO " + NEW_SUPPORTED_PERMISSION_TABLE_NAME
               + " (OwnerID, permission_name, object_type) VALUES (?,?,?)";
            PreparedStatement st = destinationConnection.prepareCall(updateSQL);
            st.setInt(1, m_ownerID);
            st.setString(2, permissionName);
            st.setString(3, objectType);
            update(destinationConnection, st);
          }
        }
      }
      finally {
        queryResult.close();
      }
    }
  }

  private static void copyPermissionMembers(String aclManagerID, Connection oldAclManagerConnection,
    Connection destinationConnection,
    int m_ownerID)
    throws Exception {

    String querySQL = "SELECT * FROM " + OLD_PERMISSION_MEMBER_TABLE_NAME;
    SqlResult queryResult = query(oldAclManagerConnection, querySQL);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          String permissionName = trim(queryResult.m_result.getString(PERMISSION_NAME_COL));
          String memberName = trim(queryResult.m_result.getString(PERMISSION_MEMBER_COL));

          // filter unsupported permissions
          if (isPermissionSupportedForManager(aclManagerID, permissionName)
             && isPermissionSupportedForManager(aclManagerID, memberName)) {
            String updateSQL = "INSERT INTO " + NEW_PERMISSION_MEMBER_TABLE_NAME
               + " (OwnerID, permission_name, member_name) VALUES (?,?,?)";
            PreparedStatement st = destinationConnection.prepareCall(updateSQL);
            st.setInt(1, m_ownerID);
            st.setString(2, permissionName);
            st.setString(3, memberName);
            update(destinationConnection, st);
          }
        }
      }
      finally {
        queryResult.close();
      }
    }
  }

  private static boolean isPermissionSupportedForManager(String aclManagerID, String permissionName) {

    if (aclManagerID.equals(FRAMEWORK_ACL_MANAGER_ID)) {
      return isFrameworkPermission(permissionName);
    }
    if (aclManagerID.equals(SERVICE_ACL_MANAGER_ID)) {
      return isServicePermission(permissionName);
    }

    return true;
  }

  private static boolean isFrameworkPermission(String permissionName) {

    final String[] FRAMEWORK_PERMISSIONS = {"fullcontrol", "read", "write", "readwrite", "delete"};

    for (int i = 0; i < FRAMEWORK_PERMISSIONS.length; i++) {
      if (permissionName.equals(FRAMEWORK_PERMISSIONS[i])) {
        return true;
      }
    }

    return false;
  }

  private static boolean isServicePermission(String permissionName) {

    final String[] SERVICE_PERMISSIONS = {"fullcontrol", "sub_unsubscribe", "sub_active", "sub_passive", "sub_folder", "changeappr", "deletecollab"};

    for (int i = 0; i < SERVICE_PERMISSIONS.length; i++) {
      if (permissionName.equals(SERVICE_PERMISSIONS[i])) {
        return true;
      }
    }

    return false;
  }

  private static SqlResult query(Connection connection, String sql)
    throws SQLException {
    return query(connection, connection.prepareCall(sql));
  }

  private static SqlResult query(Connection connection, PreparedStatement st)
    throws SQLException {
    SqlResult result = new SqlResult();

    try {
      result.m_statement = st;
      result.m_result = result.m_statement.executeQuery();
    }
    catch (SQLException e) {
      if (result.m_result != null) {
        result.m_result.close();
      }
      if (result.m_statement != null) {
        result.m_statement.close();
      }
      throw e;
    }

    if (result.m_statement == null || result.m_result == null) {
      if (result.m_result != null) {
        result.m_result.close();
      }
      if (result.m_statement != null) {
        result.m_statement.close();
      }
      return null;
    }

    return result;
  }

  private static int update(Connection connection, String sql)
    throws SQLException {
    return update(connection, connection.prepareCall(sql));
  }

  private static int update(Connection connection, PreparedStatement st)
    throws SQLException {
    int result = 0;
    if (st != null) {
      try {
        result = st.executeUpdate();
      }
      catch (SQLException e) {
            //$JL-EXC$        
        s_log.warningT("update(564)", e.toString());
      }
      finally {
        st.close();
      }
    }
    return result;
  }

  private static String trim(String s) {
    if (s == null) {
      return "";
    }
    return s.trim();
  }
}

/**
 * TBD: Description of the class.
 */
class SqlResult {

  public PreparedStatement m_statement = null;
  public ResultSet m_result = null;

  public void close()
    throws SQLException {
    if (m_result != null) {
      m_result.close();
    }
    if (m_statement != null) {
      m_statement.close();
    }
  }
}

