/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;

import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IMutableConfigurable;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionPoolManager;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.sql.Connection;

/**
 * Merge ACLs from all ACL managers in the tables of one connection pool (merge
 * framework and service ACLs). The implementation copes with framework ACLs and
 * service ACLs which reside in the same pool (which is the case for most oracle
 * installations).
 */
public final class JDBCMergeAcls {
  
  /**
   * hard-wired ACL managers
   */
  private final static String CFG_PLUGIN_CM_UTILITIES_ACL_MANAGERS = "/cm/utilities/acl_managers";
  private final static String FRAMEWORK_ACL_MANAGER_ID = "acl_jdbc_rep_mgr";
  private final static String SERVICE_ACL_MANAGER_ID = "acl_jdbc_service_mgr";
  private final static String[] MERGE_ACL_MANAGERS = {FRAMEWORK_ACL_MANAGER_ID, SERVICE_ACL_MANAGER_ID};

  /**
   * configuration attributes
   */
  private final static String CONFIG_ACL_MANAGER_POOL_ID = "poolid";

  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.acl.jdbc.JDBCMergeAcls.class);

  private String m_appID;

  /**
   * Construct
   *
   * @param appID TBD: Description of the incoming method parameter
   */
  public JDBCMergeAcls(String appID) {
    m_appID = appID;
    try {
      merge(JDBCMergeAclsNoConfig.DEFAULT_DESTINATION_CONNECTION_POOL);
    }
    catch (Exception e) {
                  //$JL-EXC$
      s_log.errorT("JDBCMergeAcls(63)", "failed to merge ACLs" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }
  }

  /**
   * Merge ACLs from all ACL managers into new tables of a destination
   * connection pool.
   *
   * @param destinationConnectionPoolID TBD: Description of the incoming method
   *      parameter
   * @exception Exception Exception raised in failure situation
   */
  public void merge(String destinationConnectionPoolID)
    throws Exception {

    final long CONFIG_LOCK_TIMEOUT = 1000L * 60L * 1000L;// 1000 minutes
    final long CONFIG_WAIT_TIMEOUT = 60L * 1000L;// 1 minute

    s_log.infoT("merge(81)", "Merging ACLs");

    /**
     * get and lock the ACL manager's configuration plugin
     */
    IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
    IConfigManager configManager = com.sapportals.config.fwk.Configuration.getInstance().getConfigManager(context);
    IConfigPlugin plugin = configManager.getAndLockConfigPlugin(CFG_PLUGIN_CM_UTILITIES_ACL_MANAGERS, CONFIG_LOCK_TIMEOUT, CONFIG_WAIT_TIMEOUT);
    try {
      /**
       * get a connection for from the destination connection pool
       */
      Connection destinationConnection = JDBCConnectionPoolManager.getInstance().getPool(destinationConnectionPoolID).getConnection();
      try {
        destinationConnection.setAutoCommit(false);
        destinationConnection.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);

        /**
         * loop over the ACL managers to - define unique ACL manager IDs - copy
         * the ACLs to the new tables - remove the old tables
         */
        for (int i = 0; i < MERGE_ACL_MANAGERS.length; i++) {

          // get the ACL manager configurationm
          IMutableConfigurable aclManager = plugin.getConfigurable(MERGE_ACL_MANAGERS[i]);

          // the ID of the ACL manager is the name of the configurable
          String aclManagerID = aclManager.getIdValue();

          // memorize the old ACL manager's connection pool ID
          String oldAclManagerConnectionPoolID = aclManager.getPropertyValue(CONFIG_ACL_MANAGER_POOL_ID);

          // get a connection from the old ACL managers connection pool
          Connection oldAclManagerConnection = JDBCConnectionPoolManager.getInstance().getPool(oldAclManagerConnectionPoolID).getConnection();
          try {
            oldAclManagerConnection.setAutoCommit(false);
            oldAclManagerConnection.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);

            // if the old ACL tables exist, merge with the new tables
            if (JDBCMergeAclsNoConfig.existOldTables(oldAclManagerConnection)) {

              // copy the ACLs
              int ownerID = JDBCMergeAclsNoConfig.populateManagerApplication(destinationConnection, aclManagerID, m_appID);
              JDBCMergeAclsNoConfig.copyTables(aclManagerID, oldAclManagerConnection, destinationConnection, ownerID);

              // change ACL manager connection pool ID
              if (!oldAclManagerConnectionPoolID.equals(destinationConnectionPoolID)) {
                aclManager.setPropertyValue(CONFIG_ACL_MANAGER_POOL_ID, destinationConnectionPoolID);
                plugin.replaceConfigurable(aclManager);
              }

              // remove the old ACL manager's tables
              JDBCMergeAclsNoConfig.removeOldTables(oldAclManagerConnection);
            }
            else {
              // if the old tables don't exist but the new tables are already there, set ACL manager connection pool ID if it's wrong
              if (JDBCMergeAclsNoConfig.existNewTables(destinationConnection)) {
                if (!oldAclManagerConnectionPoolID.equals(destinationConnectionPoolID)) {
                  aclManager.setPropertyValue(CONFIG_ACL_MANAGER_POOL_ID, destinationConnectionPoolID);
                  plugin.replaceConfigurable(aclManager);
                }
              }
            }

            // commit changes
            oldAclManagerConnection.commit();
            destinationConnection.commit();
          }
          catch (Exception e) {
            oldAclManagerConnection.rollback();
            destinationConnection.rollback();
            throw e;
          }
          finally {
            // return the connection from the old ACL managers connection pool
            oldAclManagerConnection.close();
          }
        }
      }
      finally {
        // return the connection from the the destination connection pool
        destinationConnection.close();
      }
    }
    finally {
      plugin.unlock();
    }
  }

}
