/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;

import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionPool ;
import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionPoolManager ;
import com.sap.tc.logging.Location;
import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.util.acl.*;
import com.sapportals.wcm.util.cache.ICache;
import com.sapportals.wcm.util.jdbc.escape.Escape;

import java.sql.*;
import java.util.*;

/**
 * Uncached access to the ACL database persistence
 */
public final class JDBCDatabaseConnectionUncached
extends AbstractDatabaseConnectionUncached {

  /*
  * table names
  * NOTE: ACL_TBL is made availabe to the package
  * as the TableColumnPopulator needs it
  */
  protected final static String ACL_TBL = "wcm_app_acl_acl";

  private final static String ACL_MANAGER_TBL = "wcm_app_acl_manager_appl";
  private final static String ACL_ENTRY_TABLE_NAME = "wcm_app_acl_acl_entry";
  private final static String OWNER_TABLE_NAME = "wcm_app_acl_owner";
  private final static String PERMISSION_TABLE_NAME = "wcm_app_acl_perm";
  private final static String SUPPORTED_PERMISSION_TABLE_NAME =
    "wcm_app_acl_supp_perm";
  private final static String PERMISSION_MEMBER_TABLE_NAME =
    "wcm_app_acl_perm_member";

  // columns and their attributes
  private final static String ACL_ID_COL = "RID";

  private final static String ACL_RIDSTART_COL = "RIDStart";
  private final static int RID_START_LENGTH = 900;
  private final static int RID_MAX_ESCAPED_LENGTH = 4000;

  protected final static String ACL_HASH_COL = "Hash";

  private final static String CHANGE_LEVEL_COL = "change_level";

  private final static String OBJECT_TYPE_COL = "object_type";
  private final static int OBJECT_TYPE_LENGTH = 127;

  private final static String ACL_ENTRY_ID_COL = "ID";

  private final static String PRINCIPAL_NAME_COL = "principal_name";
  private final static int PRINCIPAL_NAME_LENGTH = 255;

  private final static String PRINCIPAL_TYPE_COL = "principal_type";

  private final static String LOCKING_USER_COL = "locking_user";
  private final static int LOCKING_USER_LENGTH = 255;

  private final static String SORT_INDEX_COL = "sort_index";

  private final static String IS_NEGATIVE_COL = "is_negative";

  private final static String IS_PROPAGATED_COL = "is_propagated";

  private final static String PERMISSION_NAME_COL = "permission_name";
  private final static int PERMISSION_NAME_LENGTH = 127;

  private final static String IS_PREDEFINED_COL = "is_predefined";

  private final static String PERMISSION_MEMBER_COL = "member_name";
  private final static int PERMISSION_MEMBER_LENGTH = 127;
  private int locOwnerId ;

  static {
    s_log = Location.getLocation ( JDBCDatabaseConnectionUncached.class ) ;
  } ;

  public void terminate() {} ;

  /*
  * it's better to cache permissions on this level, because they
  * are heavily used to create ACEs
  */
  private final String m_ApplicationAndManagerSelectionCondition;
  private final boolean generateHashColumns = true;
  private boolean hasOldColumnPopulation = true;

  /**
   * Construct
   *
   * @param applicationID TBD: Description of the incoming method parameter
   * @param managerID TBD: Description of the incoming method parameter
   * @param poolID TBD: Description of the incoming method parameter
   * @param permissionCache TBD: Description of the incoming method parameter
   * @exception AclLoadClassException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public JDBCDatabaseConnectionUncached ( String applicationID,
  String managerID, String poolID, ICache permissionCache,
  JDBCConnectionPool p ) throws AclLoadClassException, AclPersistenceException {

    if (applicationID == null || managerID == null || poolID == null) {
      throw new java.lang.IllegalArgumentException();
    }

    m_applicationID = applicationID;
    m_managerID = managerID;
    m_poolID = poolID;
    conPool = p ;

    if (s_log.beDebug()) {
      s_log.debugT("JDBCDatabaseConnectionUncached(158)", "applicationID "
        + applicationID + ", managerID " + managerID + ", poolID " + poolID
        + ", permissionCache " + permissionCache.getID());
    }
    m_permissionCache = permissionCache;
    try {
      int owner = 0;
      String selCondition = "1=0";
      m_main_connection = openConnection() ;

      PreparedStatement st = m_main_connection.prepareStatement(
        "SELECT ID FROM " + ACL_MANAGER_TBL + " WHERE app=? AND manager=?");
      st.setString(1, trim(Escape.escapeSqlString(m_applicationID, 127)));
      st.setString(2, trim(Escape.escapeSqlString(m_managerID, 127)));
      ResultSet rs = st.executeQuery();
      if (rs.next()) {
        selCondition = "OwnerID=?";
        owner = rs.getInt(1);
      }
      else {
        rs.close();
        rs = null;
        st.close();
        st = null;
        st = m_main_connection.prepareStatement( "INSERT INTO "
          + ACL_MANAGER_TBL + " (app, manager) VALUES (?,?)");
        st.setString(1, trim(Escape.escapeSqlString(m_applicationID, 127)));
        st.setString(2, trim(Escape.escapeSqlString(m_managerID, 127)));
        st.executeUpdate();
        st.close();
        st = null;

        st = m_main_connection.prepareStatement( "SELECT ID FROM "
          + ACL_MANAGER_TBL + " WHERE app=? AND manager=?");
        st.setString(1, trim(Escape.escapeSqlString(m_applicationID, 127)));
        st.setString(2, trim(Escape.escapeSqlString(m_managerID, 127)));
        rs = st.executeQuery();
        if (rs.next()) {
          selCondition = "OwnerID=?";
          owner = rs.getInt(1);
        }
        else {
          throw new AclPersistenceException("Can't insert entry for manager='"
            + managerID + "' and application '" + applicationID
            + " mapped to '" + trim(Escape.escapeSqlString(m_managerID, 127))
            + "'/'" + trim(Escape.escapeSqlString(m_applicationID, 127))
            + "' in poolID '" + poolID + "'");
        }
      }
      rs.close();
      rs = null;
      st.close();
      st = null;

      locOwnerId = owner;
      ownerId = locOwnerId ;
      m_ApplicationAndManagerSelectionCondition = selCondition;

      ensureHash(m_main_connection, "wcm_app_acl_acl");
      ensureSegments(m_main_connection, "wcm_app_acl_acl");
      ensureHash(m_main_connection, "wcm_app_acl_acl_entry");
      ensureHash(m_main_connection, "wcm_app_acl_owner");

      checkInsertInitialPermissions(m_main_connection);
    }
    catch (SQLException e) {
      throw new AclPersistenceException(e);
    }
  }

  private void ensureSegments(Connection con, String table)
    throws SQLException {
    s_log.debugT("ensureSegments(225)", "searching for acls without hash columns...");
    int anz = 0;
    PreparedStatement st = con.prepareStatement("SELECT COUNT(*) FROM " + table + " WHERE Seg1 IS NULL");
    ResultSet rs = st.executeQuery();
    if (rs.next()) {
      anz = rs.getInt(1);
    }
    rs.close();
    rs = null;
    st.close();
    st = null;
    hasOldColumnPopulation = (anz > 0);

    if (!hasOldColumnPopulation) {
      s_log.debugT("ensureSegments(239)", "all acls have correct hash columns!");
      return;
    }

    if (!generateHashColumns) {
      s_log.warningT("ensureSegments(244)", anz + " acls should be converted!");
      return;
    }

    s_log.infoT("ensureSegments(248)", anz + " acl entries are going to be converted in a new thread!");
    try {
      TableColumnPopulator.getInstance().addConnection(openConnection());
    }
    catch (AclPersistenceException e) {
      s_log.errorT("ensureSegments(253)", "Can't start thread to populate acl columns!" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }
  }

  private void ensureHash(Connection con, String table)
    throws SQLException {
    PreparedStatement st = con.prepareStatement("SELECT RID FROM " + table + " WHERE Hash IS NULL");
    ResultSet rs = st.executeQuery();
    HashMap l = new HashMap();
    while (rs.next()) {
      String s = rs.getString(1);
      if (s != null) {
        int hash = s.hashCode();
        l.put(s, new Integer(hash));
      }
    }
    rs.close();
    rs = null;
    st.close();
    st = null;

    if (l.isEmpty()) {
      return;
    }

    st = con.prepareStatement("UPDATE " + table + " SET Hash=? WHERE RID=?");
    Iterator it = l.keySet().iterator();
    while (it.hasNext()) {
      String s = (String)it.next();
      st.setInt(1, ((Integer)l.get(s)).intValue());
      st.setString(2, s);
      st.executeUpdate();
    }
    st.close();
    st = null;
  }

  /**
   * @param owner TBD: Description of the incoming method parameter
   * @param externalAclID TBD: Description of the incoming method parameter
   * @param objectType TBD: Description of the incoming method parameter
   * @return a new ACL (create in the database)
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception AclExistsException Exception raised in failure situation
   */
  public JDBCAcl createAcl ( IUMPrincipal owner, String externalAclID,
  IObjectType objectType) throws AclPersistenceException, AclExistsException {

    if (owner == null || externalAclID == null || objectType == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("createAcl(991)", "owner " + owner.getId()
        + ", externalAclID " + externalAclID + ", objectType " + objectType);
    }

    synchronized (m_main_connection_lock) {
      try {

        if (uncommittedCheckAcl(m_main_connection, externalAclID)) {
          throw new AclExistsException();
        }

        /*
        * in case the database is inconsistent, entries may exist
        * for a removed acl - remove them
        */
        uncommittedRemoveAclEntries(m_main_connection, externalAclID);

        /*
        * in case the database is inconsistent, owners may exist
        * for a removed acl - remove them
        */
        uncommittedRemoveOwners(m_main_connection, externalAclID);

        //check length
        Escape.escapeSqlString(externalAclID, RID_MAX_ESCAPED_LENGTH);

        if (!uncommittedAddOwner(m_main_connection, externalAclID, owner)) {
          throw new Exception("can't create owner");
        }
        if (!uncommittedAddAcl(m_main_connection, externalAclID, objectType)) {
          throw new Exception("can't create acl");
        }

        m_main_connection.commit();
      }
      catch (AclExistsException e) {
        try {
          m_main_connection.rollback();
        }
        catch (SQLException sqlException) {
          s_log.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
        }
        throw e;
      }
      catch (Exception e) {
        try {
          m_main_connection.rollback();
        }
        catch (SQLException sqlException) {
          s_log.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
        }
        try {
          reopenMainConnection();
        }
        catch (SQLException sqlException) {
          s_log.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
        }
        throw new AclPersistenceException(e);
      }

      return getAcl(externalAclID, false); // COOKED
    }
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param externalAclIDRoot TBD: Description of the incoming method parameter
   * @return true iff ACLs exist with IDs starting with externalAclIDRoot (has
   *      children)
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean hasDescendantsWithAcl(Connection connection,
  String externalAclIDRoot) throws SQLException {
    Integer[] segments = makeHashSegments(externalAclIDRoot);
    String sql = " WHERE ((";
    if (hasOldColumnPopulation) {
      sql += "Seg1 IS NULL) OR (";
    }
    for (int i = 0; i < 10; i++) {
      if (segments[i] == null) {
        break;
      }
      if (i > 0) {
        sql += " AND ";
      }
      sql += "(Seg" + (i + 1) + "=?)";
    }
    sql += ")) AND (" + m_ApplicationAndManagerSelectionCondition + ") AND ("
      + ACL_RIDSTART_COL;
    boolean big = false;

    if (!externalAclIDRoot.endsWith("/")) {
      externalAclIDRoot += "/";
    }
    String s = Escape.escapeSqlString(externalAclIDRoot, 0).trim() + '%';
    String t = s;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
      sql += "=?) AND (" + ACL_ID_COL + " LIKE ?)";
      big = true;
    }
    else {
      sql += " LIKE ?)";
    }

    PreparedStatement st = null;
    ResultSet rs = null;
    try {
      st = connection.prepareStatement("SELECT " + ACL_ID_COL + " FROM "
        + ACL_TBL + sql);
      int i = 1;
      for (int j = 0; j < 10; j++) {
        Integer seg = segments[j];
        if (seg == null) {
          break;
        }
        st.setInt(i++, seg.intValue());
      }
      st.setInt(i++, locOwnerId);
      st.setString(i++, t);
      if (big) {
        st.setString(i++, s);
      }
      rs = st.executeQuery();
      boolean rv = rs.next() ;
      rs.close();
      rs = null;
      st.close();
      st = null;
      return rv;
    }
    finally {
      if (rs != null) {
        rs.close();
      }
      if (st != null) {
        st.close();
      }
    }
  }

  protected List getDescendantsWithAcl(Connection connection,
  String externalAclIDRoot) throws SQLException {
    List ret = new LinkedList();

    Integer[] segments = makeHashSegments(externalAclIDRoot);
    String sql = " WHERE ((";
    if (hasOldColumnPopulation) {
      sql += "Seg1 IS NULL) OR (";
    }
    for (int i = 0; i < 10; i++) {
      if (segments[i] == null) {
        break;
      }
      if (i > 0) {
        sql += " AND ";
      }
      sql += "(Seg" + (i + 1) + "=?)";
    }
    sql += ")) AND (" + m_ApplicationAndManagerSelectionCondition + ") AND ("
      + ACL_RIDSTART_COL;
    boolean big = false;

    if (!externalAclIDRoot.endsWith("/")) {
      externalAclIDRoot += "/";
    }
    String s = Escape.escapeSqlString(externalAclIDRoot, 0).trim() + '%';
    String t = s;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
      sql += "=?) AND (" + ACL_ID_COL + " LIKE ?)";
      big = true;
    }
    else {
      sql += " LIKE ?)";
    }

    PreparedStatement st = null;
    ResultSet rs = null;
    try {
      st = connection.prepareStatement("SELECT " + ACL_ID_COL + " FROM "
        + ACL_TBL + sql);
      int i = 1;
      for (int j = 0; j < 10; j++) {
        Integer seg = segments[j];
        if (seg == null) {
          break;
        }
        st.setInt(i++, seg.intValue());
      }
      st.setInt(i++, locOwnerId);
      st.setString(i++, t);
      if (big) {
        st.setString(i++, s);
      }
      rs = st.executeQuery();
      while (rs.next()) {
        ret.add(rs.getString(1));
      }
      rs.close();
      rs = null;
      st.close();
      st = null;
      return ret;
    }
    finally {
      if (rs != null) {
        rs.close();
      }
      if (st != null) {
        st.close();
      }
    }
  }


  protected void uncommittedRemoveDescendantsWithAcl(Connection connection, String externalAclIDRoot)
    throws SQLException {

    //the new call is important, do not change!!!!
    //due to some strange implementations the
    //select count(*) from table where column like 'literal%'
    //is much more performant with an empty result set than the
    //direct select of the desired column!!!!
    List childs = getDescendantsWithAcl(connection, externalAclIDRoot);
    if (childs == null || childs.size() == 0) {
      return;
    }

    if (!externalAclIDRoot.endsWith("/")) {
      externalAclIDRoot += "/";
    }
    String sql = " WHERE (" + m_ApplicationAndManagerSelectionCondition + ") AND (" + ACL_RIDSTART_COL;
    String s = Escape.escapeSqlString(externalAclIDRoot, 0).trim() + '%';
    boolean big = false;
    String t = s;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
      sql += "=?) AND (" + ACL_ID_COL + " LIKE ?)";
      big = true;
    }
    else {
      sql += " LIKE ?)";
    }

    PreparedStatement st = null;
    ResultSet rs = null;
    try {
      st = connection.prepareStatement("DELETE " + ACL_TBL + sql);
      st.setInt(1, locOwnerId);
      st.setString(2, t);
      if (big) {
        st.setString(3, s);
      }
      st.executeUpdate();
      st.close();
      st = null;

      st = connection.prepareStatement("SELECT COUNT(*) FROM " + ACL_ENTRY_TABLE_NAME + sql);
      st.setInt(1, locOwnerId);
      st.setString(2, t);
      if (big) {
        st.setString(3, s);
      }
      rs = st.executeQuery();
      int i = 0;
      if (rs.next()) {
        i = rs.getInt(1);
      }
      rs.close();
      rs = null;
      st.close();
      st = null;
      if (i > 0) {
        st = connection.prepareStatement("DELETE " + ACL_ENTRY_TABLE_NAME + sql);
        st.setInt(1, locOwnerId);
        st.setString(2, t);
        if (big) {
          st.setString(3, s);
        }
        i = st.executeUpdate();
        st.close();
        st = null;
      }

      st = connection.prepareStatement("SELECT COUNT(*) FROM " + OWNER_TABLE_NAME + sql);
      st.setInt(1, locOwnerId);
      st.setString(2, t);
      if (big) {
        st.setString(3, s);
      }
      rs = st.executeQuery();
      i = 0;
      if (rs.next()) {
        i = rs.getInt(1);
      }
      rs.close();
      rs = null;
      st.close();
      st = null;
      if (i > 0) {
        st = connection.prepareStatement("DELETE " + OWNER_TABLE_NAME + sql);
        st.setInt(1, locOwnerId);
        st.setString(2, t);
        if (big) {
          st.setString(3, s);
        }
        i = st.executeUpdate();
        st.close();
        st = null;
      }

      st = connection.prepareStatement("UPDATE wcm_acl_version SET counter=counter+1");
      st.executeUpdate();
    }
    finally {
      if (rs != null) {
        rs.close();
      }
      if (st != null) {
        st.close();
      }
    }
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param externalAclIDRoot TBD: Description of the incoming method parameter
   * @return all ACL IDs (strings) in a list which start with the specified
   *      prefix
   * @exception SQLException Exception raised in failure situation
   */
  protected List uncommittedGetDescendantsWithAcl(Connection connection, String externalAclIDRoot)
    throws SQLException {

    LinkedList result = new LinkedList();

    List childs = getDescendantsWithAcl(connection, externalAclIDRoot);
    if (childs != null) {
      Iterator it = childs.listIterator();
      while (it.hasNext()) {
        String externalAclID = trim(Escape.unescapeSqlString((String)it.next()));
        result.add(externalAclID);
      }
    }
    return result;
  }

  /**
   * Container for result sets and statements
   */
  protected class SqlResult {
    public Statement m_statement = null ;
    public ResultSet m_result = null ;

    public void close()
      throws SQLException {
      if (m_result != null) {
        m_result.close() ;
      }
      if (m_statement != null) {
        m_statement.close() ;
      }
    }
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return an ACL object for an ACL in the database (may be null)
   * @exception SQLException Exception raised in failure situation
   */
  protected AclRec uncommittedGetAcl(Connection connection, String externalAclID)
    throws SQLException {

    AclRec result = null;

    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + ACL_TBL + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");

    SqlResult queryResult = query(st, 1, Escape.escapeSqlString(externalAclID, 0), 2);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          String objectType = trim(Escape.unescapeSqlString(queryResult.m_result.getString(OBJECT_TYPE_COL)));
          long changeLevel = queryResult.m_result.getLong(CHANGE_LEVEL_COL);
          result = new AclRec(externalAclID, objectType, changeLevel);
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param entryID TBD: Description of the incoming method parameter
   * @return the ACL ID which an ACE is attached to
   * @exception SQLException Exception raised in failure situation
   */
  protected String uncommittedGetInternalAclIDFromEntryID(Connection connection, long entryID)
    throws SQLException {

    String result = null;

    PreparedStatement st = connection.prepareStatement("SELECT " + ACL_ID_COL
       + " FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " + ACL_ENTRY_ID_COL + "=?");

    st.setInt(1, (int)entryID);

    SqlResult queryResult = query(st, 0, null, -1);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = trim(Escape.unescapeSqlString(queryResult.m_result.getString(ACL_ID_COL)));
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param aclIDs TBD: Description of the incoming method parameter
   * @return ACL objects for ACLs in the database (some may be null)
   * @exception SQLException Exception raised in failure situation
   */
  protected AclRec[] uncommittedGetAclsFromDB ( Connection connection,
  String[] aclIDs ) throws SQLException {

    if (aclIDs.length == 0) {
      return new AclRec[0];
    }

    String sql = "SELECT * FROM " + ACL_TBL + " WHERE "
      + m_ApplicationAndManagerSelectionCondition + " AND ";

    sql += "(";
    HashMap idIndices = new HashMap(aclIDs.length);
    for (int i = 0; i < aclIDs.length; i++) {
      idIndices.put(aclIDs[i], new Integer(i));
      if (i > 0) {
        sql += " OR (";
      }
      String s = Escape.escapeSqlString(aclIDs[i], 0);
      sql += ACL_HASH_COL + "=" + s.hashCode() + " AND " + ACL_ID_COL + "='" + s + "'";
      if (i > 0) {
        sql += ")";
      }
    }
    sql += ")";

    AclRec[] result = new AclRec[aclIDs.length];
    for (int i = 0; i < result.length; i++) {
      result[i] = null;
    }

    SqlResult queryResult = query(connection, sql);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {

          String aclID = trim(Escape.unescapeSqlString(queryResult.m_result.getString(ACL_ID_COL)));
          String objectType = trim(Escape.unescapeSqlString(queryResult.m_result.getString(OBJECT_TYPE_COL)));
          long changeLevel = queryResult.m_result.getLong(CHANGE_LEVEL_COL);

          Integer indexObj = (Integer)idIndices.get(aclID);
          if (indexObj != null) {
            int index = indexObj.intValue();
            result[index] = new AclRec(aclIDs[index], objectType,
              changeLevel);
          }
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param changeLevel TBD: Description of the incoming method parameter
   * @return true iff the change level of the ACL in the database is not larger
   *      than the specified one
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsAclUpToDate(Connection connection, String internalAclID, long changeLevel)
    throws SQLException {

    boolean result = false;

    PreparedStatement st = connection.prepareStatement(
      "SELECT " + CHANGE_LEVEL_COL + " FROM " + ACL_TBL
       + " WHERE " + m_ApplicationAndManagerSelectionCondition + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");

    SqlResult queryResult = query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          long dbLevel = queryResult.m_result.getLong(CHANGE_LEVEL_COL);
          result = (dbLevel == changeLevel);
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclIDs TBD: Description of the incoming method parameter
   * @param changeLevels TBD: Description of the incoming method parameter
   * @return up to date flags for ACLs (check change level in the database)
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean[] uncommittedAreAclsUpToDateInDB(Connection connection, String[] internalAclIDs, long[] changeLevels)
    throws SQLException {

    if (internalAclIDs.length == 0) {
      return new boolean[0];
    }

    String sql = "SELECT * FROM " + ACL_TBL + " WHERE " + m_ApplicationAndManagerSelectionCondition + " AND ";

    sql += "(";
    HashMap idIndices = new HashMap(internalAclIDs.length);
    for (int i = 0; i < internalAclIDs.length; i++) {
      idIndices.put(internalAclIDs[i], new Integer(i));
      if (i > 0) {
        sql += " OR (";
      }
      String s = Escape.escapeSqlString(internalAclIDs[i], 0);
      sql += ACL_HASH_COL + "=" + s.hashCode() + " AND " + ACL_ID_COL + "='" + s + "'";
      if (i > 0) {
        sql += ")";
      }
    }
    sql += ")";

    boolean[] result = new boolean[internalAclIDs.length];
    for (int i = 0; i < result.length; i++) {
      result[i] = false;
    }

    SqlResult queryResult = query(connection, sql);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {

          String aclID = trim(Escape.unescapeSqlString(queryResult.m_result.getString(ACL_ID_COL)));
          long dbLevel = queryResult.m_result.getLong(CHANGE_LEVEL_COL);

          Integer indexObj = (Integer)idIndices.get(aclID);
          if (indexObj != null) {
            int index = indexObj.intValue();
            result[index] = dbLevel == changeLevels[index];
          }
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * Increase the change level of the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  protected void uncommittedIncreaseAclChangeLevel(Connection connection, String internalAclID)
    throws SQLException {
    PreparedStatement st = connection.prepareStatement(
      "UPDATE " + ACL_TBL + " SET " + CHANGE_LEVEL_COL + "=" + CHANGE_LEVEL_COL + "+1 WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");
    String s = Escape.escapeSqlString(internalAclID, 0);
    st.setInt(2, s.hashCode());
    st.setString(3, s);
    update(connection, st);
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return true iff the ACL exists in the database
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckAcl(Connection connection, String internalAclID)
    throws SQLException {
    boolean result = false;
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + ACL_TBL + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");

    SqlResult queryResult = query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = true;
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * Set the ACE sort index
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param aclEntryID TBD: Description of the incoming method parameter
   * @param sortIndex TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedSetAclEntrySortIndex(Connection connection, long aclEntryID, int sortIndex)
    throws SQLException {
    PreparedStatement st = connection.prepareStatement(
      "UPDATE " + ACL_ENTRY_TABLE_NAME
       + " SET " + SORT_INDEX_COL + " = ? WHERE " + ACL_ENTRY_ID_COL + " = ?");
    st.setInt(1, sortIndex);
    /*
    * in non-OpenSQL land we know that the entry id is only 32 bit
    */
    st.setInt(2, (int)aclEntryID);
    return update(connection, st, 0, null, -1) > 0;
  }

  /**
   * Define the propagate-flag of the ACE
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param aclEntryID TBD: Description of the incoming method parameter
   * @param propagate TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedSetAclEntryPropagation(Connection connection,
  long aclEntryID, boolean propagate ) throws SQLException {
    PreparedStatement st = connection.prepareStatement(
      "UPDATE " + ACL_ENTRY_TABLE_NAME
       + " SET " + IS_PROPAGATED_COL + " = ? WHERE "
       + ACL_ENTRY_ID_COL + " = ?");
    st.setInt(1, propagate ? 1 : 0);
    st.setInt(2, (int)aclEntryID);

    return update(connection, st, 0, null, -1) > 0;
  }


  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return true iff the ACL has ACEs
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckAclEntries(Connection connection, String internalAclID)
    throws SQLException {
    boolean result = false;
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");

    SqlResult queryResult = query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = true;
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return true iff the permission exists
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckPermission(Connection connection, String permissionName)
    throws SQLException {
    boolean result = false;
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + PERMISSION_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + PERMISSION_NAME_COL + " = ?");
    st.setString(2, Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH));

    SqlResult queryResult = query(st);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = true;
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return the ACEs of the ACL (principal may be null; otherwise only the
   *      entries belonging to the pricnipal are returned)
   * @exception SQLException Exception raised in failure situation
   */
  protected AclEntryList uncommittedGetAclEntries(Connection connection, String internalAclID, boolean raw)
    throws SQLException {
    AclEntryList result = new AclEntryList();

    SqlResult queryResult = uncommittedGetAclEntriesFromDB(connection, internalAclID);// result must be checked for internalAclID case
    if (queryResult != null) {

      class EntryData {
        public int m_aclEntryID;
        public String m_internalAclID;
        public String m_principalName;
        public int m_principalType;
        public int m_sortIndex;
        public boolean m_negative;
        public boolean m_propagated;
        public String m_permissionName;
      }

      List entryDataList = new ArrayList(50);

      try {
        while (queryResult.m_result.next()) {
          EntryData entryData = new EntryData();

          entryData.m_aclEntryID = queryResult.m_result.getInt(ACL_ENTRY_ID_COL);
          entryData.m_internalAclID = trim(Escape.unescapeSqlString(queryResult.m_result.getString(ACL_ID_COL)));
          entryData.m_principalName = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PRINCIPAL_NAME_COL)));
          entryData.m_principalType = queryResult.m_result.getInt(PRINCIPAL_TYPE_COL);
          entryData.m_sortIndex = queryResult.m_result.getInt(SORT_INDEX_COL);
          entryData.m_negative = queryResult.m_result.getInt(IS_NEGATIVE_COL) != 0;
          entryData.m_propagated = queryResult.m_result.getInt(IS_PROPAGATED_COL) != 0;
          entryData.m_permissionName = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PERMISSION_NAME_COL)));

          entryDataList.add(entryData);
        }
      }
      finally {
        queryResult.close();
      }

      Iterator iterator = entryDataList.iterator();
      while (iterator.hasNext()) {
        EntryData entryData = (EntryData)iterator.next();
        try {
          IUMPrincipal newPrincipal = getUMPrincipal(entryData.m_principalName, getPrincipalTypeFromTag(entryData.m_principalType), raw);
          if (newPrincipal != null) {
            // ignore users which are not configured in the user management
            IAclPermission permission = uncommittedGetPermission(connection, entryData.m_permissionName);
            JDBCAclEntry entry = ( raw && ( newPrincipal instanceof IUMPrincipalRaw ) ) 
                                 ? new JDBCAclEntryRaw(this, newPrincipal, entryData.m_negative, permission, entryData.m_sortIndex, entryData.m_propagated)
                                 : new JDBCAclEntry(this, newPrincipal, entryData.m_negative, permission, entryData.m_sortIndex, entryData.m_propagated);
            entry.setIDs(entryData.m_aclEntryID, entryData.m_internalAclID);
            result.add(entry);
          }
        }
        catch (InvalidClassException e) {
          ; // will not be thrown; just to keep the exception from the method prototype
        }
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param acls TBD: Description of the incoming method parameter
   * @return the ACEs of the ACLs (don't get them all at once to limit the size
   *      of the SQL statement) entries of the acls table may be null
   * @exception SQLException Exception raised in failure situation
   */
  protected AclEntryList[] uncommittedGetAclEntries(Connection connection, AclRec[] acls, boolean raw)
    throws SQLException {
    final int MAX_IDS_PER_QUERY = 100;
      class EntryData {

      public int m_aclEntryID;
      public String m_internalAclID;
      public String m_principalName;
      public int m_principalType;
      public int m_sortIndex;
      public boolean m_negative;
      public boolean m_propagated;
      public String m_permissionName;
    }

    if (acls.length == 0) {
      return new AclEntryList[0];
    }

    HashMap entriesTable = new HashMap(acls.length);
    for (int i = 0; i < acls.length; i++) {
      if (acls[i] != null) {
        entriesTable.put(acls[i].getExternalAclID(), new LinkedList());
      }
    }

    String[] consolidatedIDs = new String[entriesTable.size()];
    Iterator entriesTableIterator = entriesTable.keySet().iterator();
    int consolidatedIdx = 0;
    while (entriesTableIterator.hasNext()) {
      consolidatedIDs[consolidatedIdx++] = (String)entriesTableIterator.next();
    }

    for (int i = 0; i < consolidatedIDs.length; i += MAX_IDS_PER_QUERY) {
      int takeCount = Math.min(consolidatedIDs.length - i, MAX_IDS_PER_QUERY);
      String[] internalTakeIDs = new String[takeCount];
      for (int k = 0; k < takeCount; k++) {
        internalTakeIDs[k] = consolidatedIDs[i + k];
      }

      SqlResult queryResult = uncommittedGetAclEntriesFromDB(connection, internalTakeIDs);// result must be checked for internalAclID case
      if (queryResult != null) {
        try {
          while (queryResult.m_result.next()) {
            EntryData entryData = new EntryData();
            entryData.m_aclEntryID = queryResult.m_result.getInt(ACL_ENTRY_ID_COL);
            entryData.m_internalAclID = trim(Escape.unescapeSqlString(queryResult.m_result.getString(ACL_ID_COL)));
            entryData.m_principalName = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PRINCIPAL_NAME_COL)));
            entryData.m_principalType = queryResult.m_result.getInt(PRINCIPAL_TYPE_COL);
            entryData.m_sortIndex = queryResult.m_result.getInt(SORT_INDEX_COL);
            entryData.m_negative = queryResult.m_result.getInt(IS_NEGATIVE_COL) != 0;
            entryData.m_propagated = queryResult.m_result.getInt(IS_PROPAGATED_COL) != 0;
            entryData.m_permissionName = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PERMISSION_NAME_COL)));
            LinkedList entries = (LinkedList)entriesTable.get(entryData.m_internalAclID);
            if (entries != null) {
              entries.add(entryData);
            }
          }
        }
        finally {
          queryResult.close();
        }
      }
    }

    AclEntryList[] result = new AclEntryList[acls.length];
    for (int i = 0; i < acls.length; i++) {
      if (acls[i] != null) {
        result[i] = new AclEntryList();
        LinkedList entries = (LinkedList)entriesTable.get(acls[i].getExternalAclID());
        Iterator entriesIterator = entries.iterator();
        while (entriesIterator.hasNext()) {
          EntryData entryData = (EntryData)entriesIterator.next();
          try {
            IUMPrincipal principal = getUMPrincipal(entryData.m_principalName, getPrincipalTypeFromTag(entryData.m_principalType), raw);
            if (principal != null) {
              // ignore users which are not configured in the user management
              IAclPermission permission = uncommittedGetPermission(connection, entryData.m_permissionName);
              JDBCAclEntry aclEntry = ( raw && ( principal instanceof IUMPrincipalRaw ) )
                                      ? new JDBCAclEntryRaw(this, principal, entryData.m_negative, permission, entryData.m_sortIndex, entryData.m_propagated)
                                      : new JDBCAclEntry(this, principal, entryData.m_negative, permission, entryData.m_sortIndex, entryData.m_propagated);
              aclEntry.setIDs(entryData.m_aclEntryID, entryData.m_internalAclID);
              result[i].add(aclEntry);
            }
          }
          catch (InvalidClassException e) {
            ; // will not be thrown; just to keep the exception from the method prototype
          }
        }
      }
      else {
        result[i] = null;
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return a result set for the ACEs of the ACL (principal may be null;
   *      otherwise only the entries belonging to the pricnipal are returned)
   *      (result must be checked for internalAclID case)
   * @exception SQLException Exception raised in failure situation
   */
  private SqlResult uncommittedGetAclEntriesFromDB(Connection connection, String internalAclID)
    throws SQLException {
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=? ORDER BY " + SORT_INDEX_COL);

    return query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclIDs TBD: Description of the incoming method parameter
   * @return a result set for the ACEs of the ACLs
   * @exception SQLException Exception raised in failure situation
   */
  private SqlResult uncommittedGetAclEntriesFromDB(Connection connection, String[] internalAclIDs)
    throws SQLException {

    if (internalAclIDs.length == 0) {
      return null;
    }

    String sql = "SELECT * FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition + " AND ";

    sql += "(";
    for (int i = 0; i < internalAclIDs.length; i++) {
      if (i > 0) {
        sql += " OR (";
      }
      String s = Escape.escapeSqlString(internalAclIDs[i], 0);
      sql += ACL_HASH_COL + "=" + s.hashCode() + " AND " + ACL_ID_COL + " = '" + s + "'";
      if (i > 0) {
        sql += ")";
      }
    }
    sql += ") ";
    sql += " ORDER BY " + SORT_INDEX_COL;

    return query(connection, sql);
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return a list with the IDs of the ACLs (strings) which use the specified
   *      permissions in one of their ACEs
   * @exception SQLException Exception raised in failure situation
   */
  protected List uncommittedGetAffectedAclIDs(Connection connection, String permissionName)
    throws SQLException {

    LinkedList result = new LinkedList();

    String sql = "SELECT DISTINCT " + ACL_ID_COL + " FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PERMISSION_NAME_COL + " = '" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "'";

    SqlResult queryResult = query(connection, sql);
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          result.add(trim(Escape.unescapeSqlString(queryResult.m_result.getString(ACL_ID_COL))));
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * Add a new ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param objectType TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddAcl(Connection connection, String internalAclID, IObjectType objectType)
    throws SQLException {
    String s = Escape.escapeSqlString(internalAclID, 0);
    String t = s;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
    }

    Integer[] segments = makeHashSegments(internalAclID);

    String sql = "INSERT INTO " + ACL_TBL + " (OwnerID, Hash, RID, RIDStart, object_type, locking_user, change_level";

    for (int i = 0; i < 10; i++) {
      if (segments[i] != null) {
        sql += ", Seg" + (i + 1);
      }
    }

    sql += ") VALUES (" + "?," + s.hashCode() + ",'" + s + "','" + t + "',"
       + "'" + Escape.escapeSqlString(objectType.getName(), OBJECT_TYPE_LENGTH) + "',"
       + "'', 0";//no locking user and change level 0

    for (int i = 0; i < 10; i++) {
      if (segments[i] != null) {
        sql += ", " + segments[i].intValue();
      }
    }
    sql += ")";

    return update(connection, sql) > 0;
  }

  /**
   * Add a new ACE
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param aclEntry TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected long uncommittedAddAclEntry(Connection connection, String internalAclID, JDBCAclEntry aclEntry)
    throws SQLException {
    int ret = -1;
    IUMPrincipal principal = populate(aclEntry.getPrincipal());
    if (principal == null) {
      throw new java.lang.IllegalArgumentException();
    }

    JDBCPermission permission = (JDBCPermission)aclEntry.getPermission();
    if (permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    String s = Escape.escapeSqlString(internalAclID, 0);
    String t = s;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
    }
    CallableStatement cs = null;
    try {
      cs = connection.prepareCall("{CALL wcm_acl_insert_entry(?,?,?,?,?,?,?,?,?,?,?)}");
      cs.registerOutParameter(1, Types.INTEGER);
      cs.setInt(2, locOwnerId);
      cs.setInt(3, s.hashCode());
      cs.setString(4, s);
      cs.setString(5, t);
      cs.setString(6, Escape.escapeSqlString(principal.getId(), PRINCIPAL_NAME_LENGTH));
      cs.setInt(7, getPrincipalTypeTag(principal));
      cs.setInt(8, aclEntry.getSortIndex());
      cs.setBoolean(9, aclEntry.isNegative());
      cs.setBoolean(10, aclEntry.isPropagated());
      cs.setString(11, Escape.escapeSqlString(permission.getName(), PERMISSION_NAME_LENGTH));
      cs.executeUpdate();
      ret = cs.getInt(1);
      cs.close();
      cs = null;
    }
    catch (AclPersistenceException e) {
      ; // will not be thrown because the aclEntry is not yet assigned to an acl
    }
    finally {
      if (cs != null) {
        cs.close();
      }
    }

    return ret;
  }

  /**
   * Remove an ACE
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param aclEntryID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemoveAclEntry(Connection connection, long aclEntryID)
    throws SQLException {

    String sql = "DELETE FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " +
      ACL_ENTRY_ID_COL + " = " + aclEntryID;

    return update(connection, sql, 0) > 0;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @return true iff the specified principal is an owner of the ACL
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsOwner(Connection connection, String internalAclID, IUMPrincipal owner)
    throws SQLException {
    boolean result = false;
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + OWNER_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?"
       + " AND " + PRINCIPAL_NAME_COL + " = ?" + " AND " + PRINCIPAL_TYPE_COL + " = ?");
    st.setString(4, Escape.escapeSqlString(owner.getId(), PRINCIPAL_NAME_LENGTH));
    st.setInt(5, getPrincipalTypeTag(owner));

    SqlResult queryResult = query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = true;
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param ownerID TBD: Description of the incoming method parameter
   * @param ownerType TBD: Description of the incoming method parameter
   * @return true iff the specified principal is the last owner of the ACL
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsLastOwner(Connection connection, String internalAclID, String ownerID, int ownerType)
    throws SQLException {
    boolean result = false;
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + OWNER_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");

    SqlResult queryResult = query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          String name = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PRINCIPAL_NAME_COL)));
          int type = queryResult.m_result.getInt(PRINCIPAL_TYPE_COL);
          if (ownerID.equals(name) && ownerType == getPrincipalTypeFromTag(type)) {
            if (!queryResult.m_result.next()) {
              result = true;
            }
          }
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return the owners of the ACL
   * @exception SQLException Exception raised in failure situation
   */
  protected UMPrincipalList uncommittedGetOwners(Connection connection, String internalAclID, boolean raw)
    throws SQLException {

    UMPrincipalList result = new UMPrincipalList();

    SqlResult queryResult = uncommittedGetOwnersFromDB(connection, internalAclID);// result must be checked for character case
    if (queryResult != null) {
      try {
        while (queryResult.m_result.next()) {
          String principalName = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PRINCIPAL_NAME_COL)));
          int principalType = queryResult.m_result.getInt(PRINCIPAL_TYPE_COL);
          IUMPrincipal owner = getUMPrincipal(principalName, getPrincipalTypeFromTag(principalType), raw);
          if (owner != null) {
            result.add(owner);
          }
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param acls TBD: Description of the incoming method parameter
   * @return the owners of the ACLs (don't get them all at once to limit the
   *      size of the SQL statement; entries of the ACL array may be null)
   * @exception SQLException Exception raised in failure situation
   */
  protected UMPrincipalList[] uncommittedGetOwners(Connection connection, AclRec[] acls, boolean raw)
    throws SQLException {

    final int MAX_IDS_PER_QUERY = 100;

    if (acls.length == 0) {
      return new UMPrincipalList[0];
    }

    HashMap ownerTable = new HashMap(acls.length);
    for (int i = 0; i < acls.length; i++) {
      if (acls[i] != null) {
        ownerTable.put(acls[i].getExternalAclID(), new UMPrincipalList());
      }
    }

    String[] consolidatedIDs = new String[ownerTable.size()];
    Iterator ownerTableIterator = ownerTable.keySet().iterator();
    int consolidatedIdx = 0;
    while (ownerTableIterator.hasNext()) {
      consolidatedIDs[consolidatedIdx++] = (String)ownerTableIterator.next();
    }

    for (int i = 0; i < consolidatedIDs.length; i += MAX_IDS_PER_QUERY) {
      int takeCount = Math.min(consolidatedIDs.length - i, MAX_IDS_PER_QUERY);
      String[] internalTakeIDs = new String[takeCount];
      for (int k = 0; k < takeCount; k++) {
        internalTakeIDs[k] = consolidatedIDs[i + k];
      }

      SqlResult queryResult = uncommittedGetOwnersFromDB(connection, internalTakeIDs);
      if (queryResult != null) {
        try {
          while (queryResult.m_result.next()) {
            String id = trim(Escape.unescapeSqlString(queryResult.m_result.getString(ACL_ID_COL)));
            String principalName = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PRINCIPAL_NAME_COL)));
            int principalType = queryResult.m_result.getInt(PRINCIPAL_TYPE_COL);
            IUMPrincipal owner = getUMPrincipal(principalName, getPrincipalTypeFromTag(principalType), raw);
            if (owner != null) {
              UMPrincipalList owners = (UMPrincipalList)ownerTable.get(id);
              if (owners != null) {
                owners.add(owner);
              }
            }
          }
        }
        finally {
          queryResult.close();
        }
      }
    }

    UMPrincipalList[] result = new UMPrincipalList[acls.length];
    for (int i = 0; i < acls.length; i++) {
      if (acls[i] != null) {
        result[i] = (UMPrincipalList)ownerTable.get(acls[i].getExternalAclID());
      }
      else {
        result[i] = null;
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return the owners of the ACL (result must be checked for character case)
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedHasOwnersInDB ( Connection connection,
  String internalAclID ) throws SQLException {
    SqlResult qr = uncommittedGetOwnersFromDB ( connection, internalAclID ) ;
    boolean rv = false ;
    if ( qr != null ) {
      try {
        rv = qr.m_result.next() ;
      }
      finally {
        qr.close() ;
      }
    }
    return rv ;
  }

  private SqlResult uncommittedGetOwnersFromDB(Connection connection, String internalAclID)
    throws SQLException {
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + OWNER_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");

    return query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclIDs TBD: Description of the incoming method parameter
   * @return the owners of the ACLs (result must be checked for character case)
   * @exception SQLException Exception raised in failure situation
   */
  private SqlResult uncommittedGetOwnersFromDB(Connection connection, String[] internalAclIDs)
    throws SQLException {

    if (internalAclIDs.length == 0) {
      return null;
    }

    String sql = "SELECT * FROM " + OWNER_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition + " AND ";

    sql += "(";
    for (int i = 0; i < internalAclIDs.length; i++) {
      if (i > 0) {
        sql += " OR (";
      }
      String s = Escape.escapeSqlString(internalAclIDs[i], 0);
      sql += ACL_HASH_COL + "=" + s.hashCode() + " AND " + ACL_ID_COL + " = '" + s + "'";
      if (i > 0) {
        sql += ")";
      }
    }
    sql += ")";

    return query(connection, sql);
  }

  /**
   * Add an owner to the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddOwner(Connection connection, String internalAclID, IUMPrincipal owner)
    throws SQLException {

    owner = populate(owner);
    String s = Escape.escapeSqlString(internalAclID, 0);
    String t = s;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
    }
    String sql = "INSERT INTO " + OWNER_TABLE_NAME + " (OwnerID, Hash, RID, RIDStart, principal_name, principal_type) VALUES (" +
      "?," + s.hashCode() + ",'" + s + "','" + t + "'," +
      "'" + Escape.escapeSqlString(owner.getId(), PRINCIPAL_NAME_LENGTH) + "'," +
      getPrincipalTypeTag(owner) + ")";

    return update(connection, sql) > 0;
  }

  /**
   * Remove an owner from the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param ownerID TBD: Description of the incoming method parameter
   * @param ownerType TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemoveOwner(Connection connection, String internalAclID, String ownerID, int ownerType)
    throws SQLException {
    String s = Escape.escapeSqlString(internalAclID, 0);
    String sql = "DELETE FROM " + OWNER_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      ACL_HASH_COL + "=" + s.hashCode() + " AND " + ACL_ID_COL + " = '" + s + "' AND " +
      PRINCIPAL_NAME_COL + " = '" + Escape.escapeSqlString(ownerID, PRINCIPAL_NAME_LENGTH) + "' AND " +
      PRINCIPAL_TYPE_COL + " = " + getPrincipalTypeTag(ownerType);

    return update(connection, sql) > 0;
  }

  /**
   * Remove the owner from all ACLs
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param ownerID TBD: Description of the incoming method parameter
   * @param ownerType TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemoveOwner(Connection connection, String ownerID, int ownerType)
    throws SQLException {

    String sql = "DELETE FROM " + OWNER_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PRINCIPAL_NAME_COL + " = '" + Escape.escapeSqlString(ownerID, PRINCIPAL_NAME_LENGTH) + "' AND " +
      PRINCIPAL_TYPE_COL + " = " + getPrincipalTypeTag(ownerType);

    return update(connection, sql) > 0;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return the members of the permission
   * @exception SQLException Exception raised in failure situation
   */
  protected IAclPermissionList uncommittedGetPermissionMembers(Connection connection, String permissionName)
    throws SQLException {

    if (permissionName.equals(IAclPermission.ACL_PERMISSION_FULL_CONTROL)) {
      return uncommittedGetAllPermissionsNotFullControl(connection);
    }

    AclPermissionList result = new AclPermissionList();

    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + PERMISSION_MEMBER_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + PERMISSION_NAME_COL + " = ?");
    st.setString(2, Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH));

    SqlResult queryResult = query(st);
    if (queryResult != null) {
      LinkedList members = new LinkedList();

      try {
        while (queryResult.m_result.next()) {
          members.add(trim(Escape.unescapeSqlString(queryResult.m_result.getString(PERMISSION_MEMBER_COL))));
        }
      }
      finally {
        queryResult.close();
      }

      Iterator iterator = members.iterator();
      while (iterator.hasNext()) {
        String memberName = (String)iterator.next();
        IAclPermission permission = uncommittedGetPermission(connection, memberName);
        if (permission != null) {
          result.add(permission);
        }
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @return a list of all permissions (WITHOUT fullcontrol)
   * @exception SQLException Exception raised in failure situation
   */
  protected IAclPermissionList uncommittedGetAllPermissionsNotFullControl(Connection connection)
    throws SQLException {
    AclPermissionList result = new AclPermissionList();
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + PERMISSION_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition);

    SqlResult queryResult = query(st);
    if (queryResult != null) {

class PermissionRec {

        public String m_name;
        public boolean m_predefined;
      }

      LinkedList foundPermissions = new LinkedList();

      try {
        while (queryResult.m_result.next()) {
          String name = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PERMISSION_NAME_COL)));
          boolean predefined = queryResult.m_result.getInt(IS_PREDEFINED_COL) != 0;

          if (!IAclPermission.ACL_PERMISSION_FULL_CONTROL.equals(name)) {
            PermissionRec rec = new PermissionRec();
            rec.m_name = name;
            rec.m_predefined = predefined;
            foundPermissions.add(rec);
          }
        }
      }
      finally {
        queryResult.close();
      }

      Iterator iterator = foundPermissions.iterator();
      while (iterator.hasNext()) {
        PermissionRec rec = (PermissionRec)iterator.next();
        IAclPermissionList members = uncommittedGetPermissionMembers(connection, rec.m_name);
        result.add(new JDBCPermission(this, rec.m_name, rec.m_predefined, members));
      }
    }

    return result;
  }

  /**
   * Remove all owners of the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  protected void uncommittedRemoveOwners(Connection connection, String internalAclID)
    throws SQLException {
    String s = Escape.escapeSqlString(internalAclID, 0);
    String sql = "DELETE FROM " + OWNER_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      ACL_HASH_COL + "=" + s.hashCode() + " AND " + ACL_ID_COL + " = '" + s + "'";

    update(connection, sql);
  }

  /**
   * Remove all ACE of the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  protected void uncommittedRemoveAclEntries(Connection connection, String internalAclID)
    throws SQLException {
    String s = Escape.escapeSqlString(internalAclID, 0);
    String sql = "DELETE FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      ACL_HASH_COL + "=" + s.hashCode() + " AND " + ACL_ID_COL + " = '" + s + "'";

    update(connection, sql);
  }


  /**
   * Remove all ACE of the ACL which belong to the specified principal
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param principalID TBD: Description of the incoming method parameter
   * @param principalType TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  protected void uncommittedRemoveAclEntries(Connection connection, String principalID, int principalType)
    throws SQLException {

    String sql = "DELETE FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PRINCIPAL_NAME_COL + " = '" + Escape.escapeSqlString(principalID, PRINCIPAL_NAME_LENGTH) + "' AND " +
      PRINCIPAL_TYPE_COL + " = " + getPrincipalTypeTag(principalType);

    update(connection, sql);
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @return true iff permissions already have been created
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckInitialPermissions(Connection connection)
    throws SQLException {

    boolean result = false;

    String sql = "SELECT * FROM " + PERMISSION_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition;

    SqlResult queryResult = query(connection, sql);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = true;
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @return true iff supported permissions already have been assigned
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckInitialSupportedPermissions(Connection connection)
    throws SQLException {

    boolean result = false;

    String sql = "SELECT * FROM " + SUPPORTED_PERMISSION_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition;

    SqlResult queryResult = query(connection, sql);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = true;
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return true iff the permission is supported for the object type
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsSupportedPermission(Connection connection, String permissionName, String objectTypeName)
    throws SQLException {

    boolean result = false;

    String sql = "SELECT * FROM " + SUPPORTED_PERMISSION_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PERMISSION_NAME_COL + " = '" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "' AND " +
      OBJECT_TYPE_COL + " = '" + Escape.escapeSqlString(objectTypeName, OBJECT_TYPE_LENGTH) + "'";

    SqlResult queryResult = query(connection, sql);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = true;
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return the supported permissions for the object type
   * @exception SQLException Exception raised in failure situation
   */
  protected IAclPermissionList uncommittedGetSupportedPermissions(Connection connection, String objectTypeName)
    throws SQLException {
    AclPermissionList result = new AclPermissionList();
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + SUPPORTED_PERMISSION_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + OBJECT_TYPE_COL + " = ?");
    st.setString(2, Escape.escapeSqlString(objectTypeName, OBJECT_TYPE_LENGTH));

    SqlResult queryResult = query(st);
    if (queryResult != null) {
      List permissionNames = new ArrayList(20);

      try {
        while (queryResult.m_result.next()) {
          String permissionName = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PERMISSION_NAME_COL)));
          permissionNames.add(permissionName);
        }
      }
      finally {
        queryResult.close();
      }

      for (int i = 0, n = permissionNames.size(); i < n; ++i) {
        String name = (String)permissionNames.get(i);
        IAclPermission permission = uncommittedGetPermission(connection, name);
        if (permission != null) {
          result.add(permission);
        }
      }
    }

    if (result != null) {
      cacheAddSupportedPermissions(objectTypeName, result);
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return IAclPermission object for the specified permission name
   * @exception SQLException Exception raised in failure situation
   */
  protected IAclPermission uncommittedGetPermission(Connection connection, String permissionName)
    throws SQLException {
    IAclPermission permission = cacheGetPermission(permissionName);
    if (permission != null) {
      return permission;
    }

    JDBCPermission result = null;
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + PERMISSION_TABLE_NAME + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + PERMISSION_NAME_COL + " = ?");
    st.setString(2, Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH));

    String name = null;
    boolean predefined = false;

    SqlResult queryResult = query(st);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          name = trim(Escape.unescapeSqlString(queryResult.m_result.getString(PERMISSION_NAME_COL)));
          predefined = queryResult.m_result.getInt(IS_PREDEFINED_COL) != 0;
        }
      }
      finally {
        queryResult.close();
      }
    }

    if (name != null) {
      IAclPermissionList members = uncommittedGetPermissionMembers(connection, name);
      result = new JDBCPermission(this, name, predefined, members);
      cacheAddPermission(result);
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return true iff the permission is user in some ACL
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsUsedPermission(Connection connection, String permissionName)
    throws SQLException {

    boolean result = false;

    String sql = "SELECT * FROM " + ACL_ENTRY_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PERMISSION_NAME_COL + " = '" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "'";

    SqlResult queryResult = query(connection, sql);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          result = true;
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * Create a new permission
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param isPredefined TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddPermission(Connection connection, String permissionName, boolean isPredefined)
    throws SQLException {

    String sql = "INSERT INTO " + PERMISSION_TABLE_NAME + " (OwnerID, permission_name, is_predefined) VALUES" +
      " (?,'" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "'," +
      (isPredefined ? "1" : "0") + ")";

    return update(connection, sql) > 0;
  }

  /**
   * Remove a permission
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemovePermission(Connection connection, String permissionName)
    throws SQLException {

    cacheRemovePermission(permissionName);

    String sql = "DELETE FROM " + PERMISSION_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PERMISSION_NAME_COL + " = '" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "'";

    return update(connection, sql) > 0;
  }

  /**
   * Assign a supported permission to an object type
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddSupportedPermission(Connection connection, String permissionName, String objectTypeName)
    throws SQLException {

    String sql = "INSERT INTO " + SUPPORTED_PERMISSION_TABLE_NAME + " (OwnerID, permission_name, object_type) VALUES (" +
      "?,'" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "'," +
      "'" + Escape.escapeSqlString(objectTypeName, OBJECT_TYPE_LENGTH) + "')";

    return update(connection, sql) > 0;
  }

  /**
   * Remove a supported permission assignment
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemoveSupportedPermission(Connection connection, String permissionName, String objectTypeName)
    throws SQLException {

    String sql = "DELETE FROM " + SUPPORTED_PERMISSION_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PERMISSION_NAME_COL + " = '" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "' AND " +
      OBJECT_TYPE_COL + " = '" + Escape.escapeSqlString(objectTypeName, OBJECT_TYPE_LENGTH) + "'";

    return update(connection, sql) > 0;
  }

  /**
   * Add a new member to a permisson
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param memberName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddPermissionMember(Connection connection, String permissionName, String memberName)
    throws SQLException {

    String sql = "INSERT INTO " + PERMISSION_MEMBER_TABLE_NAME + " (OwnerID, permission_name, member_name) VALUES (" +
      "?,'" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "'," +
      "'" + Escape.escapeSqlString(memberName, PERMISSION_MEMBER_LENGTH) + "')";

    return update(connection, sql) > 0;
  }

  /**
   * Remove a member from a permission
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param memberName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemovePermissionMember(Connection connection, String permissionName, String memberName)
    throws SQLException {

    String sql = "DELETE FROM " + PERMISSION_MEMBER_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PERMISSION_NAME_COL + " = '" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "' AND " +
      PERMISSION_MEMBER_COL + " = '" + Escape.escapeSqlString(memberName, PERMISSION_MEMBER_LENGTH) + "'";

    return update(connection, sql) > 0;
  }

  /**
   * Remove all members from a permission
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemovePermissionMembers(Connection connection, String permissionName)
    throws SQLException {

    String sql = "DELETE FROM " + PERMISSION_MEMBER_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PERMISSION_NAME_COL + " = '" + Escape.escapeSqlString(permissionName, PERMISSION_NAME_LENGTH) + "'";

    return update(connection, sql) > 0;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param name TBD: Description of the incoming method parameter
   * @return true iff the permission is predefined (and thus cannot be changed)
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsPredefinedPermission(Connection connection, String name)
    throws SQLException {

    boolean result = false;

    String sql = "SELECT * FROM " + PERMISSION_TABLE_NAME + " WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      PERMISSION_NAME_COL + " = '" + Escape.escapeSqlString(name, PERMISSION_NAME_LENGTH) + "'";

    SqlResult queryResult = query(connection, sql);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          if (queryResult.m_result.getInt(IS_PREDEFINED_COL) != 0) {
            result = true;
          }
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * Remove the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   */
  protected void uncommittedRemoveAcl(Connection connection,
  String internalAclID) throws SQLException {

    try {
      String s = Escape.escapeSqlString(internalAclID, 0);
      String sql = "DELETE FROM " + ACL_TBL + " WHERE " +
        m_ApplicationAndManagerSelectionCondition + " AND " +
        ACL_HASH_COL + "=" + s.hashCode() + " AND " + ACL_ID_COL + " = '" + s + "'";

      update(connection, sql);
    }
    catch (Exception e) {
      s_log.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));      
    }
  }

  /**
   * Lock the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param lockingUser TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedLockAcl(Connection connection, String internalAclID, IUMPrincipal lockingUser)
    throws SQLException {
    lockingUser = populate(lockingUser);
    PreparedStatement st = connection.prepareStatement(
      "UPDATE " + ACL_TBL + " SET " + LOCKING_USER_COL + " = ? "
       + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");
    st.setString(1, Escape.escapeSqlString(lockingUser.getId(), LOCKING_USER_LENGTH));

    return update(connection, st, 2, Escape.escapeSqlString(internalAclID, 0), 3) > 0;
  }

  /**
   * Unlock the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedUnlockAcl(Connection connection, String internalAclID)
    throws SQLException {
    PreparedStatement st = connection.prepareStatement(
      "UPDATE " + ACL_TBL + " SET " + LOCKING_USER_COL + " = ? "
       + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");
    st.setString(1, "");
    return update(connection, st, 2, Escape.escapeSqlString(internalAclID, 0), 3) > 0;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return true iff the ACL is locked
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsAclLocked(Connection connection, String internalAclID)
    throws SQLException {
    boolean result = false;
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + ACL_TBL + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + "=?");

    SqlResult queryResult = query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          String userID = trim(Escape.unescapeSqlString(queryResult.m_result.getString(LOCKING_USER_COL)));
          if (userID != null) {
            result = userID.length() > 0;
          }
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return the user which locks the ACL (or null if it is not locked)
   * @exception SQLException Exception raised in failure situation
   */
  protected IUser uncommittedGetAclLockingUser(Connection connection, String internalAclID)
    throws SQLException {
    IUser result = null;
    PreparedStatement st = connection.prepareStatement(
      "SELECT * FROM " + ACL_TBL + " WHERE " + m_ApplicationAndManagerSelectionCondition
       + " AND " + ACL_HASH_COL + "=? AND " + ACL_ID_COL + " = ?");

    SqlResult queryResult = query(st, 1, Escape.escapeSqlString(internalAclID, 0), 2);
    if (queryResult != null) {
      try {
        if (queryResult.m_result.next()) {
          String userID = trim(Escape.unescapeSqlString(queryResult.m_result.getString(LOCKING_USER_COL)));
          if (userID != null && userID.length() > 0) {
            result = (IUser)getUMPrincipal(userID, IUMPrincipal.IUSER, false);
          }
        }
      }
      finally {
        queryResult.close();
      }
    }

    return result;
  }


  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param oldID TBD: Description of the incoming method parameter
   * @param newID TBD: Description of the incoming method parameter
   * @return the internalAclIDs for the specified externAclIDs
   * @exception SQLException Exception raised in failure situation
   */
  /*
   * not used any more
   * private String[] uncommittedGetInternalAclIDs(String[] externalAclIDs) throws SQLException {
   * if (externalAclIDs == null) throw new java.lang.IllegalArgumentException();
   * if (externalAclIDs.length == 0) return new String[0];
   * String[] result = new String[externalAclIDs.length];
   * / resolve short ids (no db lookup necessary)
   * for (int i = 0; i < externalAclIDs.length; i++) {
   * result[i] = Escape.escapeSqlString(externalAclIDs[i], 0);
   * }
   * return result;
   * }
   */
  /**
   * Change ACL ID in the ACLs table
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param oldID TBD: Description of the incoming method parameter
   * @param newID TBD: Description of the incoming method parameter
   * @return the internalAclIDs for the specified externAclIDs
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedChangeAclID_in_AclTable(Connection connection, String oldID, String newID)
    throws SQLException {
    String o = Escape.escapeSqlString(oldID, 0);
    String n = Escape.escapeSqlString(newID, 0);

    Integer[] segments = makeHashSegments(n);

    String t = n;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
    }
    String sql = "UPDATE " + ACL_TBL + " SET " +
      ACL_HASH_COL + "=" + n.hashCode() + ", " + ACL_ID_COL + " = '" + n + "', "
       + ACL_RIDSTART_COL + "='" + t + "'";

    for (int i = 0; i < 10; i++) {
      sql += ", Seg" + (i + 1) + "=" + (segments[i] == null ? "NULL" : segments[i].toString());
    }
    sql += " WHERE " + m_ApplicationAndManagerSelectionCondition + " AND " +
      ACL_HASH_COL + "=" + o.hashCode() + " AND " + ACL_ID_COL + " = '" + o + "'";

    return update(connection, sql) > 0;
  }

  /**
   * Change ACL ID in the ACEs table
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param oldID TBD: Description of the incoming method parameter
   * @param newID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedChangeAclID_in_AclEntryTable(
  Connection connection, String oldID, String newID ) throws SQLException {
    String o = Escape.escapeSqlString(oldID, 0);
    String n = Escape.escapeSqlString(newID, 0);
    String t = n;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
    }
    String sql = "UPDATE " + ACL_ENTRY_TABLE_NAME + " SET " +
      ACL_HASH_COL + "=" + n.hashCode() + ", " + ACL_ID_COL + " = '" + n + "', "
       + ACL_RIDSTART_COL + "='" + t + "' WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      ACL_HASH_COL + "=" + o.hashCode() + " AND " + ACL_ID_COL + " = '" + o + "'";

    return update(connection, sql) > 0;
  }

  /**
   * Change ACL ID in the owners table
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param oldID TBD: Description of the incoming method parameter
   * @param newID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedChangeAclID_in_OwnerTable(Connection connection, String oldID, String newID)
    throws SQLException {
    String o = Escape.escapeSqlString(oldID, 0);
    String n = Escape.escapeSqlString(newID, 0);
    String t = n;
    if (t.length() > RID_START_LENGTH) {
      t = t.substring(0, RID_START_LENGTH);
    }
    String sql = "UPDATE " + OWNER_TABLE_NAME + " SET " +
      ACL_HASH_COL + "=" + n.hashCode() + ", " + ACL_ID_COL + " = '" + n + "', "
       + ACL_RIDSTART_COL + "='" + t + "' WHERE " +
      m_ApplicationAndManagerSelectionCondition + " AND " +
      ACL_HASH_COL + "=" + o.hashCode() + " AND " + ACL_ID_COL + " = '" + o + "'";

    return update(connection, sql) > 0;
  }

  /**
   * Execute an update SQL statement
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param sql TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  private final int update ( Connection connection, String sql )
  throws SQLException {
    return update ( connection, sql, 1 ) ;
  }

  /**
   * Execute an update SQL statement
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param st TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  private final int update ( Connection connection, PreparedStatement st )
  throws SQLException {
    return update(connection, st, 1, null, -1) ;
  }

  private int update(Connection connection, String sql, int owner)
    throws SQLException {

    int result = 0;
    PreparedStatement statement = connection.prepareStatement(sql);
    if (statement != null) {
      try {
        if (owner > 0) {
          statement.setInt(owner, locOwnerId);
        }
        result = statement.executeUpdate();
      }
      finally {
        statement.close();
      }
      statement = null;
      try {
        statement = connection.prepareStatement("UPDATE wcm_acl_version SET counter=counter+1");
        statement.executeUpdate();
      }
      catch (SQLException e) {
        s_log.warningT("update(4640)", "can't update wcm_acl_version...");
      }
      finally {
        if (statement != null) {
          statement.close();
        }
      }
    }
    return result;
  }

  private int update(Connection connection, PreparedStatement st, int owner, String rid, int ridIndex)
    throws SQLException {
    int result = 0;
    if (st != null) {
      try {
        if (owner > 0) {
          st.setInt(owner, locOwnerId);
        }
        if (rid != null) {
          st.setInt(ridIndex, rid.hashCode());
          st.setString(ridIndex + 1, rid);
        }
        result = st.executeUpdate();
      }
      finally {
        st.close();
      }
      st = null;
      try {
        st = connection.prepareStatement("UPDATE wcm_acl_version SET counter=counter+1");
        st.executeUpdate();
      }
      catch (SQLException e) {
        s_log.warningT("update(4687)", "can't update wcm_acl_version...");
      }
      finally {
        if (st != null) {
          st.close();
        }
      }
    }
    return result;
  }

  /**
   * Returns the Modifycounter of the acl tables or -1 if not possible to
   * retrieve.
   *
   * @return dBVersion
   */
  public long getDBVersion() {
    PreparedStatement st = null;
    ResultSet rs = null;
    try {
      st = m_main_connection.prepareCall("SELECT counter FROM wcm_acl_version");
      rs = st.executeQuery();
      if (rs.next()) {
        return rs.getLong(1);
      }
    }
    catch (SQLException ignore) {
      s_log.errorT("getDBVersion(4715)", "Can't retrieve DB Version!");
    }
    finally {
      try {
        if (rs != null) {
          rs.close();
        }
      }
      catch (Exception ignore) {
        s_log.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ignore));
      }
      try {
        if (st != null) {
          st.close();
        }
      }
      catch (Exception ignore) {
        s_log.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ignore));        
      }
    }
    return -1;
  }

  /**
   * Execute an query SQL statement
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param sql TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  private final SqlResult query ( Connection connection, String sql )
  throws SQLException {
    return query(connection, sql, 1) ;
  }

  /**
   * Execute an query SQL statement
   *
   * @param st TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  private final SqlResult query ( PreparedStatement st ) throws SQLException {
    return query(st, 1, null, -1) ;
  }

  private SqlResult query(Connection connection, String sql, int owner)
    throws SQLException {
    SqlResult result = new SqlResult();

    try {
      PreparedStatement st = connection.prepareStatement(sql);
      if (owner > 0) {
        st.setInt(owner, locOwnerId);
      }
      result.m_statement = st;
      result.m_result = st.executeQuery();
    }
    catch (SQLException e) {
      if (result.m_result != null) {
        result.m_result.close();
      }
      if (result.m_statement != null) {
        result.m_statement.close();
      }
      throw e;
    }

    if (result.m_statement == null || result.m_result == null) {
      if (result.m_result != null) {
        result.m_result.close();
      }
      if (result.m_statement != null) {
        result.m_statement.close();
      }
      return null;
    }

    return result;
  }

  private SqlResult query(PreparedStatement st, int owner, String rid, int ridIndex)
    throws SQLException {

    SqlResult result = new SqlResult();

    try {
      if (owner > 0) {
        st.setInt(owner, locOwnerId);
      }
      if (rid != null) {
        st.setInt(ridIndex, rid.hashCode());
        st.setString(ridIndex + 1, rid);
      }
      result.m_statement = st;
      result.m_result = st.executeQuery();
    }
    catch (SQLException e) {
      if (result.m_result != null) {
        result.m_result.close();
      }
      if (result.m_statement != null) {
        result.m_statement.close();
      }
      throw e;
    }

    if (result.m_statement == null || result.m_result == null) {
      if (result.m_result != null) {
        result.m_result.close();
      }
      if (result.m_statement != null) {
        result.m_statement.close();
      }
      return null;
    }

    return result;
  }

  protected final Connection getDBConnection() throws SQLException,
  AclPersistenceException {
    return m_main_connection ;
  }

  final protected void returnDBConnection ( Connection c ) {
    /* just nothing */
  }

  final private static Connection openConnection() throws SQLException,
  AclPersistenceException {

    JDBCConnectionPoolManager pm = JDBCConnectionPoolManager.getInstance() ;
    if ( pm == null ) {
      throw new AclPersistenceException (
        "JDBCConnectionPoolManager.getInstance() failed" ) ;
    }

    JDBCConnectionPool pool = pm.getPool ( m_poolID ) ;
    if ( pool == null ) {
      throw new AclPersistenceException (
        "JDBCConnectionPoolManager.getPool() failed" ) ;
    }

    Connection con = pool.getConnection() ;
    if ( con == null ) {
      throw new AclPersistenceException (
        "JDBCConnectionPool.getConnection() failed" ) ;
    }

    con.setAutoCommit ( false ) ;
    con.setTransactionIsolation ( Connection.TRANSACTION_READ_COMMITTED ) ;
    return con ;
  }

  final private static void closeConnection ( Connection connection )
  throws SQLException {
    if ( connection == null ) {
      throw new java.lang.IllegalArgumentException() ;
    }
    connection.close() ;
  }

  protected final void reopenMainConnection() throws SQLException,
  AclPersistenceException {
    try {
      closeConnection ( m_main_connection ) ;
    }
    catch ( Exception e1 ) {
      String s = e1.getMessage() ;
      /* log something here */
    }
    finally {
      m_main_connection = null ;
    }
    m_main_connection = openConnection() ;
  }
}
