/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;

import com.sapportals.wcm.util.acl.*;
import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionPool;
import java.util.*;
import java.sql.*;

/**
 * Factory for instances of JDBCDatabaseConnection
 */
public final class JDBCDatabaseConnectionManager {
  private static HashMap m_connections = null ;

  /**
   * @param applicationID TBD: Description of the incoming method parameter
   * @param managerID TBD: Description of the incoming method parameter
   * @param poolID TBD: Description of the incoming method parameter
   * @param permissionCache TBD: Description of the incoming method parameter
   * @return a database connection with the specified parameters
   * @exception AclLoadClassException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public static synchronized AbstractDatabaseConnectionUncached
  getDatabaseConnection ( String applicationID, String managerID,
  String poolID, com.sapportals.wcm.util.cache.ICache permissionCache )
  throws AclLoadClassException, AclPersistenceException {

    if (applicationID == null || managerID == null || poolID == null) {
      throw new java.lang.IllegalArgumentException();
    }
    
    if ( m_connections == null ) {
      m_connections = new HashMap() ;
    }
    
    /**
     * check if a connection with the specified parameters has already been
     * created
     */
    String key = getConnectionKey(applicationID, managerID, poolID);
    AbstractDatabaseConnectionUncached connection =
      (AbstractDatabaseConnectionUncached)m_connections.get(key);
    if ( connection != null ) {
      return connection;
    }

    JDBCConnectionPool p = null ;
    try {
      p = AbstractDatabaseConnectionUncached.getPool ( poolID ) ;
    }
    catch ( SQLException se ) {
      // ignored
      com.sap.tc.logging.Location.getLocation(JDBCDatabaseConnectionManager.class.getName()).debugT(se.getMessage());
    }
    if ( p == null ) {
      throw new java.lang.IllegalArgumentException() ;
    }
    if ( p.isInOpenSQLMode()) {
      connection = new OpenSQLDatabaseConnectionUncached ( applicationID,
        managerID, poolID, permissionCache, p ) ;
    }
    else {
      connection = new JDBCDatabaseConnectionUncached ( applicationID,
        managerID, poolID, permissionCache, p ) ;
    }

    /**
     * create a new connection
     */
    m_connections.put(key, connection);
    return connection;
  }

  /**
   * @param applicationID TBD: Description of the incoming method parameter
   * @param managerID TBD: Description of the incoming method parameter
   * @param poolID TBD: Description of the incoming method parameter
   * @return a unique key for a database connection
   */
  private static String getConnectionKey(String applicationID, String managerID, String poolID) {

    final String SEPARATOR = "~";

    StringBuffer key = new StringBuffer();
    key.append(applicationID);
    key.append(SEPARATOR);
    key.append(managerID);
    key.append(SEPARATOR);
    key.append(poolID);

    return key.toString();
  }

  public static synchronized void reset() {
    if ( m_connections != null ) {
      Iterator i = m_connections.values().iterator() ;
      while ( i.hasNext()) {
        ((AbstractDatabaseConnectionUncached)i.next()).terminate() ;
      }
      m_connections.clear() ;
      m_connections = null ;
    }
  }
}
