/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;


import java.util.List;

import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.util.acl.*;


/**
 * Default implementation of IAclRaw for database-persisted ACLs
 */
public final class JDBCAclRaw 
extends JDBCAcl
implements IAclRaw {

  // ==================
  // Instance Variables =======================================================
  // ==================

  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(JDBCAclRaw.class);


  // ==================
  // Instance Variables =======================================================
  // ==================

  private UMPrincipalList m_rawOwners;
  private AclEntryList m_rawEntries;


  // ===============
  // Private Methods ==========================================================
  // ===============

  // ------------------------------------------------------------------------
  private void setEntriesAcl() {

    super.m_entries = new AclEntryList();
    
    IAclEntryListIterator entryIterator = this.m_rawEntries.iterator();
    while( entryIterator.hasNext() ) {
      JDBCAclEntry entry = (JDBCAclEntry)entryIterator.next();
      if( !(entry instanceof JDBCAclEntryRaw) ) {
        super.m_entries.add(entry);
      }
      entry.setAcl(this);
    }
  }


  // ------------------------------------------------------------------------
  private void setOwnersAcl() {

    super.m_owners = new UMPrincipalList();
    
    IUMPrincipalListIterator principalIterator = this.m_rawOwners.iterator();
    while( principalIterator.hasNext() ) {
      IUMPrincipal principal = (IUMPrincipal)principalIterator.next();
      if( !(principal instanceof IUMPrincipalRaw) ) {
        super.m_owners.add(principal);
      }
    }
  }


  // ===========
  // Protected Methods ========================================================    
  // ===========

  // ------------------------------------------------------------------------
  protected void reloadAclEntries()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("reloadAclEntries(652)", "getChangeLevel");
    }

    this.m_rawOwners = m_databaseConnection.getOwners(m_aclID, true); // RAW
    this.setOwnersAcl();

    this.m_rawEntries = m_databaseConnection.getAclEntries(m_aclID, true); // RAW   
    this.setEntriesAcl();

  }


  // ===========
  // Constructor ==============================================================    
  // ===========

  // ------------------------------------------------------------------------
  public JDBCAclRaw(AbstractDatabaseConnectionUncached databaseConnection,
                    String aclID,
                    IObjectType objectType,
                    UMPrincipalList owners,
                    AclEntryList entries,
                    long changeLevel) {

    super(databaseConnection, aclID, objectType, changeLevel);

    if (   ( databaseConnection == null )
        || ( aclID == null )
        || ( objectType == null )
        || ( owners == null )
        || ( entries == null )
       ) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("JDBCAclRaw", "objectType " + objectType.getName() + ", changeLevel " + changeLevel);
    }

    this.m_rawOwners = owners; 
    this.setOwnersAcl();

    this.m_rawEntries = entries;   
    this.setEntriesAcl();
    
  }


  // ==============
  // Interface IAcl ===========================================================
  // ==============

  // ------------------------------------------------------------------------
  public IAclEntryList getEntries(IUMPrincipal principal)
                           throws AclPersistenceException {

    if (principal == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if( principal instanceof IUMPrincipalRaw ) {
      if (s_log.beDebug()) {
        s_log.debugT("getEntries", "principal " + principal.getId());
      } 
      AclEntryList result = new AclEntryList();
      IAclEntryListIterator iterator = this.m_rawEntries.iterator();
      while( iterator.hasNext() ) {
        JDBCAclEntry entry = (JDBCAclEntry)iterator.next();
        if( entry.getPrincipal().equals(principal) ) {
          result.add(entry);
        }
      }
      return result;
    } else {
      return super.getEntries(principal);
    }

  }


  // ------------------------------------------------------------------------
  public boolean isOwner(IUMPrincipal principal)
    throws AclPersistenceException {

    if (principal == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if( principal instanceof IUMPrincipalRaw ) {
      if (s_log.beDebug()) {
        s_log.debugT("isOwner", "principal " + principal.getId());
      } 
      IUMPrincipalListIterator iterator = m_rawOwners.iterator();
      while( iterator.hasNext() ) {
        IUMPrincipal listOwner = iterator.next();
        if( listOwner.equals(principal) ) {
          return true;
        }
      }
      return false;
    } else {
      return super.isOwner(principal);
    }

  }


  // ------------------------------------------------------------------------
  public boolean addEntry(IUMPrincipal caller,
                          IAclEntry aclEntry)
                   throws AclPersistenceException,
                          InvalidClassException,
                          NotAuthorizedException,
                          AlreadyAssignedToAclException,
                          PermissionNotSupportedException {

    if( basicAddEntry(caller, aclEntry) ) { 
      m_rawEntries = m_databaseConnection.getAclEntries(m_aclID, true); // RAW
      setEntriesAcl();
      return true;
    } else {
      return false;
    }
  }


  // ------------------------------------------------------------------------
  public boolean removeEntry(IUMPrincipal caller,
                             IAclEntry aclEntry)
                      throws AclPersistenceException, 
                             InvalidClassException, 
                             NotAuthorizedException {

    if( basicRemoveEntry(caller, aclEntry) ) {
      m_rawEntries = m_databaseConnection.getAclEntries(m_aclID, true); // RAW
      setEntriesAcl();
      return true;
    } else {
      return false;
    }
                               
  }


  // ------------------------------------------------------------------------
  public boolean addOwner(IUMPrincipal caller, IUMPrincipal principal)
    throws AclPersistenceException, NotAuthorizedException {

    if( basicAddOwner(caller, principal) ) {
      m_rawOwners = m_databaseConnection.getOwners(m_aclID, true); // RAW
      setOwnersAcl();
      return true;
    } else {
      return false;
    }

  }


  // ------------------------------------------------------------------------
  public boolean removeOwner(IUMPrincipal caller, IUMPrincipal owner)
    throws AclPersistenceException, NotAuthorizedException, LastOwnerException {

    if( basicRemoveOwner(caller, owner) ) {
      m_rawOwners = m_databaseConnection.getOwners(m_aclID, true); // RAW
      setOwnersAcl();
      return true;
    } else {
      return false;
    }

  }



  // =================
  // Interface IAclRaw ========================================================
  // =================

  // ------------------------------------------------------------------------
  public IUMPrincipalList getOwnersRaw()
                                throws AclPersistenceException {

    if( s_log.beDebug() ) {
      s_log.debugT("getOwnersRaw", "get raw owners");
    }

    return m_rawOwners;
    
  }
  
  
  // ------------------------------------------------------------------------
  public IAclEntryList getEntriesRaw()
                              throws AclPersistenceException {

    if( s_log.beDebug() ) {
      s_log.debugT("getEntriesRaw", "get raw entries");
    }

    return m_rawEntries;
    
  }


}
