/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;

import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigurable;
import com.sapportals.portal.security.usermanagement.IUMPrincipal;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.wcm.util.acl.*;
import com.sapportals.wcm.util.cache.*;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import java.util.List;
import java.util.Properties;

/**
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Dirk Sabiwalsky
 * @version $Id: JDBCAclManager.java,v 1.12 2002/09/02 10:06:26 sei Exp $
 */
public final class JDBCAclManager
implements IAclManagerRaw {

  private final static String CFG_PLUGIN_CM_UTILITIES_ACL_MANAGERS = "/cm/utilities/acl_managers";
  private final static String CFG_POOL_KEY = "poolid";
  private final static String CFG_ACL_CACHE_KEY = "cacheid";
  private final static String CFG_PERMISSION_CACHE_KEY = "permission_cacheid";

  private final static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.acl.jdbc.JDBCAclManager.class);

  //private static boolean mergerCalled = false;

  private final AbstractDatabaseConnectionUncached m_databaseConnection;
  private final String m_applicationID;
  private final String m_managerID;
  private IClusterCacheArea m_cacheArea = null;
  private com.sapportals.wcm.util.cache.ICache m_aclCache = null;
  private com.sapportals.wcm.util.cache.ICache m_permissionCache = null;// used by the uncached connection, too (permissions are heavily used to create ACEs there)

  /**
   * Construct
   *
   * @param applicationID TBD: Description of the incoming method parameter
   * @param managerID TBD: Description of the incoming method parameter
   * @exception AclException Exception raised in failure situation
   * @exception AclLoadClassException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception InvalidConfigException Exception raised in failure situation
   */
  public JDBCAclManager(String applicationID, String managerID, IClusterCacheArea cacheArea)
    throws AclException, AclLoadClassException, AclPersistenceException, InvalidConfigException {
    if (applicationID == null || managerID == null) {
      throw new java.lang.IllegalArgumentException();
    }

    m_applicationID = applicationID;
    m_managerID = managerID;
    this.m_cacheArea = cacheArea;

    if (s_log.beDebug()) {
      s_log.debugT("JDBCAclManager(71)", "JDBCAclManager");
    }

    /**
     * merge framework and service ACLs
     */
//    if (!mergerCalled) {
//      JDBCMergeAcls merger = new JDBCMergeAcls(applicationID);
//      mergerCalled = true;
//    }

    String poolID = null;
    String aclCacheID = null;
    String permissionCacheID = null;

    try {
      IConfigClientContext context = IConfigClientContext.createContext(
        ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_ACL_MANAGERS);
      IConfigurable configurable = plugin.getConfigurable(m_managerID);
            
      Properties properties = ConfigCrutch.getConfigurableProperties(configurable);

      poolID = properties.getProperty(CFG_POOL_KEY);
      aclCacheID = properties.getProperty(CFG_ACL_CACHE_KEY);
      permissionCacheID = properties.getProperty(CFG_PERMISSION_CACHE_KEY);
    }
    catch (Exception e) {
      throw new InvalidConfigException(e.getMessage());
    }

    createCaches(aclCacheID, permissionCacheID);
    m_databaseConnection = JDBCDatabaseConnectionManager.getDatabaseConnection(applicationID, managerID, poolID, m_permissionCache);
  }
  
  public synchronized void shutdown() {
    JDBCDatabaseConnectionManager.reset() ;
  }

  /**
   * @param aclID TBD: Description of the incoming method parameter
   * @param contextDBVersion TBD: Description of the incoming method parameter
   * @return the ACL with the specified ID (returns null if the ACL does not
   *      exist)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public synchronized IAcl getAcl(String aclID, long contextDBVersion)
    throws AclPersistenceException {
    if (aclID == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAcl", "aclID " + aclID);
    }

    try {
      return getValidAclCache(aclID, contextDBVersion);
    }
    catch (AclNotFoundException e) {
      return cacheAddAcl(aclID, m_databaseConnection.getAcl(aclID, false)); // COOKED
    }
  }

  /**
   * @param aclID TBD: Description of the incoming method parameter
   * @param contextDBVersion TBD: Description of the incoming method parameter
   * @return the ACL with the specified ID (returns null if the ACL does not
   *      exist)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public synchronized IAcl getAclRaw(String aclID,
                                     long contextDBVersion)
                              throws AclPersistenceException {
    if (aclID == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAclRaw", "aclID " + aclID);
    }

    return m_databaseConnection.getAcl(aclID, true); // RAW
    
  }

  /**
   * @param aclIDs TBD: Description of the incoming method parameter
   * @return the ACLs with the specified IDs (returns null entries for the IDs
   *      without ACLs)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public synchronized IAcl[] getAcls(String[] aclIDs)
    throws AclPersistenceException {
    if (aclIDs == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAcls", "aclIDs-length " + aclIDs.length);
    }

    if (aclIDs.length == 0) {
      return new JDBCAcl[0];
    }

    JDBCAcl[] result = new JDBCAcl[aclIDs.length];

    // check cached acls
    int missing = getValidAclCache(aclIDs, result);

    if (missing > 0) {
      int pos;

      // get array with acls that are not cached
      String[] remainingIDs = new String[missing];
      pos = 0;
      for (int i = 0; i < result.length; ++i) {
        if (result[i] == null) {
          remainingIDs[pos] = aclIDs[i];
          ++pos;
        }
      }

      // get acls from db
      JDBCAcl[] acls = m_databaseConnection.getAcls(remainingIDs, false); // COOKED

      // write acls to the cache and to the result array
      pos = 0;
      for (int i = 0; i < result.length; i++) {
        if (result[i] == null) {
          result[i] = cacheAddAcl(remainingIDs[pos], acls[pos]);
          ++pos;
        }
      }
    }

    return result;
  }

  /**
   * @param aclIDs TBD: Description of the incoming method parameter
   * @return the ACLs with the specified IDs (returns null entries for the IDs
   *      without ACLs)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public synchronized IAcl[] getAclsRaw(String[] aclIDs)
    throws AclPersistenceException {
    if (aclIDs == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAclsRaw", "aclIDs-length " + aclIDs.length);
    }

    if (aclIDs.length == 0) {
      return new JDBCAcl[0];
    }

    return m_databaseConnection.getAcls(aclIDs, true); // RAW

  }

  /**
   * @return the supported object types
   */
  public IObjectTypeList getSupportedObjectTypes() {
    if (s_log.beDebug()) {
      s_log.debugT("getSupportedObjectTypes(185)", "getSupportedObjectTypes");
    }

    return m_databaseConnection.getSupportedObjectTypes();
  }

  /**
   * @param objectType TBD: Description of the incoming method parameter
   * @return the permissions which are supported for the specified object type
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public IAclPermissionList getSupportedPermissions(IObjectType objectType)
    throws AclPersistenceException {
    if (objectType == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getSupportedPermissions(203)", "objectType " + objectType.getName());
    }

    return m_databaseConnection.getSupportedPermissions(objectType);
  }

  /**
   * @param name TBD: Description of the incoming method parameter
   * @return an IAclPermission object for the specified permission
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public IAclPermission getPermission(String name)
    throws AclPersistenceException {
    if (name == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getPermission(221)", "name " + name);
    }

    return m_databaseConnection.getPermission(name);
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @return true iff the specified permission is used in some ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean isPermissionUsedInAcl(IAclPermission permission)
    throws AclPersistenceException {
    if (permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("isPermissionUsedInAcl(239)", "permission " + permission.getName());
    }

    return m_databaseConnection.isPermissionUsedInAcl(permission);
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @return a list of all ACLs which use the specified permission
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public List getAffectedAclIDs(IAclPermission permission)
    throws AclPersistenceException {
    if (permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAffectedAclIDs(257)", "permission " + permission.getName());
    }

    return m_databaseConnection.getAffectedAclIDs(permission);
  }

  /**
   * @return true iff this ACL manager is read only
   */
  public boolean isReadOnly() {
    if (s_log.beDebug()) {
      s_log.debugT("isReadOnly(268)", "isReadOnly");
    }

    return false;
  }

  /**
   * Create an ACE
   *
   * @param principal TBD: Description of the incoming method parameter
   * @param negative TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param sortIndex TBD: Description of the incoming method parameter
   * @param propagate TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception InvalidClassException Exception raised in failure situation
   */
  public IAclEntry createAclEntry(IUMPrincipal principal, boolean negative, IAclPermission permission, int sortIndex, boolean propagate)
    throws AclPersistenceException, InvalidClassException {
    if (principal == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("createAclEntry(293)", "principal " + principal.getId() + ", negative " + negative + ", permission " + permission.getName() + ", sortIndex " + sortIndex + ", propagate " + propagate);
    }

    if (!(permission instanceof JDBCPermission)) {
      throw new InvalidClassException();
    }

    return new JDBCAclEntry(m_databaseConnection, principal, negative, permission, sortIndex, propagate);
  }

  /**
   * Create an ACL
   *
   * @param owner TBD: Description of the incoming method parameter
   * @param aclID TBD: Description of the incoming method parameter
   * @param objectType TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception AclExistsException Exception raised in failure situation
   */
  public synchronized IAcl createAcl(IUMPrincipal owner, String aclID, IObjectType objectType)
    throws AclPersistenceException, AclExistsException {
    if (owner == null || aclID == null || objectType == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("createAcl(320)", "owner " + owner.getId() + ", aclID " + aclID + ", objectType " + objectType.getName());
    }

    return cacheAddAcl(aclID, m_databaseConnection.createAcl(owner, aclID, objectType));
  }

  /**
   * For each ACL check whether the principal has the specified permissions
   *
   * @param acls TBD: Description of the incoming method parameter
   * @param principal TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean[] checkPermissions(IAcl[] acls, IUMPrincipal principal, IAclPermission[] permissions)
    throws AclPersistenceException {
    if (acls == null || principal == null || permissions == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("checkPermissions(342)", "acls-length " + acls.length + ", principal " + principal.getId() + ", permissions-length " + permissions.length);
    }

    if (acls.length == 0) {
      return new boolean[0];
    }

    boolean[] result = new boolean[acls.length];

    if (!(principal instanceof IUser) || !((IUser)principal).isAuthenticated()) {
      return result;
    }

    if (permissions.length == 0) {
      for (int i = 0; i < result.length; i++) {
        result[i] = true;
      }
      return result;
    }

    for (int i = 0; i < acls.length; i++) {
      result[i] = ((JDBCAcl)acls[i]).checkPermissions(principal, permissions);
    }

    return result;
  }

  /**
   * Remove an ACL
   *
   * @param caller TBD: Description of the incoming method parameter
   * @param aclID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public boolean removeAcl(IUMPrincipal caller, String aclID)
    throws AclPersistenceException, NotAuthorizedException {
    if (caller == null || aclID == null) {
      throw new java.lang.IllegalArgumentException();
    }

    IAcl acl = getAcl(aclID, -1);
    if (acl == null) {
      return false;
    }

    try {
      return removeAcl(caller, acl);
    }
    catch (InvalidClassException e) {
      return false;
    }
  }

  public synchronized boolean removeAcl(IUMPrincipal caller, IAcl acl)
    throws AclPersistenceException, InvalidClassException, NotAuthorizedException {
    if (caller == null || acl == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (!(acl instanceof JDBCAcl)) {
      throw new InvalidClassException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("removeAcl(407)", "caller " + caller.getId() + ", acl " + ((JDBCAcl)acl).getID());
    }

    if (!acl.isAuthorized(caller)) {
      throw new NotAuthorizedException();
    }

    if (!m_databaseConnection.removeAcl(((JDBCAcl)acl).getID())) {
      return false;
    }

    cacheRemoveAcl(((JDBCAcl)acl).getID());
    return true;
  }

  /**
   * Ment to be called as reaction to an event from the usermanagement: a
   * principal has been removed: clean up the ACL database
   *
   * @param principalID TBD: Description of the incoming method parameter
   * @param principalType TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public synchronized void deletePrincipal(String principalID, int principalType)
    throws AclPersistenceException {
    if (principalID == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("deletePrincipal(437)", "principalID " + principalID + ", principalType " + principalType);
    }

    m_databaseConnection.deletePrincipal(principalID, principalType);

    cacheClearAcls();
  }

  /**
   * Ment to be called as reaction to an event from the usermanagement: a
   * principal has been changed: clear the cache (this is the lazy approach)
   *
   * @param principalID TBD: Description of the incoming method parameter
   * @param principalType TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public synchronized void updatePrincipal(String principalID, int principalType)
    throws AclPersistenceException {
    if (principalID == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("updatePrincipal(460)", "principalID " + principalID + ", principalType " + principalType);
    }

    cacheClearAcls();
  }

  /**
   * Define a permission to be applied for the specified object type
   *
   * @param objectType supportedPermission to be added
   * @param permission supportedPermission to be added
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   */
  public boolean addSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException {
    if (objectType == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("addSupportedPermission(483)", "objectType " + objectType.getName() + ", permission " + permission.getName());
    }

    return m_databaseConnection.addSupportedPermission(objectType, permission);
  }

  /**
   * Define a permission to NOT be applied anymore for the specified object type
   *
   * @param objectType TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public boolean removeSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException {
    if (objectType == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("removeSupportedPermission(507)", "objectType " + objectType.getName() + ", permission " + permission.getName());
    }

    return m_databaseConnection.removeSupportedPermission(objectType, permission);
  }

  /**
   * Create a permission
   *
   * @param name TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PermissionExistsException Exception raised in failure situation
   */
  public IAclPermission createPermission(String name)
    throws AclPersistenceException, PermissionExistsException {
    if (name == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("createPermission(528)", "name " + name);
    }

    return m_databaseConnection.createPermission(name);
  }

  /**
   * Remove a permission
   *
   * @param permission TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public boolean removePermission(IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException {
    if (permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("removePermission(551)", "permission " + permission.getName());
    }

    return m_databaseConnection.removePermission(permission);
  }

  /**
   * Change the ID of an ACL
   *
   * @param caller TBD: Description of the incoming method parameter
   * @param acl TBD: Description of the incoming method parameter
   * @param newID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   * @exception AclExistsException Exception raised in failure situation
   */
  public synchronized boolean changeAclID(IUMPrincipal caller, IAcl acl, String newID)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException {
    if (caller == null || acl == null || newID == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("changeAclID(575)", "caller " + caller.getId() + ", newID " + newID);
    }

    String oldID = ((JDBCAcl)acl).getID();
    if (!m_databaseConnection.changeAclID(oldID, newID)) {
      return false;
    }

    ((JDBCAcl)acl).setID(newID);
    cacheRemoveAcl(oldID);
    return true;
  }

  /**
   * Change the IDs of all ACLs that start with the oldAclIdPrefix so that they
   * start with the newAclIdPrefix (just replace the old prefix by the new one)
   * (rename folder)
   *
   * @param oldAclIdPrefix TBD: Description of the incoming method parameter
   * @param newAclIdPrefix TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception AclExistsException Exception raised in failure situation
   */
  public synchronized void changeMultipleAclIDs(String oldAclIdPrefix, String newAclIdPrefix)
    throws AclPersistenceException, AclExistsException {
    if (newAclIdPrefix == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("changeMultipleAclIDs(605)", "oldAclIdPrefix " + oldAclIdPrefix + ", newAclIdPrefix " + newAclIdPrefix);
    }

    m_databaseConnection.changeMultipleAclIDs(oldAclIdPrefix, newAclIdPrefix);

    cacheRemoveMultipleAcls(oldAclIdPrefix);
  }

  /**
   * Remove all ACL whose ID start with the aclIdPrefix (remove children)
   *
   * @param aclIdPrefix TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public synchronized void removeMultipleAcls(String aclIdPrefix)
    throws AclPersistenceException {
    if (aclIdPrefix == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("removeMultipleAcls(626)", "aclIdPrefix " + aclIdPrefix);
    }

    m_databaseConnection.removeMultipleAcls(aclIdPrefix);

    cacheRemoveMultipleAcls(aclIdPrefix);
  }

  /**
   * isAclUpToDate() masscall
   *
   * @param acls TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean[] areAclsUpToDate(IAcl[] acls)
    throws AclPersistenceException {
    if (acls == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (s_log.beDebug()) {
      s_log.debugT("areAclsUpToDate(648)", "acls-length " + acls.length);
    }
    return m_databaseConnection.areAclsUpToDate((JDBCAcl[])acls) ;
  }

  public long getDBVersion() {
    return m_databaseConnection.getDBVersion();
  }


  public IClusterCacheArea getCacheArea() {
    return this.m_cacheArea;
  }
  
  
  /**
   * Create caches
   *
   * @param aclCacheID TBD: Description of the incoming method parameter
   * @param permissionCacheID TBD: Description of the incoming method parameter
   */
  private void createCaches(String aclCacheID, String permissionCacheID) {
    if (aclCacheID == null && permissionCacheID == null) {
      return;
    }

    try {
      CacheFactory cacheFactory = CacheFactory.getInstance();
      if (cacheFactory != null) {
        if (aclCacheID != null) {
          m_aclCache = cacheFactory.getCache(aclCacheID);
        }
        if (permissionCacheID != null) {
          m_permissionCache = cacheFactory.getCache(permissionCacheID);
        }
      }
    }
    catch (CacheException e) {
      s_log.warningT("createCaches(688)", "failed to create caches: " + LoggingFormatter.extractCallstack(e));
    }
  }

  private JDBCAcl cacheAddAcl(String aclID) {
    if (m_aclCache != null) {
      try {
        m_aclCache.addEntry(aclID, new Long(m_databaseConnection.getDBVersion()));
      }
      catch (CacheException e) {
        s_log.warningT("cacheAddAcl(698)", "acl cache error: " + LoggingFormatter.extractCallstack(e));
      }
    }

    return null;
  }

  /**
   * Add an ACL to the cache
   *
   * @param aclID TBD: Description of the incoming method parameter
   * @param acl TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private JDBCAcl cacheAddAcl(String aclID, JDBCAcl acl) {
    if (acl == null) {
      return cacheAddAcl(aclID);
    }
    if (m_aclCache == null) {
      return acl;
    }

    try {
      m_aclCache.addEntry(aclID, acl);
    }
    catch (CacheException e) {
      s_log.warningT("cacheAddAcl(724)", "acl cache error: " + LoggingFormatter.extractCallstack(e));
    }
    return acl;
  }

  /**
   * Remove an ACL from the cache
   *
   * @param aclID TBD: Description of the incoming method parameter
   */
  private void cacheRemoveAcl(String aclID) {
    if (m_aclCache == null) {
      return;
    }

    if( this.m_cacheArea != null ) {
      this.m_cacheArea.invalidate();
    }

    try {
      m_aclCache.removeEntry(aclID);
    }
    catch (CacheException e) {
      s_log.warningT("cacheRemoveAcl(743)", "acl cache error: " + LoggingFormatter.extractCallstack(e));
    }
  }

  /**
   * Remove all ACLs from the cache
   */
  private void cacheClearAcls() {
    if (m_aclCache == null) {
      return;
    }

    if( this.m_cacheArea != null ) {
      this.m_cacheArea.invalidate();
    }

    try {
      m_aclCache.clearCache();
    }
    catch (CacheException e) {
      s_log.warningT("cacheClearAcls(759)", "acl cache error: " + LoggingFormatter.extractCallstack(e));
    }
  }

  /**
   * Remove all ACLs from the cache whose IDs start with rootAclIdPrefix (remove
   * children)
   *
   * @param rootAclIdPrefix TBD: Description of the incoming method parameter
   */
  private void cacheRemoveMultipleAcls(String rootAclIdPrefix) {
    if (m_aclCache == null) {
      return;
    }

    if( this.m_cacheArea != null ) {
      this.m_cacheArea.invalidate();
    }

    try {
      m_aclCache.removeEntriesStartingWith(rootAclIdPrefix);
    }
    catch (CacheException e) {
      s_log.warningT("cacheRemoveMultipleAcls(778)", "acl cache error: " + LoggingFormatter.extractCallstack(e));
    }
  }

  /**
   * Remove the ACL from the cache if it is not up to date
   *
   * @param aclID TBD: Description of the incoming method parameter
   * @param contextDBVersion TBD: Description of the incoming method parameter
   * @return validAclCache
   * @exception AclNotFoundException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   */
  private JDBCAcl getValidAclCache(String aclID, long contextDBVersion)
  throws AclNotFoundException, AclPersistenceException {
    if (m_aclCache != null) {
      try {
        ICacheEntry cacheEntry = m_aclCache.getEntry(aclID);
        if (cacheEntry != null) {
          Object o = cacheEntry.getObject();
          if (o != null) {
            if (o instanceof Long) {
              if ((this.m_cacheArea != null) && (this.m_cacheArea.isValid())) {
                return null; // cache is valid and there is no ACL for the ID
              }
              else {
                // cache has become invalid
                if (this.m_cacheArea != null) {
                  this.m_cacheArea.refresh(this.m_aclCache);
                  this.m_cacheArea.validate();
                }
                boolean invalidate = false;
                if (contextDBVersion < 0) {
                  invalidate = (m_databaseConnection.getDBVersion()
                    > ((Long)o).longValue());
                }
                else {
                  invalidate = (contextDBVersion > ((Long)o).longValue());
                }
                if (invalidate) {
                  m_aclCache.removeEntry(cacheEntry);
                }
                else {
                  return null;
                }
              }
            }
            else if (o instanceof JDBCAcl) {
              JDBCAcl acl = (JDBCAcl)o;
              if ((this.m_cacheArea != null) && (this.m_cacheArea.isValid())) {
                return acl;
              }
              else {
                if (this.m_cacheArea != null) {
                  this.m_cacheArea.validate();
                }
                if (!m_databaseConnection.isAclUpToDate(acl)) {
                  m_aclCache.removeEntry(cacheEntry);
                }
                else {
                  return acl;
                }
              }
            }
          }
        }
      }
      catch (CacheException e) {
        s_log.warningT(
          "getValidAclCache(828)",
          "acl cache error: " + LoggingFormatter.extractCallstack(e));
      }
    }

    throw new AclNotFoundException();
  }

  /**
   * Return the valid JDBCAcls and remove the rest from the cache.
   *
   * @param aclIDs TBD: Description of the incoming method parameter
   * @param result TBD: Description of the incoming method parameter
   * @return number of missing acls
   * @exception AclPersistenceException Exception raised in failure situation
   */
  private int getValidAclCache(String[] aclIDs, JDBCAcl[] result)
    throws AclPersistenceException {
    int missing = result.length;
    if (aclIDs.length != result.length) {
      throw new IllegalArgumentException("parameters sizes differ");
    }

    if (m_aclCache != null) {
      try {
        int nCached = 0, found = 0;
        long t0 = System.currentTimeMillis() ;
        for (int i = 0; i < aclIDs.length; i++) {
          ICacheEntry cacheEntry = m_aclCache.getEntry(aclIDs[i]);
          if ((cacheEntry != null) && (cacheEntry.getObject() instanceof JDBCAcl)) {
            JDBCAcl acl = (JDBCAcl)cacheEntry.getObject();
            if (acl == null) {
              m_aclCache.removeEntry(cacheEntry);
            }
            else {
              result[i] = acl;
              if ( acl.getDeadline() <= t0 ) {
                found++;
              }
              else {
                nCached++ ;
              }
            }
          }
        }
        missing -= nCached ;

        if (found > 0) {
          String[] checkIDs = new String[found];
          long[] checkLevels = new long[found];

          int pos = 0;
          for (int i = 0; i < result.length; ++i) {
            JDBCAcl acl = result[i];
            if (acl != null && acl.getDeadline() <= t0 ) {
              checkIDs[pos] = aclIDs[i];
              checkLevels[pos] = acl.getChangeLevel();
              ++pos;
            }
          }
          boolean[] upToDateFlags = m_databaseConnection.areAclsUpToDate(checkIDs, checkLevels);
          pos = 0;
          for (int i = 0; i < result.length; ++i) {
            JDBCAcl acl = result[i];
            if (acl != null) {
              if (!upToDateFlags[pos]) {
                m_aclCache.removeEntry(checkIDs[pos]);
                --found;
                result[i] = null;
                ++pos;
              }
              else {
                acl.extendDeadline() ;
              }
            }
          }
          missing -= found ;
        }
      }
      catch (CacheException e) {
        s_log.warningT("getValidAclCache(898)", "acl cache error: " + LoggingFormatter.extractCallstack(e));
      }
    }

    return missing;
  }
}
