/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;
import com.sapportals.portal.security.usermanagement.IUMPrincipal;
import com.sapportals.wcm.util.acl.*;

/**
 * Default implementation of IAclEntry for database-persisted ACLs Methods which
 * change the ACE may be used by any caller unless the ACE is part of an ACL;
 * then the caller needs to be an owner of the ACL; an ACE can only be part of
 * one ACL
 */
public class JDBCAclEntry implements IAclEntry {
  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.acl.jdbc.JDBCAclEntry.class);

  private long m_ID = -1;// set, when the ace is added to an acl (and thus to the database)
  private String m_aclID = null;// set, when the ace is added to an acl (and thus to the database)
  private JDBCAcl m_acl = null;// set, when the ace is added to an acl

  private AbstractDatabaseConnectionUncached m_databaseConnection = null;

  private IUMPrincipal m_principal = null;
  private boolean m_negative = false;
  private IAclPermission m_permission = null;

  private boolean m_propagate = false;// may be changed in database; use m_databaseConnection to request
  private int m_sortIndex = 0;// may be changed in database; use m_databaseConnection to request

  /**
   * Construct
   *
   * @param databaseConnection TBD: Description of the incoming method parameter
   * @param principal TBD: Description of the incoming method parameter
   * @param negative TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param sortIndex TBD: Description of the incoming method parameter
   * @param propagate TBD: Description of the incoming method parameter
   * @exception InvalidClassException Exception raised in failure situation
   */
  public JDBCAclEntry(AbstractDatabaseConnectionUncached databaseConnection, IUMPrincipal principal, boolean negative, IAclPermission permission, int sortIndex, boolean propagate)
    throws InvalidClassException {

    if (databaseConnection == null || principal == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (!(permission instanceof JDBCPermission)) {
      throw new InvalidClassException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("JDBCAclEntry(64)", "principal " + principal.getId() + ", negative " + negative + ", permission " + ((JDBCPermission)permission).getName() + ", sortIndex " + sortIndex + ", propagate " + propagate);
    }

    init(databaseConnection, principal, negative, (JDBCPermission)permission, sortIndex, propagate);
  }

  public JDBCAclEntry(AbstractDatabaseConnectionUncached databaseConnection, IUMPrincipal principal, boolean negative, JDBCPermission permission, int sortIndex, boolean propagate) {

    if (databaseConnection == null || principal == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("JDBCAclEntry(77)", "principal " + principal.getId() + ", negative " + negative + ", permission " + permission.getName() + ", sortIndex " + sortIndex + ", propagate " + propagate);
    }

    init(databaseConnection, principal, negative, (JDBCPermission)permission, sortIndex, propagate);
  }

  /**
   * Set the sort index
   *
   * @param caller sortIndex to be set
   * @param sortIndex sortIndex to be set
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public final void setSortIndex(IUMPrincipal caller, int sortIndex)
    throws AclPersistenceException, NotAuthorizedException {

    if (s_log.beDebug()) {
      s_log.debugT("setSortIndex(95)", "caller " + caller.getId() + ", sortIndex " + sortIndex);
    }

    // caller may be null in case the ace is not yet part of an acl
    if (m_ID <= 0 || m_aclID == null) {
      m_sortIndex = sortIndex;
      return;
    }

    if (caller == null) {
      throw new java.lang.IllegalArgumentException();
    }

    JDBCAcl acl = m_databaseConnection.getAcl(m_aclID, false); // COOKED
    if (acl == null || !acl.isAuthorized(caller)) {
      throw new NotAuthorizedException();
    }

    if (m_databaseConnection.setAclEntrySortIndex(m_ID, sortIndex)) {
      m_sortIndex = sortIndex;
      if (m_acl != null) {
        m_acl.reloadAclEntries();
      }
    }
  }

  /**
   * Set the propagation flag
   *
   * @param caller propagation to be set
   * @param propagate propagation to be set
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public final void setPropagation(IUMPrincipal caller, boolean propagate)
    throws AclPersistenceException, NotAuthorizedException {

    if (s_log.beDebug()) {
      s_log.debugT("setPropagation(133)", "caller " + caller.getId() + ", propagate " + propagate);
    }

    // caller may be null in case the ace is not yet part of an acl
    if (m_ID <= 0 || m_aclID == null) {
      m_propagate = propagate;
      return;
    }

    if (caller == null) {
      throw new java.lang.IllegalArgumentException();
    }

    JDBCAcl acl = m_databaseConnection.getAcl(m_aclID, false); // COOKED
    if (acl == null || !acl.isAuthorized(caller)) {
      throw new NotAuthorizedException();
    }

    if (m_databaseConnection.setAclEntryPropagation(m_ID, propagate)) {
      m_propagate = propagate;
      if (m_acl != null) {
        m_acl.reloadAclEntries();
      }
    }
  }

  /**
   * @return the permission
   */
  public final IAclPermission getPermission() {

    if (s_log.beDebug()) {
      s_log.debugT("getPermission(165)", "getPermission");
    }

    return m_permission;
  }

  /**
   * @return true iff the ACE is a deny
   */
  public final boolean isNegative() {

    if (s_log.beDebug()) {
      s_log.debugT("isNegative(177)", "isNegative");
    }

    return m_negative;
  }

  /**
   * @return the principal
   */
  public final IUMPrincipal getPrincipal() {

    if (s_log.beDebug()) {
      s_log.debugT("getPrincipal(189)", "getPrincipal");
    }

    return m_principal;
  }

  /**
   * @return true iff the ACE is read only
   */
  public final boolean isReadOnly() {

    if (s_log.beDebug()) {
      s_log.debugT("isReadOnly(201)", "isReadOnly");
    }

    return false;
  }

  /**
   * @return the sort index of the ACE
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final int getSortIndex()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("getSortIndex(215)", "getSortIndex");
    }

    return m_sortIndex;
  }

  /**
   * @return the propagation flag of the ACE
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean isPropagated()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("isPropagated(229)", "isPropagated");
    }

    return m_propagate;
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @return true iff the ACE permission contains the specified one
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean checkPermission(IAclPermission permission)
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("checkPermission(244)", "permission " + permission.getName());
    }

    return m_permission.isContained(permission);
  }

  /**
   * Set the ACE ID and the ACL ID This method is called when the ACE is
   * assigned to an ACL
   *
   * @param id iDs to be set
   * @param aclID iDs to be set
   */
  protected final void setIDs(long id, String aclID) {

    if (s_log.beDebug()) {
      s_log.debugT("setIDs(260)", "id " + id + ", aclID " + aclID);
    }

    m_ID = id;
    m_aclID = aclID;
  }

  /**
   * @return the ACE ID
   */
  protected final long getID() {

    if (s_log.beDebug()) {
      s_log.debugT("getID(273)", "getID");
    }

    return m_ID;
  }

  /**
   * @return the ACL ID
   */
  protected final String getAclID() {

    if (s_log.beDebug()) {
      s_log.debugT("getAclID(285)", "getAclID");
    }

    return m_aclID;
  }

  /**
   * Clear the ACE ID and the ACL ID This method is called when the ACE is
   * removed from its ACL
   */
  protected final void clearIDs() {

    if (s_log.beDebug()) {
      s_log.debugT("clearIDs(298)", "clearIDs");
    }

    m_ID = -1;
    m_aclID = null;
    m_acl = null;
  }

  /**
   * Set the related ACL
   *
   * @param acl acl to be set
   */
  protected final void setAcl(JDBCAcl acl) {

    if (s_log.beDebug()) {
      s_log.debugT("setAcl(314)", "setAcl");
    }

    m_acl = acl;
  }

  /**
   * Initialize the ACE
   *
   * @param databaseConnection TBD: Description of the incoming method parameter
   * @param principal TBD: Description of the incoming method parameter
   * @param negative TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param sortIndex TBD: Description of the incoming method parameter
   * @param propagate TBD: Description of the incoming method parameter
   */
  private void init(AbstractDatabaseConnectionUncached databaseConnection, IUMPrincipal principal, boolean negative, JDBCPermission permission, int sortIndex, boolean propagate) {

    m_databaseConnection = databaseConnection;
    m_principal = principal;
    m_negative = negative;
    m_permission = permission;
    m_sortIndex = sortIndex;
    m_propagate = propagate;
  }

  public boolean equals(Object another) {
    if (another == null || !(another instanceof JDBCAclEntry)) {
      return false;
    }

    JDBCAclEntry x = (JDBCAclEntry)another;
    try {
      return this.getPermission().equals(x.getPermission())
        && this.getPrincipal().equals(x.getPrincipal())
        && this.m_negative == x.m_negative ;
    }
    catch (Exception ignore) {
      return false;
    }
  }

  public int hashCode() {
    return this.getPrincipal().hashCode() + this.getPermission().hashCode();
  }

}
