/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;

import com.sapportals.portal.security.usermanagement.IGroup;
import com.sapportals.portal.security.usermanagement.IRole;
import com.sapportals.portal.security.usermanagement.IUMPrincipal;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.wcm.util.acl.*;
import com.sapportals.wcm.util.systemconfig.SystemConfig;

import java.util.List;

/**
 * Default implementation of IAcl for database-persisted ACLs
 */
public class JDBCAcl implements IAcl {
  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.acl.jdbc.JDBCAcl.class);

  protected AbstractDatabaseConnectionUncached m_databaseConnection = null;
  protected String m_aclID = null;
  protected IObjectType m_objectType = null;
  protected UMPrincipalList m_owners;
  protected AclEntryList m_entries;
  protected long m_changeLevel = 0;
  private long deadline = System.currentTimeMillis() + 1000 ;
  
  protected final long getDeadline() {
    return deadline ;
  }
  
  protected final void extendDeadline() {
    deadline = System.currentTimeMillis() + 1000 ;
  }

  /**
   * Construct a JDBCAcl
   */
  protected JDBCAcl(AbstractDatabaseConnectionUncached databaseConnection,
    String aclID,
    IObjectType objectType,
    long changeLevel) {

    m_aclID = aclID;
    m_databaseConnection = databaseConnection;
    m_objectType = objectType;
    m_changeLevel = changeLevel;
    m_owners = null;
    m_entries = null;

  }


  /**
   * Construct a JDBCAcl
   *
   * @param databaseConnection TBD: Description of the incoming method parameter
   * @param aclID TBD: Description of the incoming method parameter
   * @param objectType TBD: Description of the incoming method parameter
   * @param owners TBD: Description of the incoming method parameter
   * @param entries TBD: Description of the incoming method parameter
   * @param changeLevel TBD: Description of the incoming method parameter
   */
  public JDBCAcl(AbstractDatabaseConnectionUncached databaseConnection,
    String aclID,
    IObjectType objectType,
    UMPrincipalList owners,
    AclEntryList entries,
    long changeLevel) {

    if (databaseConnection == null
       || aclID == null
       || objectType == null
       || owners == null
       || entries == null) {
      throw new java.lang.IllegalArgumentException();
    }

    m_aclID = aclID;

    if (s_log.beDebug()) {
      s_log.debugT("JDBCAcl(73)", "objectType " + objectType.getName() + ", changeLevel " + changeLevel);
    }

    m_databaseConnection = databaseConnection;
    m_objectType = objectType;
    m_owners = owners;
    m_entries = entries;
    m_changeLevel = changeLevel;

    setEntriesAcl();
  }

  /**
   * @return the ACEs of this ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final IAclEntryList getEntries()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("getEntries(93)", "getEntries");
    }

    return m_entries;
  }

  /**
   * @param principal TBD: Description of the incoming method parameter
   * @return the ACEs of this ACL which are assigned to a specific principal
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public IAclEntryList getEntries(IUMPrincipal principal)
    throws AclPersistenceException {

    if (principal == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getEntries(112)", "principal " + principal.getId());
    }

    AclEntryList result = new AclEntryList();

    IAclEntryListIterator iterator = m_entries.iterator();
    while (iterator.hasNext()) {
      JDBCAclEntry entry = (JDBCAclEntry)iterator.next();
      if (entry.getPrincipal().equals(principal)) {
        result.add(entry);
      }
    }

    return result;
  }

  /**
   * @return true iff this ACL is read only
   */
  public final boolean isReadOnly() {

    if (s_log.beDebug()) {
      s_log.debugT("isReadOnly(134)", "isReadOnly");
    }

    return false;
  }

  /**
   * @param principal TBD: Description of the incoming method parameter
   * @return true iff the specified principal is an owner of this ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean isOwner(IUMPrincipal principal)
    throws AclPersistenceException {

    if (principal == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("isOwner(153)", "principal " + principal.getId());
    }

    try {
      if (SystemConfig.getInstance().canChangePermissions(principal)) {
        return true;
      }
    }
    catch (Exception e) {
      // ignore
      s_log.debugT(e.getMessage());
    }

    IUMPrincipalListIterator iterator = m_owners.iterator();
    while (iterator.hasNext()) {
      IUMPrincipal listOwner = iterator.next();
      if (listOwner.equals(principal)) {
        return true;
      }
    }

    return false;
  }

  /**
   * @return the ACL ID
   */
  public final String getID() {

    if (s_log.beDebug()) {
      s_log.debugT("getID(182)", "getID");
    }

    return m_aclID;
  }

  /**
   * @return all owners of this ACL in a list
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final IUMPrincipalList getOwners()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("getOwners(196)", "getOwners");
    }

    return m_owners;
  }

  /**
   * @return true iff this ACL is locked (the ACL locks are cooperative)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean isLocked()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("isLocked(210)", "isLocked");
    }

    return m_databaseConnection.isAclLocked(m_aclID);
  }

  /**
   * @return the user which locked this ACL (null if the ACL is not locked)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final IUMPrincipal getLockingUser()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("getLockingUser(224)", "getLockingUser");
    }

    return m_databaseConnection.getAclLockingUser(m_aclID);
  }

  /**
   * @param postFix TBD: Description of the incoming method parameter
   * @return a list of ACL IDs (strings) of all ACLs whose IDs start with the ID
   *      of this ACL followed by the specified postfix
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final List getDescendantsWithAcl(String postFix)
    throws AclPersistenceException {

    if (postFix == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("getDescendantsWithAcl(244)", "postFix " + postFix);
    }

    return m_databaseConnection.getDescendantsWithAcl(m_aclID + postFix);
  }

  /**
   * @param principal TBD: Description of the incoming method parameter
   * @return true iff the principal is authorized to change this ACL (and
   *      thereby has full control)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean isAuthorized(IUMPrincipal principal)
    throws AclPersistenceException {

    if (principal == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("isAuthorized(264)", "principal " + principal.getId());
    }

    if (!(principal instanceof IUser)) {
      return false;
    }

    IUser user = (IUser)principal;
    if (!user.isAuthenticated()) {
      return false;
    }

    try {
      if (SystemConfig.getInstance().canChangePermissions(user)) {
        return true;
      }
    }
    catch (Exception e) {
      // ignore
      s_log.debugT(e.getMessage());      
    }

    IUMPrincipalListIterator ownerIterator = m_owners.iterator();
    while (ownerIterator.hasNext()) {
      IUMPrincipal listOwner = ownerIterator.next();
      if (containsUser(listOwner, (IUser)user)) {
        return true;
      }
    }

    return false;
  }

  /**
   * Add an ACE to this ACL
   *
   * @param caller entry to be added
   * @param aclEntry entry to be added
   * @return true iff successful
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception InvalidClassException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   * @exception AlreadyAssignedToAclException Exception raised in failure
   *      situation
   * @exception PermissionNotSupportedException Exception raised in failure
   *      situation
   */
  public boolean addEntry(IUMPrincipal caller, IAclEntry aclEntry)
    throws AclPersistenceException, InvalidClassException, NotAuthorizedException, AlreadyAssignedToAclException, PermissionNotSupportedException {

    if( basicAddEntry(caller, aclEntry) ) {
      m_entries = m_databaseConnection.getAclEntries(m_aclID, false); // COOKED
      setEntriesAcl();
      return true;
    } else {
      return false;
    }
    
  }

  /**
   * Remove an ACE from this ACL
   *
   * @param caller TBD: Description of the incoming method parameter
   * @param aclEntry TBD: Description of the incoming method parameter
   * @return true iff successful
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception InvalidClassException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public boolean removeEntry(IUMPrincipal caller, IAclEntry aclEntry)
    throws AclPersistenceException, InvalidClassException, NotAuthorizedException {

    if( basicRemoveEntry(caller, aclEntry) ) {
      m_entries = m_databaseConnection.getAclEntries(m_aclID, false); // COOKED
      setEntriesAcl();
      return true;
    } else {
      return false;
    }
    
  }

  /**
   * Add an owner to this ACL
   *
   * @param caller owner to be added
   * @param principal owner to be added
   * @return true iff successful
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public boolean addOwner(IUMPrincipal caller, IUMPrincipal principal)
    throws AclPersistenceException, NotAuthorizedException {

    if( basicAddOwner(caller, principal) ) {
      m_owners = m_databaseConnection.getOwners(m_aclID, false); // COOKED
      return true;
    } else {
      return false;
    }
    
  }

  /**
   * Remove an owner from this ACL
   *
   * @param caller TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @return true iff successful
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   * @exception LastOwnerException Exception raised in failure situation
   */
  public boolean removeOwner(IUMPrincipal caller, IUMPrincipal owner)
    throws AclPersistenceException, NotAuthorizedException, LastOwnerException {

    if( basicRemoveOwner(caller, owner) ) {
      m_owners = m_databaseConnection.getOwners(m_aclID, false); // COOKED
      return true;
    } else {
      return false;
    }
    
  }

  /**
   * @param principal TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @return true iff the principal has the specified permission on this ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean checkPermission(IUMPrincipal principal, IAclPermission permission)
    throws AclPersistenceException {

    if (principal == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("checkPermission(471)", "principal " + principal.getId() + ", permission " + permission.getName());
    }

    if (isAuthorized(principal)) {
      return true;
    }

    IUser user = (IUser)principal;
    if (!user.isAuthenticated()) {
      return false;
    }

    return basicCheckPermission(user, permission);
  }

  /**
   * @param principal TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @return true iff the principal has the specified permissions on this ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean checkPermissions(IUMPrincipal principal, IAclPermission[] permissions)
    throws AclPersistenceException {

    if (principal == null || permissions == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("checkPermissions(500)", "principal " + principal.getId() + ", permissions-length " + permissions.length);
    }

    if (isAuthorized(principal)) {
      return true;
    }

    IUser user = (IUser)principal;
    if (!user.isAuthenticated()) {
      return false;
    }

    for (int i = 0; i < permissions.length; i++) {
      if (!basicCheckPermission(user, permissions[i])) {
        return false;
      }
    }

    return true;
  }

  /**
   * Lock this ACL
   *
   * @param caller TBD: Description of the incoming method parameter
   * @return true iff successful
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public final boolean lock(IUMPrincipal caller)
    throws AclPersistenceException, NotAuthorizedException {

    if (caller == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("lock(537)", "caller " + caller.getId());
    }

    if (!isAuthorized(caller)) {
      throw new NotAuthorizedException();
    }

    return m_databaseConnection.lockAcl(m_aclID, caller);
  }

  /**
   * Unlock this ACL
   *
   * @param caller TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public final void unlock(IUMPrincipal caller)
    throws AclPersistenceException, NotAuthorizedException {

    if (caller == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("unlock(562)", "caller " + caller.getId());
    }

    IUMPrincipal lockingUser = m_databaseConnection.getAclLockingUser(m_aclID);
    if (lockingUser == null || !caller.equals(lockingUser)) {
      throw new NotAuthorizedException();
    }

    m_databaseConnection.unlockAcl(m_aclID);
  }

  /**
   * @param postFix TBD: Description of the incoming method parameter
   * @return true iff ACLs exist with IDs that start with the ID of this ACL
   *      followed by the specified postfix
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean hasDescendantsWithAcl(String postFix)
    throws AclPersistenceException {

    if (postFix == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("hasDescendantsWithAcl(587)", "postFix " + postFix);
    }

    return m_databaseConnection.hasDescendantsWithAcl(m_aclID + postFix);
  }

  /**
   * @return true iff the persisted ACL has the same change level
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean isUpToDate()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("isUpToDate(601)", "isUpToDate");
    }
    if ( System.currentTimeMillis() < this.deadline ) {
      return true ;
    }
    if ( m_databaseConnection.isAclUpToDate(m_aclID, m_changeLevel)) {
      deadline = System.currentTimeMillis() + 1000 ;
      return true ;
    }
    return false ;
  }

  /**
   * Set a new ACL ID
   *
   * @param id iD to be set
   * @exception AclPersistenceException Exception raised in failure situation
   */
  protected final void setID(String id)
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("setID(617)", "id " + id);
    }

    m_aclID = id;
    IAclEntryListIterator entryIterator = m_entries.iterator();
    while (entryIterator.hasNext()) {
      JDBCAclEntry entry = (JDBCAclEntry)entryIterator.next();
      entry.setIDs(entry.getID(), id);
      entry.setAcl(this);
    }
  }

  /**
   * @return the ACL change level
   * @exception AclPersistenceException Exception raised in failure situation
   */
  protected final long getChangeLevel()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("getChangeLevel(637)", "getChangeLevel");
    }

    return m_changeLevel;
  }

  /**
   * Reload the ACEs
   *
   * @exception AclPersistenceException Exception raised in failure situation
   */
  protected void reloadAclEntries()
    throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("reloadAclEntries(652)", "getChangeLevel");
    }

    m_entries = m_databaseConnection.getAclEntries(m_aclID, false); // COOKED
    setEntriesAcl();
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @return true iff the specified permission is supported for the object tyxpe
   *      of this ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  protected final boolean isPermissionSupported(IAclPermission permission)
    throws AclPersistenceException {

    IAclPermissionList permissions = m_databaseConnection.getSupportedPermissions(m_objectType);
    IAclPermissionListIterator iterator = permissions.iterator();
    while (iterator.hasNext()) {
      IAclPermission listPermission = iterator.next();
      if (listPermission.equals(permission)) {
        return true;
      }
    }

    return false;
  }

  /**
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @return true iff the principal has the specified permission on this ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  private final boolean basicCheckPermission(IUser user, IAclPermission permission)
    throws AclPersistenceException {

    IAclEntryListIterator entryIterator = m_entries.iterator();
    while (entryIterator.hasNext()) {
      IAclEntry listEntry = entryIterator.next();
      IUMPrincipal listPrincipal = listEntry.getPrincipal();
      if (containsUser(listPrincipal, user)) {
        if (listEntry.checkPermission(permission)) {
          if (listEntry.isNegative()) {
            return false;
          }
          return true;
        }
      }
    }
    return false;
  }

  /**
   * @param principal TBD: Description of the incoming method parameter
   * @param user TBD: Description of the incoming method parameter
   * @return true iff the principal equals or contains the specified user
   */
  private boolean containsUser(IUMPrincipal principal, IUser user) {
    if (principal == null || user == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (principal.equals(user)) {
      return true;
    }
    if ((principal.getType() == IUMPrincipal.IGROUP
    && ((IGroup)principal).containsUser(user.getId()))
    || (principal.getType() == IUMPrincipal.IROLE
    && ((IRole)principal).containsUser(user.getId()))) {
      return true;
    }
    return false;
  }

  /**
   * Set ACL at entries
   */
  private void setEntriesAcl() {

    IAclEntryListIterator entryIterator = m_entries.iterator();
    while (entryIterator.hasNext()) {
      JDBCAclEntry entry = (JDBCAclEntry)entryIterator.next();
      entry.setAcl(this);
    }
  }
  

  // --------------------------------------------------------------------------
  protected final boolean basicAddEntry(IUMPrincipal caller,
                                        IAclEntry aclEntry)
                                 throws AclPersistenceException,
                                        InvalidClassException, 
                                        NotAuthorizedException, 
                                        AlreadyAssignedToAclException, 
                                        PermissionNotSupportedException {

    if (caller == null || aclEntry == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (!(aclEntry instanceof JDBCAclEntry)) {
      throw new InvalidClassException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("basicAddEntry", "caller " + caller.getId() + ", aclEntryID " + ((JDBCAclEntry)aclEntry).getID());
    }

    if (!isAuthorized(caller)) {
      throw new NotAuthorizedException();
    }

    if (!isPermissionSupported(aclEntry.getPermission())) {
      throw new PermissionNotSupportedException();
    }

    if (m_entries != null) {
      IAclEntryListIterator it = m_entries.iterator();
      while (it.hasNext()) {
        IAclEntry entry = it.next();
        if (entry.equals(aclEntry)) {
          return false;
        }
      }
    }

    long id = m_databaseConnection.addAclEntry(m_aclID, (JDBCAclEntry)aclEntry);
    if (id <= 0) {
      return false;
    }

    ((JDBCAclEntry)aclEntry).setIDs(id, m_aclID);
    ((JDBCAclEntry)aclEntry).setAcl(this);
    return true;

  }

  // --------------------------------------------------------------------------
  protected final boolean basicRemoveEntry(IUMPrincipal caller,
                                           IAclEntry aclEntry)
                                    throws AclPersistenceException, 
                                           InvalidClassException, 
                                           NotAuthorizedException {

    if (caller == null || aclEntry == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (!(aclEntry instanceof JDBCAclEntry)) {
      throw new InvalidClassException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("basicRemoveEntry", "caller " + caller.getId() + ", aclEntryID " + ((JDBCAclEntry)aclEntry).getID());
    }

    if (!isAuthorized(caller)) {
      throw new NotAuthorizedException();
    }

    if (!m_databaseConnection.removeAclEntry((JDBCAclEntry)aclEntry)) {
      return false;
    }

    ((JDBCAclEntry)aclEntry).clearIDs();
    return true;
    
  }

  // --------------------------------------------------------------------------
  protected final boolean basicAddOwner(IUMPrincipal caller,
                                        IUMPrincipal principal)
                                 throws AclPersistenceException,
                                        NotAuthorizedException {

    if (caller == null || principal == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("basicAddOwner", "caller " + caller.getId() + ", principal " + principal.getId());
    }

    if (!isAuthorized(caller)) {
      throw new NotAuthorizedException();
    }

    if (!m_databaseConnection.addOwner(m_aclID, principal)) {
      return false;
    }

    return true;
    
  }


  // --------------------------------------------------------------------------
  protected final boolean basicRemoveOwner(IUMPrincipal caller,
                                           IUMPrincipal owner)
                                    throws AclPersistenceException, 
                                           NotAuthorizedException, 
                                           LastOwnerException {

    if (caller == null || owner == null) {
      throw new java.lang.IllegalArgumentException();
    }

    if (s_log.beDebug()) {
      s_log.debugT("basicRemoveOwner", "caller " + caller.getId() + ", owner " + owner.getId());
    }

    if (!isAuthorized(caller)) {
      throw new NotAuthorizedException();
    }

    if (!m_databaseConnection.removeOwner(m_aclID, owner)) {
      return false;
    }

    return true;
    
  }

}
