/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc ;
import com.sap.tc.logging.Location ;

import com.sapportals.portal.security.usermanagement.* ;
import com.sapportals.wcm.util.acl.* ;
import com.sapportals.wcm.util.cache.CacheException ;
import com.sapportals.wcm.util.cache.ICache ;
import com.sapportals.wcm.util.cache.ICacheEntry ;
import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionPool ;
import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionPoolManager ;
import com.sapportals.wcm.util.logging.LoggingFormatter ;
import com.sapportals.wcm.util.usermanagement.WPUMFactory ;

import java.sql.* ;
import java.util.* ;

/**
 * Uncached access to the ACL database persistence.
 * <p>
 * Copyright SAP AG 2001-2004
 * @author paul.goetz@sap.com
 */
public abstract class AbstractDatabaseConnectionUncached {

  private final static com.sap.tc.logging.Location LOG = com.sap.tc.logging.Location.getLocation(AbstractDatabaseConnectionUncached.class);

  protected final static int PRINCIPAL_TYPE_TAG_USER = 100 ;
  protected final static int PRINCIPAL_TYPE_TAG_GROUP = 101 ;
  protected final static int PRINCIPAL_TYPE_TAG_ROLE = 102 ;

  /*
  * for the permissions cache
  */
  private final static String SEPARATOR = "-" ;
  private final static String KEY_PERMISSION_PREFIX = "p" ;
  private final static String KEY_SUPPORTED_PERMISSIONS_PREFIX = "s" ;

  protected static Location s_log ;
  protected static String m_poolID ;
  protected String m_applicationID ;
  protected String m_managerID ;
  protected long ownerId ;
  protected ICache m_permissionCache ;

  /*
  * it's better to cache permissions on this level, because they
  * are heavily used to create ACEs
  */
  protected Connection m_main_connection = null ;
  protected Object m_main_connection_lock = new Object() ;
  protected JDBCConnectionPool conPool = null ;

  abstract public void terminate() ;

  /**
   * TBD: Description of the class.
   */
  class AclRec {
    private String m_externalAclID ;
    private long aclRidID ;
    private String m_objectTypeName ;
    private long m_changeLevel ;

    public AclRec ( String externalAclID, String objectTypeName,
    long changeLevel ) {
      m_externalAclID = externalAclID ;
      m_objectTypeName = objectTypeName ;
      m_changeLevel = changeLevel ;
      aclRidID = -1 ;
    }

    public AclRec ( String externalAclID, long aclRidID, String objectTypeName,
    long changeLevel ) {
      m_externalAclID = externalAclID ;
      m_objectTypeName = objectTypeName ;
      m_changeLevel = changeLevel ;
      this.aclRidID = aclRidID ;
    }

    public String getExternalAclID() {
      return m_externalAclID ;
    }

    public long getAclRidID() {
      return aclRidID ;
    }

    public String getObjectTypeName() {
      return m_objectTypeName ;
    }

    public long getChangeLevel() {
      return m_changeLevel ;
    }
  }


  private void commit(Connection c) throws Exception {
    //if (!TxManager.isTxActive()) c.commit();
    try{
      c.commit();
    }
    catch(Exception ignore) {
      //
    }
    
  }
  
  private void rollback(Connection c) throws SQLException {
/*    try{
      if (TxManager.isTxActive())
        TxManager.setRollbackOnly();
      else
        c.rollback();
    }
    catch(TxException e) {
      c.rollback();
    }
    */
    c.rollback();
  }
  
  /**
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return the ACL with the specified ID (or null if the ACL doesn't exist)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final JDBCAcl getAcl ( String externalAclID, boolean raw )
  throws AclPersistenceException {

    if (externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAcl(303)", "externalAclID " + externalAclID) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;

      synchronized (m_main_connection_lock) {
        try {
          JDBCAcl result = null ;

          AclRec acl = uncommittedGetAcl( c, externalAclID) ;
          if (acl != null) {
            UMPrincipalList owners = uncommittedGetOwners ( c, externalAclID, raw );
            AclEntryList entries = uncommittedGetAclEntries ( c, externalAclID, raw );
            if(   ( raw )
               && ( owners.containsRawEntries() || entries.containsRawEntries() )
              ) { 
              result = new JDBCAclRaw(this, externalAclID,
                new ObjectType(acl.getObjectTypeName()), owners, entries,
                acl.getChangeLevel()) ;
            } else {
              result = new JDBCAcl(this, externalAclID,
                new ObjectType(acl.getObjectTypeName()), owners, entries,
                acl.getChangeLevel()) ;
            }
          }
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
             if (LOG.beDebug()) {
               LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
             }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param externalAclIDs TBD: Description of the incoming method parameter
   * @return the ACLs for the specified IDs (array elements are nuul for ACLs
   *      that doesn't exist)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final JDBCAcl[] getAcls(String[] externalAclIDs, boolean raw)
    throws AclPersistenceException {

    if (externalAclIDs == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT ( "getAcls(356)", "externalAclIDs-length "
        + externalAclIDs.length) ;
    }

    if (externalAclIDs.length == 0) {
      return new JDBCAcl[0] ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          JDBCAcl[] result = new JDBCAcl[externalAclIDs.length] ;
          AclRec[] acls = uncommittedGetAcls ( c, externalAclIDs ) ;
          UMPrincipalList[] owners = uncommittedGetOwners(c, acls, raw) ;
          AclEntryList[] entries = uncommittedGetAclEntries(c, acls, raw) ;
          for (int i = 0 ; i < externalAclIDs.length ; i++) {
            if (acls[i] != null) {
              if(   ( raw )
                 && ( owners[i].containsRawEntries() || entries[i].containsRawEntries() )
                ) {
                result[i] = new JDBCAclRaw(this, externalAclIDs[i],
                  new ObjectType(acls[i].getObjectTypeName()), owners[i],
                  entries[i], acls[i].getChangeLevel()) ;
              } else {
                result[i] = new JDBCAcl(this, externalAclIDs[i],
                  new ObjectType(acls[i].getObjectTypeName()), owners[i],
                  entries[i], acls[i].getChangeLevel()) ;
              }
            }
            else {
              result[i] = null ;
            }
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }


  /**
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return the ACEs of the specified ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final AclEntryList getAclEntries ( String externalAclID, boolean raw )
  throws AclPersistenceException {

    if (externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAclEntries(420)", "externalAclID " + externalAclID) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          AclEntryList result = uncommittedGetAclEntries(c, externalAclID, raw) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Set the sort index of an ACE
   *
   * @param aclEntryID aclEntrySortIndex to be set
   * @param sortIndex aclEntrySortIndex to be set
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean setAclEntrySortIndex ( long aclEntryID, int sortIndex )
  throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("setAclEntrySortIndex(463)", "aclEntryID " + aclEntryID
        + ", sortIndex " + sortIndex) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = uncommittedSetAclEntrySortIndex(c,
            aclEntryID, sortIndex) ;
          if (result) {
            uncommittedIncreaseAclChangeLevel(c,
              uncommittedGetInternalAclIDFromEntryID(c,
              aclEntryID)) ;
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Set the propagate-flag of an ACE
   *
   * @param aclEntryID aclEntryPropagation to be set
   * @param propagate aclEntryPropagation to be set
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean setAclEntryPropagation ( long aclEntryID,
  boolean propagate ) throws AclPersistenceException {

    if (s_log.beDebug()) {
      s_log.debugT("setAclEntryPropagation(510)", "aclEntryID " + aclEntryID
        + ", propagate " + propagate) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = uncommittedSetAclEntryPropagation(c,
            aclEntryID, propagate) ;
          if (result) {
            uncommittedIncreaseAclChangeLevel(c,
              uncommittedGetInternalAclIDFromEntryID(c,
              aclEntryID)) ;
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @return the object types that are supported by the ACL manager
   */
  public final IObjectTypeList getSupportedObjectTypes() {

    if (s_log.beDebug()) {
      s_log.debugT("getSupportedObjectTypes(551)", "getSupportedObjectTypes") ;
    }

    ObjectTypeList objectTypes = new ObjectTypeList() ;

    objectTypes.add(new ObjectType(IObjectType.OBJECT_TYPE_NODE)) ;
    objectTypes.add(new ObjectType(IObjectType.OBJECT_TYPE_LEAF)) ;

    return objectTypes ;
  }

  /**
   * @param objectType TBD: Description of the incoming method parameter
   * @return the permissions which are supported for the specified object type
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final IAclPermissionList getSupportedPermissions (
  IObjectType objectType ) throws AclPersistenceException {

    if (objectType == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    IAclPermissionList supportedPermissions =
      cacheGetSupportedPermissions(objectType.getName()) ;
    if (supportedPermissions != null) {
      return supportedPermissions ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getSupportedPermissions(580)", "objectType "
        + objectType.getName()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          IAclPermissionList permissions = uncommittedGetSupportedPermissions (
            c, objectType.getName()) ;
          commit(c) ;
          return permissions ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return the owners of the specified ACL
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final UMPrincipalList getOwners ( String externalAclID, boolean raw )
  throws AclPersistenceException {

    if (externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getOwners(625)", "externalAclID " + externalAclID) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          UMPrincipalList result = uncommittedGetOwners (c, externalAclID, raw) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param permissionName TBD: Description of the incoming method parameter
   * @return the members of the permission
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final IAclPermissionList getPermissionMembers ( String permissionName )
  throws AclPersistenceException {

    if (permissionName == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getPermissionMembers(669)", "permissionName "
        + permissionName) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          IAclPermissionList members = uncommittedGetPermissionMembers (
            c, permissionName) ;
          commit(c) ;
          return members ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @return true iff the specified permission is used in at least one ACE
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean isPermissionUsedInAcl ( IAclPermission permission )
  throws AclPersistenceException {

    if (permission == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("isPermissionUsedInAcl(714)", "permission "
        + permission.getName()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = uncommittedIsUsedPermission (
            c, permission.getName()) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @return a list of all IDs of ACLs (strings) which use the specified
   *      permission in one of their ACEs
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final List getAffectedAclIDs ( IAclPermission permission )
  throws AclPersistenceException {

    if (permission == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAffectedAclIDs(760)", "permission "
        + permission.getName()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          List result = uncommittedGetAffectedAclIDs (
            c, permission.getName()) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param permissionName TBD: Description of the incoming method parameter
   * @return the IAclPermission object for the specifoed permission name
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final IAclPermission getPermission ( String permissionName )
  throws AclPersistenceException {
    if (permissionName == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getPermission(804)", "permissionName " + permissionName) ;
    }
    IAclPermission permission = cacheGetPermission(permissionName) ;
    if (permission != null) {
      return permission ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          IAclPermission result = uncommittedGetPermission (
            c, permissionName) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return true iff the specified ACL is locked
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean isAclLocked ( String externalAclID )
  throws AclPersistenceException {

    if (externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("isAclLocked(853)", "externalAclID " + externalAclID) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = uncommittedIsAclLocked (
            c, externalAclID) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return the user which locks the specified ACL (or null if the ACL is
   *      currently not locked)
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final IUser getAclLockingUser ( String externalAclID )
  throws AclPersistenceException {

    if (externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getAclLockingUser(899)", "externalAclID " + externalAclID) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          IUser result = uncommittedGetAclLockingUser (
            c, externalAclID) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param externalAclIDRoot TBD: Description of the incoming method parameter
   * @return a list of all IDs of ACLs (strings) which start with the specified
   *      ID
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public List getDescendantsWithAcl ( String externalAclIDRoot )
  throws AclPersistenceException {
    if (externalAclIDRoot == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("getDescendantsWithAcl(943)", "externalAclIDRoot "
        + externalAclIDRoot) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          List result = uncommittedGetDescendantsWithAcl (
            c, externalAclIDRoot) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  abstract public JDBCAcl createAcl ( IUMPrincipal owner, String externalAclID,
    IObjectType objectType) throws AclPersistenceException, AclExistsException ;

  /**
   * Add a new ACE to the specified ACL (create in database)
   *
   * @param externalAclID aclEntry to be added
   * @param entry aclEntry to be added
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception AlreadyAssignedToAclException Exception raised in failure
   *      situation
   */
  public final long addAclEntry ( String externalAclID, JDBCAclEntry entry )
  throws AclPersistenceException, AlreadyAssignedToAclException {

    if (externalAclID == null || entry == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("addAclEntry(1072)", "externalAclID " + externalAclID
        + ", entryID " + entry.getID()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          long result = -1 ;

          result = uncommittedAddAclEntry (
            c, externalAclID, entry) ;
          if (result > 0) {
            uncommittedIncreaseAclChangeLevel(c, externalAclID) ;
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Remove an ACL
   *
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean removeAcl ( String externalAclID )
  throws AclPersistenceException {

    if (externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("removeAcl(1124)", "externalAclID " + externalAclID) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = false ;

          if (uncommittedGetAcl(c, externalAclID) != null) {
            uncommittedRemoveAclEntries(c, externalAclID) ;
            uncommittedRemoveOwners(c, externalAclID) ;
            uncommittedRemoveAcl(c, externalAclID) ;
            result = true ;
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Remove an ACE from the database
   *
   * @param entry TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean removeAclEntry ( JDBCAclEntry entry )
  throws AclPersistenceException {

    if (entry == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("removeAclEntry(1178)", "entryID " + entry.getID()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = false ;

          String internalAclID = uncommittedGetInternalAclIDFromEntryID (
            c, entry.getID()) ;
          if (internalAclID != null) {
            result = uncommittedRemoveAclEntry(c, entry.getID()) ;
            if (result) {
              uncommittedIncreaseAclChangeLevel ( c, internalAclID) ;
            }
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Add a supported permission for an object type
   *
   * @param objectType supportedPermission to be added
   * @param permission supportedPermission to be added
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   */
  public final boolean addSupportedPermission ( IObjectType objectType,
  IAclPermission permission ) throws AclPersistenceException,
  PredefinedPermissionException {

    if (objectType == null || permission == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("addSupportedPermission(1236)", "objectType "
        + objectType.getName() + ", permission " + permission.getName()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          if (uncommittedIsPredefinedPermission ( c,
          permission.getName())) {
            throw new PredefinedPermissionException() ;
          }

          boolean result = false ;
          if (!uncommittedIsSupportedPermission(c,
          permission.getName(), objectType.getName())) {
            result = uncommittedAddSupportedPermission(c,
              permission.getName(), objectType.getName()) ;
          }
          commit(c) ;
          return result ;
        }
        catch (PredefinedPermissionException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Remove a supported permission for an object type
   *
   * @param objectType TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public final boolean removeSupportedPermission ( IObjectType objectType,
  IAclPermission permission ) throws AclPersistenceException,
  PredefinedPermissionException, PermissionUsedException {

    if (objectType == null || permission == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("removeSupportedPermission(1305)", "objectType "
        + objectType.getName() + ", permission " + permission.getName()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          String permissionName = permission.getName() ;

          if (uncommittedIsPredefinedPermission(c, permissionName)) {
            throw new PredefinedPermissionException() ;
          }
          if (uncommittedIsUsedPermission(c, permissionName)) {
            throw new PermissionUsedException() ;
          }

          boolean result = false ;
          if (uncommittedIsSupportedPermission(c, permissionName,
          objectType.getName())) {
            result = uncommittedRemoveSupportedPermission(c,
              permissionName, objectType.getName()) ;
            if (result) {
              cacheRemovePermission(permissionName) ;
            }
          }

          commit(c) ;
          return result ;
        }
        catch (PredefinedPermissionException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (PermissionUsedException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @param name TBD: Description of the incoming method parameter
   * @return a new permission (create in database)
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PermissionExistsException Exception raised in failure situation
   */
  public final IAclPermission createPermission ( String name )
  throws AclPersistenceException, PermissionExistsException {
    if (name == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("createPermission(1387)", "name " + name) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          if (uncommittedCheckPermission(c, name)) {
            throw new PermissionExistsException() ;
          }

          /*
          * in case the database is inconsistent, entries may exist
          * for a removed permission - remove them
          */
          uncommittedRemovePermissionMembers(c, name) ;
          uncommittedRemoveSupportedPermission(c, name,
            IObjectType.OBJECT_TYPE_LEAF) ;
          uncommittedRemoveSupportedPermission(c, name,
            IObjectType.OBJECT_TYPE_NODE) ;
          uncommittedAddPermission(c, name, false) ;
          commit(c) ;
          return new JDBCPermission(this, name, false, new AclPermissionList()) ;
        }
        catch (PermissionExistsException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Remove a permission (from database)
   *
   * @param permission TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public final boolean removePermission ( IAclPermission permission )
  throws AclPersistenceException, PredefinedPermissionException,
  PermissionUsedException {
    if (permission == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("removePermission(1457)", "permission "
        + permission.getName()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          String permissisonName = permission.getName() ;

          if (uncommittedIsPredefinedPermission(c, permissisonName)) {
            throw new PredefinedPermissionException() ;
          }
          if (uncommittedIsUsedPermission(c, permissisonName)) {
            throw new PermissionUsedException() ;
          }

          uncommittedRemovePermissionMembers(c, permissisonName) ;
          uncommittedRemoveSupportedPermission(c,
            permissisonName, IObjectType.OBJECT_TYPE_LEAF) ;
          uncommittedRemoveSupportedPermission(c,
            permissisonName, IObjectType.OBJECT_TYPE_NODE) ;

          boolean result = uncommittedRemovePermission(c,
            permissisonName) ;
          if (result) {
            cacheRemovePermission(permissisonName) ;
          }

          commit(c) ;
          return result ;
        }
        catch (PredefinedPermissionException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (PermissionUsedException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Add a new member to a permission
   *
   * @param permissionName permissionMember to be added
   * @param permission permissionMember to be added
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PermissionExistsException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public final boolean addPermissionMember ( String permissionName,
  IAclPermission permission ) throws AclPersistenceException,
  PermissionExistsException, PredefinedPermissionException,
  PermissionUsedException {

    if (permissionName == null || permission == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("addPermissionMember(1547)", "permissionName "
        + permissionName + ", permission " + permission.getName()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          IAclPermission parent = uncommittedGetPermission(c,
            permissionName) ;
          if (isPermissionMember(parent, permission)) {
            throw new PermissionExistsException() ;
          }
          if (isPermissionMember(permission, parent)) {
            throw new PermissionExistsException() ;
          }
          if (uncommittedIsPredefinedPermission(c,
          permissionName)) {
            throw new PredefinedPermissionException() ;
          }
          if (uncommittedIsUsedPermission(c, permissionName)) {
            throw new PermissionUsedException() ;
          }

          boolean result = uncommittedAddPermissionMember(c,
            permissionName, permission.getName()) ;
          if (result) {
            cacheRemovePermission(permissionName) ;
          }

          commit(c) ;
          return result ;
        }
        catch (PermissionExistsException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (PredefinedPermissionException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (PermissionUsedException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Remove a member from a permission
   *
   * @param permissionName TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public final boolean removePermissionMember(String permissionName,
  IAclPermission permission) throws AclPersistenceException,
  PredefinedPermissionException, PermissionUsedException {

    if (permissionName == null || permission == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("removePermissionMember(1648)", "permissionName "
        + permissionName + ", permission " + permission.getName()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          if (uncommittedIsPredefinedPermission(c, permissionName)) {
            throw new PredefinedPermissionException() ;
          }
          if (uncommittedIsUsedPermission(c, permissionName)) {
            throw new PermissionUsedException() ;
          }

          boolean result = uncommittedRemovePermissionMember(c, permissionName,
            permission.getName()) ;
          if (result) {
            cacheRemovePermission(permissionName) ;
          }

          commit(c) ;
          return result ;
        }
        catch (PredefinedPermissionException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (PermissionUsedException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Add an owner to an ACL
   *
   * @param externalAclID owner to be added
   * @param owner owner to be added
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean addOwner(String externalAclID, IUMPrincipal owner)
  throws AclPersistenceException {

    if (externalAclID == null || owner == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("addOwner(1728)", "externalAclID " + externalAclID
        + ", owner " + owner.getId()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = false ;

          if (!uncommittedIsOwner(c, externalAclID, owner)) {
            result = uncommittedAddOwner ( c, externalAclID, owner ) ;
            if (result) {
              uncommittedIncreaseAclChangeLevel ( c, externalAclID ) ;
            }
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Remove an owner from an ACL
   *
   * @param externalAclID TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception LastOwnerException Exception raised in failure situation
   */
  public final boolean removeOwner ( String externalAclID, IUMPrincipal owner )
  throws AclPersistenceException, LastOwnerException {

    if (externalAclID == null || owner == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("removeOwner(1784)", "externalAclID " + externalAclID
        + ", owner " + owner.getId()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = false ;

          if (uncommittedIsLastOwner(c, externalAclID, owner)) {
            throw new LastOwnerException() ;
          }
          result = uncommittedRemoveOwner ( c, externalAclID,
            owner ) ;
          if (result) {
            uncommittedIncreaseAclChangeLevel(c, externalAclID) ;
          }

          commit(c) ;
          return result ;
        }
        catch (LastOwnerException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Delete a principal from the tables (is usually called as reaction to an
   * event from the usermanagement)
   *
   * @param principalID TBD: Description of the incoming method parameter
   * @param principalType TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final void deletePrincipal ( String principalID, int principalType )
  throws AclPersistenceException {

    if (principalID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("deletePrincipal(1851)", "principalID " + principalID
        + ", principalType " + getPrincipalTypeText(principalType)) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          // remove the principal as owner from all ACLs
          uncommittedRemoveOwner(c, principalID, principalType) ;

          // remove all ACEs concerning the principal
          uncommittedRemoveAclEntries(c, principalID,
            principalType) ;

          commit(c) ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Lock an ACL
   *
   * @param externalAclID TBD: Description of the incoming method parameter
   * @param lockingUser TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean lockAcl ( String externalAclID,
  IUMPrincipal lockingUser ) throws AclPersistenceException {

    if (lockingUser == null || externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("lockAcl(1902)", "externalAclID " + externalAclID
        + ", lockingUser " + lockingUser.getId()) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = false ;

          if (!uncommittedIsAclLocked(c, externalAclID)) {
            result = uncommittedLockAcl ( c, externalAclID, lockingUser ) ;
            if (result) {
              uncommittedIncreaseAclChangeLevel ( c, externalAclID ) ;
            }
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Unlock an ACL
   *
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean unlockAcl ( String externalAclID )
  throws AclPersistenceException {

    if (externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("unlockAcl(1956)", "externalAclID " + externalAclID) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = false ;

          result = uncommittedUnlockAcl(c, externalAclID) ;
          if (result) {
            uncommittedIncreaseAclChangeLevel(c, externalAclID) ;
          }

          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Change the ID of an ACL (rename)
   *
   * @param oldExternalAclID TBD: Description of the incoming method parameter
   * @param newExternalAclID TBD: Description of the incoming method parameter
   * @return true on success
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception AclExistsException Exception raised in failure situation
   */
  public final boolean changeAclID ( String oldExternalAclID,
  String newExternalAclID ) throws AclPersistenceException, AclExistsException {

    if (oldExternalAclID == null || newExternalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("changeAclID(2010)", "oldExternalAclID " + oldExternalAclID
        + ", newExternalAclID " + newExternalAclID) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          if (oldExternalAclID.equals(newExternalAclID)) {
            throw new AclExistsException() ;
          }

          boolean result = false ;

          if (uncommittedGetAcl(c, newExternalAclID) != null) {
            throw new AclExistsException() ;
          }

          if (uncommittedCheckAcl(c, newExternalAclID)) {
            throw new AclExistsException() ;
          }
          if (uncommittedCheckAclEntries(c, newExternalAclID)) {
            throw new AclExistsException() ;
          }
          if (uncommittedCheckOwners(c, newExternalAclID)) {
            throw new AclExistsException() ;
          }

          if (!uncommittedChangeAclID_in_AclTable(c,
          oldExternalAclID, newExternalAclID)) {
            throw new Exception("can't change acl id in acl table") ;
          }
          if (uncommittedCheckAclEntries(c, oldExternalAclID)) {
            if (!uncommittedChangeAclID_in_AclEntryTable(c,
            oldExternalAclID, newExternalAclID)) {
              throw new Exception("can't change acl id in acl entry table") ;
            }
          }
          if (!uncommittedChangeAclID_in_OwnerTable(c,
          oldExternalAclID, newExternalAclID)) {
            throw new Exception("can't change acl id in owner table") ;
          }

          result = true ;

          commit(c) ;
          return result ;
        }
        catch (AclExistsException e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw e ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * Change all IDs of ACLs which start with oldAclIdPrefix by replacing it with
   * newAclIdPrefix
   *
   * @param oldAclIdPrefix TBD: Description of the incoming method parameter
   * @param newAclIdPrefix TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception AclExistsException Exception raised in failure situation
   */
  public final void changeMultipleAclIDs ( String oldAclIdPrefix,
  String newAclIdPrefix ) throws AclPersistenceException, AclExistsException {

    if (oldAclIdPrefix == null || newAclIdPrefix == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("changeMultipleAclIDs(2102)", "oldAclIdPrefix "
        + oldAclIdPrefix + ", newAclIdPrefix " + newAclIdPrefix) ;
    }

    synchronized (m_main_connection_lock) {
      List descendants = getDescendantsWithAcl(oldAclIdPrefix) ;
      if (descendants == null) {
        return ;
      }

      Iterator iterator = descendants.iterator() ;
      while (iterator.hasNext()) {
        String id = (String)iterator.next() ;
        if (id != null) {
          changeAclID ( id, newAclIdPrefix + id.substring (
            oldAclIdPrefix.length())) ;
        }
      }
    }
  }

  /**
   * Remove all ACLs with IDs starting with aclIdPrefix (remove children)
   *
   * @param aclIdPrefix TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final void removeMultipleAcls ( String aclIdPrefix )
  throws AclPersistenceException {

    if (aclIdPrefix == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("removeMultipleAcls(2135)", "aclIdPrefix " + aclIdPrefix) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          uncommittedRemoveDescendantsWithAcl(c, aclIdPrefix) ;

          commit(c) ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  public final boolean hasDescendantsWithAcl ( String externalAclIDRoot )
  throws AclPersistenceException {
    Connection c = null ;
    try {
      c = getDBConnection() ;
      boolean rv = hasDescendantsWithAcl(c, externalAclIDRoot) ;
      return rv ;
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    catch (Exception e) {
      throw new AclPersistenceException(e) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  protected final static Integer[] makeHashSegments(String path) {
    Integer[] ret = new Integer[10] ;

    if (path.startsWith("/")) {
      path = path.substring(1) ;
    }
    for (int nrOfSeg = 0 ; nrOfSeg < 10 ; nrOfSeg++) {
      if (path == null || path.equals("")) {
        ret[nrOfSeg] = null ;
      }
      else {
        int j = path.indexOf('/') ;
        if (j < 0) {
          ret[nrOfSeg] = new Integer(path.hashCode()) ;
          break ;
        }
        ret[nrOfSeg] = new Integer(path.substring(0, j).hashCode()) ;
        path = path.substring(j + 1) ;
      }
    }
    return ret ;
  }

  abstract protected boolean hasDescendantsWithAcl ( Connection connection,
    String externalAclIDRoot ) throws SQLException ;

  abstract protected void uncommittedRemoveDescendantsWithAcl (
    Connection connection, String externalAclIDRoot ) throws SQLException ;

  abstract protected List uncommittedGetDescendantsWithAcl (
    Connection connection, String externalAclIDRoot ) throws SQLException ;

  public final boolean isAclUpToDate ( JDBCAcl acl )
  throws AclPersistenceException {
    if ( System.currentTimeMillis() < acl.getDeadline()) {
      return true ;
    }
    if ( this.isAclUpToDate(acl.getID(),acl.getChangeLevel())) {
      acl.extendDeadline() ;
      return true ;
    }
    return false ;
  }
  /**
   * @param externalAclID TBD: Description of the incoming method parameter
   * @param changeLevel TBD: Description of the incoming method parameter
   * @return true iff the change level of the ACL in the database is not larger
   *      than the specified one
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean isAclUpToDate ( String externalAclID, long changeLevel )
  throws AclPersistenceException {

    if (externalAclID == null) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("isAclUpToDate(2428)", "externalAclID " + externalAclID
        + ", changeLevel " + changeLevel) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean result = uncommittedIsAclUpToDate ( c, externalAclID,
            changeLevel) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  public final boolean[] areAclsUpToDate ( JDBCAcl[] acls )
  throws AclPersistenceException {
    if ( acls == null || acls.length == 0 ) {
      return new boolean[0] ;
    }
    long t0 = System.currentTimeMillis() ;
    int i ;
    boolean[] rv = null ;
    for ( i = 0 ; i < acls.length && t0 < acls[i].getDeadline() ; i++ ) ;
    if ( i == acls.length ) {
      /*
      * all valid!
      */
      rv = new boolean[acls.length] ;
      for ( i = 0 ; i < acls.length ; i++ ) {
        rv[i] = true ;
      }
      return rv ;
    }
    String[] aclIDs = new String[acls.length];
    long[] changeLevels = new long[acls.length];
    for ( i = 0 ; i < acls.length ; i++ ) {
      aclIDs[i] = ((JDBCAcl)acls[i]).getID();
      changeLevels[i] = ((JDBCAcl)acls[i]).getChangeLevel();
    }
    rv = this.areAclsUpToDate(aclIDs,changeLevels) ;
    for ( i = 0 ; i < rv.length ; i++ ) {
      if ( rv[i] ) {
        acls[i].extendDeadline() ;
      }
    }
    return rv ;
  }

  /**
   * @param externalAclIDs TBD: Description of the incoming method parameter
   * @param changeLevels TBD: Description of the incoming method parameter
   * @return up to date flags for ACLs
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public final boolean[] areAclsUpToDate ( String[] externalAclIDs,
  long[] changeLevels ) throws AclPersistenceException {

    if (( externalAclIDs == null ) || ( changeLevels == null )
    || ( externalAclIDs.length != changeLevels.length )) {
      throw new java.lang.IllegalArgumentException() ;
    }

    if (s_log.beDebug()) {
      s_log.debugT("areAclsUpToDate(2473)", "externalAclIDs-length "
        + externalAclIDs.length + ", changeLevels-length "
        + changeLevels.length) ;
    }

    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized (m_main_connection_lock) {
        try {
          boolean[] result = uncommittedAreAclsUpToDate ( c,
            externalAclIDs, changeLevels) ;
          commit(c) ;
          return result ;
        }
        catch (Exception e) {
          try {
            rollback(c) ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          try {
            reopenMainConnection() ;
          }
          catch (SQLException sqlException) {
            if (LOG.beDebug()) {
              LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
            }
          }
          throw new AclPersistenceException(e) ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  /**
   * @return the ID of the used connection pool
   */
  protected final String getPoolID() {
    return m_poolID ;
  }

  /**
   * @param principal TBD: Description of the incoming method parameter
   * @return the value that's written into the database to identify the type of
   *      the principal
   */
  protected final int getPrincipalTypeTag ( IUMPrincipal principal ) {
    return getPrincipalTypeTag(principal.getType()) ;
  }

  /**
   * @param principalType TBD: Description of the incoming method parameter
   * @return the value that's written into the database to identify the
   *      principal type
   */
  protected final int getPrincipalTypeTag ( int principalType ) {
    switch (principalType) {
      case IUMPrincipal.IUSER:
        return PRINCIPAL_TYPE_TAG_USER ;
      case IUMPrincipal.IGROUP:
        return PRINCIPAL_TYPE_TAG_GROUP ;
      case IUMPrincipal.IROLE:
        return PRINCIPAL_TYPE_TAG_ROLE ;
    }
    return PRINCIPAL_TYPE_TAG_USER ;
  }

  /**
   * @param tag TBD: Description of the incoming method parameter
   * @return the principal type that belongs to the tag (i.e. the value that's
   *      written into the database to identify a principal type)
   */
  protected final int getPrincipalTypeFromTag ( int tag ) {
    switch (tag) {
      case PRINCIPAL_TYPE_TAG_USER:
        return IUMPrincipal.IUSER ;
      case PRINCIPAL_TYPE_TAG_GROUP:
        return IUMPrincipal.IGROUP ;
      case PRINCIPAL_TYPE_TAG_ROLE:
        return IUMPrincipal.IROLE ;
    }
    return IUMPrincipal.IUSER ;
  }

  /**
   * @param id principal ID
   * @param type principal type
   * @param raw when <code>true</code>, return an object even if the user management doesn't know that principal anymore
   * @return a IUMPrincipal object for the specified principal (<code>null</code> if the
   *      principal does not exist) The principal is NOT populated!
   */
  protected IUMPrincipal getUMPrincipal(String id, int type, boolean raw) {

    IUMPrincipal result = null;
    
    try {
      switch (type) {
        case IUMPrincipal.IUSER:
          IUserFactory userFactory = WPUMFactory.getUserFactory();
          if (userFactory != null) {
            result = userFactory.getUser(id, IUserFactory.TRUSTED);
          }
          break;
        case IUMPrincipal.IGROUP:
          IGroupFactory groupFactory = WPUMFactory.getGroupFactory();
          if (groupFactory != null) {
            result = groupFactory.getGroup(id);
          }
          break;
        case IUMPrincipal.IROLE:
          IRoleFactory roleFactory = WPUMFactory.getRoleFactory();
          if (roleFactory != null) {
            result = roleFactory.getRole(id);
          }
          break;
      }
    }
    catch (Exception e) {
      s_log.warningT("getUMPrincipal(2593)", "can't get principal " + id + " (" + getPrincipalTypeText(type) + ") from user management: " + LoggingFormatter.extractCallstack(e));
      return null;
    }
    
    if (result == null && raw) {
      try {
        result = new UMPrincipalRaw(type, id);
      }
      catch (Exception e) {
        s_log.warningT("getUMPrincipal", "can't get raw principal " + id + " (" + getPrincipalTypeText(type) + "): " + LoggingFormatter.extractCallstack(e));
      }
    }
    
    return result;
  }

  /**
   * Populate a principal
   *
   * @param principal TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  protected final IUMPrincipal populate ( IUMPrincipal principal ) {
    try {
      switch (principal.getType()) {
        case IUMPrincipal.IUSER:
          IUserFactory userFactory = WPUMFactory.getUserFactory() ;
          if (userFactory != null) {
            return userFactory.getUser(principal.getId()) ;
          }
          break ;
        case IUMPrincipal.IGROUP:
          IGroupFactory groupFactory = WPUMFactory.getGroupFactory() ;
          if (groupFactory != null) {
            return groupFactory.getGroup(principal.getId()) ;
          }
          break ;
        case IUMPrincipal.IROLE:
          IRoleFactory roleFactory = WPUMFactory.getRoleFactory() ;
          if (roleFactory != null) {
            return roleFactory.getRole(principal.getId()) ;
          }
          break ;
      }
    }
    catch (Exception e) {
      s_log.warningT("populate(2631)", "can't get populated principal "
        + principal.getId() + " (" + getPrincipalTypeText(principal.getType())
        + ") from user management: " + LoggingFormatter.extractCallstack(e)) ;
      return null ;
    }

    return null ;
  }

  /**
   * @param type TBD: Description of the incoming method parameter
   * @return a displayname for a principal type (for logging)
   */
  private final String getPrincipalTypeText(int type) {
    switch (type) {
      case IUMPrincipal.IUSER:
        return "user" ;
      case IUMPrincipal.IGROUP:
        return "group" ;
      case IUMPrincipal.IROLE:
        return "role" ;
    }
    return "" ;
  }

  /**
   * Add permissions to the database which are always supported by this ACL
   * manager (i.e. fullcontrol)
   *
   * @exception AclPersistenceException Exception raised in failure situation
   */
  protected final void checkInsertInitialPermissions ( Connection c )
  throws AclPersistenceException {

    try {
      if (!uncommittedCheckInitialPermissions(c)) {
        uncommittedInsertInitialPermissions(c) ;
      }
      if (!uncommittedCheckInitialSupportedPermissions(c)) {
        uncommittedInsertInitialSupportedPermissions(c) ;
      }

      commit(c) ;
    }
    catch (Exception e) {
      try {
        rollback(c) ;
      }
      catch (SQLException sqlException) {
        if (LOG.beDebug()) {
          LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(sqlException));
        }
      }
      throw new AclPersistenceException(e) ;
    }
  }

  abstract protected AclRec uncommittedGetAcl ( Connection connection,
    String externalAclID ) throws SQLException ;

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param aclIDs TBD: Description of the incoming method parameter
   * @return ACL objects for ACLs in the database (some may be null) (don't get
   *      them all at once to limit the size of the SQL statement)
   * @exception SQLException Exception raised in failure situation
   */
  private final AclRec[] uncommittedGetAcls ( Connection connection,
  String[] aclIDs ) throws SQLException {

    final int MAX_ACLS_PER_QUERY = 100 ;

    if (aclIDs.length == 0) {
      return new AclRec[0] ;
    }

    AclRec[] result = new AclRec[aclIDs.length] ;

    int takeCount = Math.min(aclIDs.length, MAX_ACLS_PER_QUERY) ;
    String[] localAclIDs = new String[takeCount] ;
    for (int i = 0 ; i < aclIDs.length ; i += MAX_ACLS_PER_QUERY) {
      takeCount = Math.min(aclIDs.length - i, MAX_ACLS_PER_QUERY) ;
      if ( takeCount != localAclIDs.length ) {
        localAclIDs = new String[takeCount] ;
      }
      System.arraycopy ( aclIDs, i, localAclIDs, 0, takeCount ) ;
      AclRec[] acls = uncommittedGetAclsFromDB ( connection, localAclIDs ) ;
      System.arraycopy ( acls, 0, result, i, takeCount ) ;
    }

    return result ;
  }

  abstract protected String uncommittedGetInternalAclIDFromEntryID (
    Connection connection, long entryID ) throws SQLException ;

  abstract protected AclRec[] uncommittedGetAclsFromDB ( Connection connection,
    String[] internalAclIDs ) throws SQLException ;

  abstract protected boolean uncommittedIsAclUpToDate ( Connection connection,
    String internalAclID, long changeLevel ) throws SQLException ;

  abstract protected boolean[] uncommittedAreAclsUpToDateInDB (
    Connection connection, String[] internalAclIDs, long[] changeLevels )
    throws SQLException ;

  abstract protected void uncommittedIncreaseAclChangeLevel (
    Connection connection, String internalAclID ) throws SQLException ;

  abstract protected boolean uncommittedCheckAcl ( Connection connection,
    String internalAclID ) throws SQLException ;

  abstract protected boolean uncommittedSetAclEntrySortIndex (
    Connection con, long aclEntryID, int sortIndex ) throws SQLException ;

  abstract protected boolean uncommittedSetAclEntryPropagation (
    Connection connection, long aclEntryID, boolean propagate )
    throws SQLException ;

  abstract protected boolean uncommittedCheckAclEntries ( Connection connection,
    String internalAclID ) throws SQLException ;

  abstract protected boolean uncommittedCheckPermission ( Connection connection,
    String permissionName ) throws SQLException ;

  abstract protected AclEntryList uncommittedGetAclEntries (
    Connection connection, String internalAclID, boolean raw ) throws SQLException ;

  abstract protected AclEntryList[] uncommittedGetAclEntries (
    Connection connection, AclRec[] acls, boolean raw ) throws SQLException ;

  abstract protected List uncommittedGetAffectedAclIDs (
    Connection connection, String permissionName ) throws SQLException ;

  abstract protected boolean uncommittedAddAcl ( Connection connection,
    String internalAclID, IObjectType objectType ) throws SQLException ;

  abstract protected long uncommittedAddAclEntry ( Connection connection,
    String internalAclID, JDBCAclEntry aclEntry ) throws SQLException ;

  abstract protected boolean uncommittedRemoveAclEntry ( Connection connection,
    long aclEntryID ) throws SQLException ;

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalIDs TBD: Description of the incoming method parameter
   * @param changeLevels TBD: Description of the incoming method parameter
   * @return up to date flags for ACLs (don't get them all at once to limit the
   *      size of the SQL statement)
   * @exception SQLException Exception raised in failure situation
   */
  private final boolean[] uncommittedAreAclsUpToDate ( Connection connection,
  String[] internalIDs, long[] changeLevels ) throws SQLException {

    final int MAX_ACLS_PER_QUERY = 100 ;

    if (internalIDs.length == 0) {
      return new boolean[0] ;
    }

    boolean[] result = new boolean[internalIDs.length] ;
    int takeCount = Math.min(internalIDs.length, MAX_ACLS_PER_QUERY) ;
    String[] internalTakeIDs = new String[takeCount] ;
    long[] changeTakeLevels = new long[takeCount] ;

    for (int i = 0 ; i < internalIDs.length ; i += MAX_ACLS_PER_QUERY) {
      takeCount = Math.min(internalIDs.length - i, MAX_ACLS_PER_QUERY) ;
      if ( takeCount != internalTakeIDs.length ) { 
        internalTakeIDs = new String[takeCount] ;
        changeTakeLevels = new long[takeCount] ;
      }
      System.arraycopy ( internalIDs, i, internalTakeIDs, 0, takeCount ) ;
      System.arraycopy ( changeLevels, i, changeTakeLevels, 0, takeCount ) ;
      boolean[] flags = uncommittedAreAclsUpToDateInDB ( connection,
        internalTakeIDs, changeTakeLevels ) ;
      System.arraycopy ( flags, 0, result, i, takeCount ) ;
    }

    return result ;
  }

  abstract protected boolean uncommittedIsOwner ( Connection connection,
    String internalAclID, IUMPrincipal owner ) throws SQLException ;

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @return true iff the specified principal is the last owner of the ACL
   * @exception SQLException Exception raised in failure situation
   */
  private boolean uncommittedIsLastOwner ( Connection connection,
  String internalAclID, IUMPrincipal owner ) throws SQLException {
    return uncommittedIsLastOwner ( connection, internalAclID, owner.getId(),
      owner.getType()) ;
  }

  abstract protected boolean uncommittedIsLastOwner ( Connection connection,
    String internalAclID, String ownerID, int ownerType ) throws SQLException ;

  abstract protected UMPrincipalList uncommittedGetOwners (
    Connection connection, String internalAclID, boolean raw ) throws SQLException ;

  abstract protected UMPrincipalList[] uncommittedGetOwners (
    Connection connection, AclRec[] acls, boolean raw ) throws SQLException ;

  /**
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return true iff the ACL has some owners
   * @exception SQLException Exception raised in failure situation
   */
  abstract boolean uncommittedHasOwnersInDB ( Connection connection,
    String internalAclID ) throws SQLException ;

  private final boolean uncommittedCheckOwners ( Connection connection,
  String internalAclID ) throws SQLException {
    boolean rv = uncommittedHasOwnersInDB ( connection, internalAclID ) ;
    return rv ;
  }

  abstract protected boolean uncommittedAddOwner ( Connection connection,
    String internalAclID, IUMPrincipal owner ) throws SQLException ;

  /**
   * Remove an owner from the ACL
   *
   * @param connection TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  private final boolean uncommittedRemoveOwner ( Connection connection,
  String internalAclID, IUMPrincipal owner ) throws SQLException {
    return uncommittedRemoveOwner ( connection, internalAclID, owner.getId(),
      owner.getType()) ;
  }

  abstract protected boolean uncommittedRemoveOwner ( Connection connection,
    String internalAclID, String ownerID, int ownerType ) throws SQLException ;

  abstract protected boolean uncommittedRemoveOwner ( Connection connection,
    String ownerID, int ownerType ) throws SQLException ;

  abstract protected IAclPermissionList uncommittedGetPermissionMembers (
    Connection connection, String permissionName ) throws SQLException ;

  abstract protected IAclPermissionList
    uncommittedGetAllPermissionsNotFullControl ( Connection connection )
    throws SQLException ;

  abstract protected void uncommittedRemoveOwners ( Connection connection,
    String internalAclID ) throws SQLException ;

  abstract protected void uncommittedRemoveAclEntries ( Connection connection,
    String internalAclID ) throws SQLException ;

  abstract protected void uncommittedRemoveAclEntries ( Connection connection,
    String principalID, int principalType ) throws SQLException ;

  /**
   * Insert the initial permissions (i.e. fullcontrol)
   *
   * @param connection TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  private final void uncommittedInsertInitialPermissions (
  Connection connection ) throws SQLException {
    uncommittedAddPermission ( connection,
      IAclPermission.ACL_PERMISSION_FULL_CONTROL, true ) ;
  }

  /**
   * Insert the initial supported permissions (i.e. fullcontrol)
   *
   * @param connection TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  private final void uncommittedInsertInitialSupportedPermissions (
  Connection connection ) throws SQLException {

    uncommittedAddSupportedPermission(connection,
      IAclPermission.ACL_PERMISSION_FULL_CONTROL,
      IObjectType.OBJECT_TYPE_LEAF) ;
    uncommittedAddSupportedPermission(connection,
      IAclPermission.ACL_PERMISSION_FULL_CONTROL,
      IObjectType.OBJECT_TYPE_NODE) ;
  }

  abstract protected boolean uncommittedCheckInitialPermissions (
    Connection connection ) throws SQLException ;

  abstract protected boolean uncommittedCheckInitialSupportedPermissions (
    Connection connection ) throws SQLException ;

  abstract protected boolean uncommittedIsSupportedPermission (
    Connection connection, String permissionName, String objectTypeName )
    throws SQLException ;

  abstract protected IAclPermissionList uncommittedGetSupportedPermissions (
    Connection connection, String objectTypeName ) throws SQLException ;

  abstract protected IAclPermission uncommittedGetPermission (
    Connection connection, String permissionName ) throws SQLException ;

  abstract protected boolean uncommittedIsUsedPermission (
    Connection connection, String permissionName ) throws SQLException ;

  abstract protected boolean uncommittedAddPermission ( Connection connection,
    String permissionName, boolean isPredefined ) throws SQLException ;

  abstract protected boolean uncommittedRemovePermission (
    Connection connection, String permissionName ) throws SQLException ;

  abstract protected boolean uncommittedAddSupportedPermission (
    Connection connection, String permissionName, String objectTypeName )
    throws SQLException ;

  abstract protected boolean uncommittedRemoveSupportedPermission (
    Connection connection, String permissionName, String objectTypeName )
    throws SQLException ;

  abstract protected boolean uncommittedAddPermissionMember (
    Connection connection, String permissionName, String memberName )
    throws SQLException ;

  abstract protected boolean uncommittedRemovePermissionMember (
    Connection connection, String permissionName, String memberName )
    throws SQLException ;

  abstract protected boolean uncommittedRemovePermissionMembers (
    Connection connection, String permissionName ) throws SQLException ;

  abstract protected boolean uncommittedIsPredefinedPermission (
    Connection connection, String name ) throws SQLException ;

  abstract protected void uncommittedRemoveAcl ( Connection connection,
    String internalAclID ) throws SQLException ;

  abstract protected boolean uncommittedLockAcl ( Connection connection,
    String internalAclID, IUMPrincipal lockingUser ) throws SQLException ;

  abstract protected boolean uncommittedUnlockAcl ( Connection connection,
    String internalAclID ) throws SQLException ;

  abstract protected boolean uncommittedIsAclLocked ( Connection connection,
    String internalAclID ) throws SQLException ;

  abstract protected IUser uncommittedGetAclLockingUser ( Connection connection,
    String internalAclID ) throws SQLException ;

  abstract protected boolean uncommittedChangeAclID_in_AclTable (
    Connection connection, String oldID, String newID ) throws SQLException ;

  abstract protected boolean uncommittedChangeAclID_in_AclEntryTable (
    Connection connection, String oldID, String newID ) throws SQLException ;

  abstract protected boolean uncommittedChangeAclID_in_OwnerTable (
    Connection connection, String oldID, String newID ) throws SQLException ;

  abstract public long getDBVersion() ;

  /*
  * ancient 'single connection' connection handling methods;
  * moved down to the various implementations so that in case
  * native jdbc these methods can just work on the m_main_connection
  * member and bounce this connection around;
  *
  * in the case of OpenSQL m_main_connection will just be an object
  * used for synchronization only; further, 'openConnection',
  * 'reopenMainConnection', and 'closeConnection' will be dummy
  * routines that just fake some work on m_main_connection; the
  * real handling of database connections will be done in the
  * two new methods 'getDBConnection' and 'returnDBConnection'.
  *
  * Of course, these two new methods need to be implemented by the
  * native jdbc connection handling although they'll be just dummies
  * over there.
  *
  * why all this? well, in the OpenSQL you're simply not supposed to
  * hog a connection; further, using one and the same connection
  * across thread boundaries is likely not going to work as rumor has
  * it that some internal information on the connection is kept in
  * thread local data space.
  */
/*
  abstract protected Connection openConnection ( String poolId )
    throws SQLException, AclPersistenceException ;
  abstract protected void closeConnection ( Connection connection )
    throws SQLException ;
*/
  abstract protected void reopenMainConnection() throws SQLException,
    AclPersistenceException ;

  abstract protected Connection getDBConnection() throws SQLException,
    AclPersistenceException ;
  abstract protected void returnDBConnection ( Connection c ) ;

  /**
   * Open a new database connection
   *
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   */
//  protected static Connection openConnection ( String poolId )
//  throws SQLException, AclPersistenceException {
//
//    JDBCConnectionPoolManager poolManager =
//      JDBCConnectionPoolManager.getInstance() ;
//    if (poolManager == null) {
//      throw new AclPersistenceException (
//        "JDBCConnectionPoolManager.getInstance() failed") ;
//    }
//
//    JDBCConnectionPool pool = poolManager.getPool ( poolId ) ;
//    if (pool == null) {
//      throw new AclPersistenceException (
//        "JDBCConnectionPoolManager.getPool() failed") ;
//    }
//
//    Connection connection = pool.getConnection() ;
//    if ( connection == null ) {
//      throw new AclPersistenceException (
//        "JDBCConnectionPool.getConnection() failed") ;
//    }
//
//    connection.setAutoCommit(false) ;
//    connection.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED) ;
//    return connection ;
//  }
//
//  /**
//   * Close a database connection
//   *
//   * @param connection TBD: Description of the incoming method parameter
//   * @exception SQLException Exception raised in failure situation
//   */
//  private static void closeConnection ( Connection connection )
//  throws SQLException {
//    if (connection == null) {
//      throw new java.lang.IllegalArgumentException() ;
//    }
//    connection.close() ;
//  }
//
//  /**
//   * Reopen the main database connection (done in case of database errors to
//   * survive a database restart)
//   *
//   * @exception SQLException Exception raised in failure situation
//   * @exception AclPersistenceException Exception raised in failure situation
//   */
//  protected final void reopenMainConnection() throws SQLException,
//  AclPersistenceException {
//    try {
//    closeConnection(m_main_connection) ;
//    }
//    catch ( Exception e1 ) {
//      String s = e1.getMessage() ;
//      /* log something here */
//    }
//    finally {
//      m_main_connection = null ;
//    }
//    m_main_connection = openConnection(m_poolID) ;
//  }

  protected static JDBCConnectionPool getPool ( String poolId )
  throws SQLException, AclPersistenceException {

    JDBCConnectionPoolManager pm = JDBCConnectionPoolManager.getInstance() ;
    if ( pm == null ) {
      throw new AclPersistenceException (
        "JDBCConnectionPoolManager.getInstance() failed") ;
    }

    JDBCConnectionPool pool = pm.getPool ( poolId ) ;
    if ( pool == null ) {
      throw new AclPersistenceException (
        "JDBCConnectionPoolManager.getPool() failed") ;
    }
    return pool ;
  }

  /**
   * @param name TBD: Description of the incoming method parameter
   * @return a string used as a key in the permissions cache
   */
  private final String cacheGetPermissionKey ( String name ) {
    return m_applicationID + SEPARATOR + m_managerID + SEPARATOR
      + KEY_PERMISSION_PREFIX + SEPARATOR + name ;
  }

  /**
   * Get a permission from the cache
   *
   * @param name TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  protected final IAclPermission cacheGetPermission ( String name ) {
    if (m_permissionCache != null) {
      try {
        ICacheEntry cacheEntry = m_permissionCache.getEntry (
          cacheGetPermissionKey(name)) ;
        if (cacheEntry != null) {
          return (IAclPermission)cacheEntry.getObject() ;
        }
      }
      catch (CacheException e) {
        s_log.warningT("cacheGetPermission(4936)", "permission cache error: "
          + LoggingFormatter.extractCallstack(e)) ;
      }
    }
    return null ;
  }

  /**
   * Add a permission to the cache
   *
   * @param permission TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  protected final IAclPermission cacheAddPermission(IAclPermission permission) {
    if (m_permissionCache != null) {
      try {
        m_permissionCache.addEntry(cacheGetPermissionKey(
          permission.getName()), permission) ;
      }
      catch (Exception e) {
        s_log.warningT("cacheAddPermission(4955)", "permission cache error: "
          + LoggingFormatter.extractCallstack(e)) ;
      }
    }

    return permission ;
  }

  /**
   * Remove a permission from the permissions cache
   *
   * @param permissionName TBD: Description of the incoming method parameter
   */
  protected final void cacheRemovePermission(String permissionName) {
    if (m_permissionCache != null) {
      try {
        m_permissionCache.removeEntry(cacheGetPermissionKey(permissionName)) ;
      }
      catch (CacheException e) {
        s_log.warningT("cacheRemovePermission(4973)",
          "permission cache error: " + LoggingFormatter.extractCallstack(e)) ;
      }
    }
  }

  /**
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return the key for the permissions that apply to the specified object type
   *      in the permissions cache
   */
  private final String cacheGetSupportedPermissionsKey (
  String objectTypeName ) {
    return m_applicationID + SEPARATOR + m_managerID + SEPARATOR
      + KEY_SUPPORTED_PERMISSIONS_PREFIX + SEPARATOR + objectTypeName ;
  }

  /**
   * Add supported permissions to the permissions cache
   *
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  protected final IAclPermissionList cacheAddSupportedPermissions (
  String objectTypeName, IAclPermissionList permissions ) {
    if (m_permissionCache != null) {
      try {
        m_permissionCache.addEntry ( cacheGetSupportedPermissionsKey (
          objectTypeName ), permissions) ;
      }
      catch ( CacheException e ) {
        s_log.warningT("cacheAddSupportedPermissions(5001)",
          "permission cache error: " + LoggingFormatter.extractCallstack(e)) ;
      }
    }

    return permissions ;
  }

  /**
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return supported permissions as they are stored in the permissions cache
   *      (returns null if they are not stored)
   */
  private final IAclPermissionList cacheGetSupportedPermissions (
  String objectTypeName ) {
    if (m_permissionCache != null) {
      try {
        ICacheEntry cacheEntry = m_permissionCache.getEntry (
          cacheGetSupportedPermissionsKey(objectTypeName)) ;
        if (cacheEntry != null) {
          return (IAclPermissionList)cacheEntry.getObject() ;
        }
      }
      catch (CacheException e) {
        s_log.warningT("cacheGetSupportedPermissions(5022)",
          "permission cache error: " + LoggingFormatter.extractCallstack(e)) ;
      }
    }

    return null ;
  }

  /**
   * @param parent TBD: Description of the incoming method parameter
   * @param child TBD: Description of the incoming method parameter
   * @return true if the child permission is already contained in the parent
   *      permission
   * @exception AclPersistenceException Exception raised in failure situation
   */
  private final boolean isPermissionMember ( IAclPermission parent,
  IAclPermission child ) throws AclPersistenceException {
    IAclPermissionList children = parent.getMembers() ;
    IAclPermissionListIterator iterator = children.iterator() ;
    while (iterator.hasNext()) {
      IAclPermission listChild = iterator.next() ;
      if (listChild.equals(child)) {
        return true ;
      }
      if (isPermissionMember(listChild, child)) {
        return true ;
      }
    }
    return false ;
  }

  /**
   * Remove leading and trailing spaces from a string ; if the string is null,
   * return an empty string
   *
   * @param s TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  protected final String trim ( String s ) {
    if (s == null) {
      return "" ;
    }
    return s.trim() ;
  }
}
