/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 */
package com.sap.tc.cmi.util;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Abstract implementation of the ICMIObservableList interface.</p>
 * 
 * This class is not at all thread safe (for peformance reasons).</p>
 * 
 * This implementation modifies the contract that AbstractList defines for subclasses.</p>
 * All modifying methods of the List interface have been overridden using the Template/Hook 
 * pattern. The template implementations delegate all changes to hook methods and afterwards
 * send out notification events about the changes made. The hook methods only have to take care
 * about the change itself and <strong>must not</strong> fire events. To identify them and to 
 * better reflect their behavior, the names of the hook methods end in <tt>*Silently<l</tt>. <p>
 * 
 * Preferably subclasses should override the hook methods only as these are easier to implement. 
 * But if the template implementation of a modifying method doesn't fit, subclasses can override 
 * or extend that template method as well. If they override it, they have to implement the 
 * notification on their own. If they extend it, they must take care to inform listeners about 
 * a change <strong>after</strong> it has been completed. So calls to <tt>super</tt> typically will be 
 * the last step in an extended method.</p>
 * 
 * In general, subclasses must be careful in their modifier implementations when calling other 
 * methods that potentially result in notification events. Listeners usually expect to be notified 
 * only once about a single change.</p>
 *  
 * To implement an unmodifiable list, subclasses have to implement the methods <tt>size()</tt> and 
 * <tt>get(int index)</tt> as introduced by AbstractList.</p>
 * 
 * To implement a modifiable list, subclasses can override the <tt>setSilently(int index, Object o)</tt> 
 * method. If the list is variable-size, they additionally have to override the 
 * <tt>addSilently(int index, Object element)</tt> and <tt>removeSilently(int index)</tt> 
 * methods.<p>
 * 
 * This type can be called or extended by applications or frameworks using CMI.
 * 
 * @SAPCMIPart 1 
 * @author Frank Weigel
 * @version $Id: //tc/CommonModelInterface/630_VAL_REL/src/_cmi_api/java/com/sap/tc/cmi/util/CMIAbstractObservableList.java#1 $
 */
public abstract class CMIAbstractObservableList extends AbstractList implements ICMIObservableList {


  // ---- constructors -----------------------------------------------------------------
  
  /**
   * Constructor for CMIAbstractObservableList to be called by subclasses.
   */
  protected CMIAbstractObservableList() {
    super();
  }
  
  // ---- listener management -------------------------------------------------------------
 
  private List listeners = null;    

  /**
   * @see com.sap.tc.cmi.util.ICMIObservableList#addChangeListener(ICMIObservableListChangeListener)
   */
  public void addChangeListener(ICMIObservableListChangeListener listener) {
    if ( listeners == null )
      listeners = new ArrayList();
    listeners.add(listener);
  }

  /**
   * @see com.sap.tc.cmi.util.ICMIObservableList#removeChangeListener(ICMIObservableListChangeListener)
   */
  public void removeChangeListener(ICMIObservableListChangeListener listener) {
    if ( listeners != null )
      listeners.remove(listener);
  }

  /**
   * Private event object reused for ALL notifications made by this list. </p>
   * 
   * Note that this implies that changes can not be nested! If that constraint can not be 
   * fulfilled, this implementation will fail or has to be changed.
   */
  private final CMIObservableListEvent event = new CMIObservableListEvent(this);
  
  /**
   * fire an element changed event for the given element and index
   */
  protected final void fireElementChanged(Object o, int index) {
    if ( listeners != null && !listeners.isEmpty() ) {
      event.setData(o, index, index, CMIObservableListEvent.ELEMENT_CHANGED);
      for(Iterator it = listeners.iterator(); it.hasNext(); ) {
        ICMIObservableListChangeListener listener = (ICMIObservableListChangeListener) it.next();
        listener.elementChanged(event);
      }
    }
  }
    
  /**
   * fire an element added event for the given element and index
   */
  protected final void fireElementAdded(Object o, int index) {
    if ( listeners != null && !listeners.isEmpty() ) {
      event.setData(o, index, index, CMIObservableListEvent.ELEMENT_ADDED);
      for(Iterator it = listeners.iterator(); it.hasNext(); ) {
        ICMIObservableListChangeListener listener = (ICMIObservableListChangeListener) it.next();
        listener.elementAdded(event);
      }
    }
  }
    
  /**
   * fire an element removed event for the given element and index
   */
  protected final void fireElementRemoved(Object o, int index) {
    if ( listeners != null && !listeners.isEmpty() ) {
      event.setData(o, index, index, CMIObservableListEvent.ELEMENT_REMOVED);
      for(Iterator it = listeners.iterator(); it.hasNext(); ) {
        ICMIObservableListChangeListener listener = (ICMIObservableListChangeListener) it.next();
        listener.elementRemoved(event);
      }
    }
  }
    
  /**
   * fire a complex change event of the given type
   */
  protected final void fireComplexChange(int fromIndex, int toIndex, int type) {
    if ( listeners != null && !listeners.isEmpty() ) {
      event.setData(null, fromIndex, toIndex, type);
      for(Iterator it = listeners.iterator(); it.hasNext(); ) {
        ICMIObservableListChangeListener listener = (ICMIObservableListChangeListener) it.next();
        listener.complexChange(event);
      }
    }
  }

  // ---- standard list functionality, notifies all registered listeners ------
  
  /**
   * @see java.util.List#add(int, Object)
   */
  public void add(int index, Object element) {
    addSilently(index, element);
    fireElementAdded(element, index);
  }

  /**
   * @see java.util.Collection#add(Object)
   */
  public boolean add(Object o) {
    add(size(), o);
    return true;
  }

  /**
   * @see java.util.Collection#addAll(Collection)
   */
  public boolean addAll(Collection c) {
    return addAll(size(), c);
  }

  /**
   * @see java.util.List#addAll(int, Collection)
   */
  public boolean addAll(int index, Collection c) {
    int endIndex = addAllSilently(index, c);
    if (endIndex > index) {
      fireComplexChange(index, endIndex-1, CMIObservableListEvent.CONTIGUOUS_ELEMENTS_ADDED);
      return true;
    }
    return false;
  }

  /**
   * @see java.util.Collection#clear()
   */
  public void clear() {
    removeRangeSilently(0, size());
    fireComplexChange(-1, -1, CMIObservableListEvent.LIST_CLEARED);
  }

  /**
   * @see java.util.List#remove(int)
   */
  public Object remove(int index) {
    Object o = removeSilently(index);
    fireElementRemoved(o, index);
    return o;
  }

  /**
   * @see java.util.Collection#remove(Object)
   */
  public boolean remove(Object o) {
    int index = indexOf(o);
    if ( index >= 0 ) {
      o = remove(index);
      return true;
    }
    return false;
  }

  protected void removeRange(int fromIndex, int toIndex) {
    removeRangeSilently(fromIndex, toIndex);
    fireComplexChange(fromIndex, toIndex, CMIObservableListEvent.CONTIGUOUS_ELEMENTS_REMOVED);
  }
  
  /**
   * @see java.util.Collection#removeAll(Collection)
   */
  public boolean removeAll(Collection c) {
    boolean modified = false;
    for(int i=size(); --i >= 0; ) {
      if ( c.contains(get(i)) ) {
        removeSilently(i);
        modified = true;
      }
    }
    if ( modified )
      fireComplexChange(-1, -1, CMIObservableListEvent.MULTIPLE_ELEMENTS_REMOVED);
    return modified;
  }

  /**
   * @see java.util.Collection#retainAll(Collection)
   */
  public boolean retainAll(Collection c) {
    boolean modified = false;
    for(int i=size(); --i >= 0; ) {
      if ( !c.contains(get(i)) ) {
        removeSilently(i);
        modified = true;
      }
    }
    if ( modified )
      fireComplexChange(-1, -1, CMIObservableListEvent.MULTIPLE_ELEMENTS_REMOVED);
    return modified;
  }

  /**
   * @see java.util.List#set(int, Object)
   */
  public Object set(int index, Object element) {
    Object old = setSilently(index, element);
    fireElementChanged(old, index);
    return old;
  }

  // ---- hook methods without notification -------------------------------------

  public Object setSilently(int index, Object element) {
    throw new UnsupportedOperationException();
  }

  public void addSilently(int index, Object element) {
    throw new UnsupportedOperationException();
  }

  public Object removeSilently(int index) {
    throw new UnsupportedOperationException();
  }
  
  public int addAllSilently(int index, Collection c) {
    Iterator it = c.iterator();
    while (it.hasNext()) {
      addSilently(index++, it.next());
    }
    return index;
  }

  public void removeRangeSilently(int fromIndex, int toIndex) {
    for(int i=fromIndex; i<toIndex; i++)
      // always remove at the first position
      removeSilently(fromIndex);
  }

}
