/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 */
package com.sap.tc.cmi.model;

import java.util.Collection;

import com.sap.tc.cmi.exception.CMIException;
import com.sap.tc.cmi.metadata.ICMIModelObjectCollectionInfo;
import com.sap.tc.cmi.metadata.ICMIModelClassInfo;

/**
 * represents a query in a Model.
 * This type can be called, extended or implemented by applications or frameworks using CMI.
 *
 * @SAPCMIPart 1
 * @author Stephan Ritter, Frank Weigel
 * @version $Id: //tc/CommonModelInterface/630_VAL_REL/src/_cmi_api/java/com/sap/tc/cmi/model/ICMIQuery.java#1 $
 */

public interface ICMIQuery extends ICMIModelClassExecutable, ICMIModelClass
{
  /**
   * Constant string for implicit inputParamter relation role. If a query implementation also implements
   * ICMIGenericModelClass, it should return the input parameter as a result to 
   * <pre>
   *   getRelatedModelObject(ROLE_NAME_INPUT_PARAMETER);
   * </pre>
   * An implementation is free to additionally accept its own role name or names.
   */
  static final String ROLE_NAME_INPUT_PARAMETER = "com.sap.tc.cmi:queryInputParameter";

  /**
   * Constant string for implicit result relation role. If a query implementation also implements
   * ICMIGenericModelClass, it should return the query result as a result to 
   * <pre>
   *   getRelatedModelObject(ROLE_NAME_RESULT);
   * </pre>
   * An implementation is free to additionally accept its own role name or names.
   */
  static final String ROLE_NAME_RESULT = "com.sap.tc.cmi:queryResult";
  
  /**
   * executes the query. When called immediately after query execution <code>isDirty()</code>
   * will return <code>false</code>. 
   * @see com.sap.tc.cmi.model.ICMIModelClassExecutable#execute()
   */
  void execute() throws CMIException;

  /**
   * returns the number all matching model objects for the parameterized query independent of the
   * fetch size. If this is not supported, the method returns -1; if <code>countOf()</code> is too big or
   * too expensive to calculate {@link java.lang.Long#MAX_VALUE} will be returned as <code>long</code>.
   */
  long countOf();

  /**
   * returns <code>true</code> if the query is not yet executed or the result is not consistent with
   * input parameters any more and <code>false</code> otherwise.
   */
  boolean isDirty();

  /**
   * Returns the model object which holds the query input parameters or null if the query doesn't
   * support parameters. The returned object must be compliant with the CMI. 
   * That means it must be an instance of a generic Model Class (therefore implementing 
   * {@link ICMIGenericModelClass} or an instance of a typed Model Class (therefore supporting typed
   * accessor methods etc.)
   */
  Object getInputParameter();

  /**
   * returns the collection which matches the query. The collection may or may not be a 
   * {@link ICMIModelObjectCollection}.
   */
  Collection getResult();

  /**
   * returns the metadata for the model class holding the input parameters or 
   * <code>null</code> if no metadata is available
   */
  ICMIModelClassInfo associatedInputParameterInfo();

  /**
   * returns the metadata for the model object collection holding the query result or 
   * <code>null</code> if no such metadata is available
   */
  ICMIModelObjectCollectionInfo associatedResultInfo();

  /**
   * @directed
   * @supplierCardinality 0..1
   * @supplierRole inputParameterInfo
   * @label / 
   */
  /*# ICMIModelClassInfo lnkICMIModelClassInfo; */

  /**
   * @directed
   * @supplierCardinality 0..1
   * @supplierRole resultInfo
   * @label / 
   */
  /*# com.sap.tc.cmi.metadata.ICMIModelObjectCollectionInfo lnkICMIModelObjectCollectionInfo; */

  /**
   * @directed
   * @label /
   * @supplierCardinality 0..1
   * @supplierRole inputParameter 
   */
  /*# ICMIModelClass lnkICMIModelClass; */
}