/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 */
package com.sap.tc.cmi.model;

import java.util.Collection;

import com.sap.tc.cmi.metadata.ICMIModelClassInfo;

/**
 * Interface for a generic model class.
 * 
 * Model classes that implement this interface are treated specially from the
 * context. Instead of using reflection to access attributes, it uses
 * {@link #getAttributeValue(String)} and {@link #setAttributeValue(String, Object)}.
 * 
 * This type can be called, extended or implemented by applications or frameworks using CMI.
 *
 * @SAPCMIPart 1
 * @author Stephan Ritter, Uwe Reeder, Markus Cherdron
 * @version $Id: //tc/CommonModelInterface/630_VAL_REL/src/_cmi_api/java/com/sap/tc/cmi/model/ICMIGenericModelClass.java#1 $
 */
public interface ICMIGenericModelClass extends ICMIModelClass {

  /**
   * Returns the value of the attribute with the given <code>name</code> as an Object.
   * It is unspecified, if the attribute is copied, before it is returned, or not. 
   * @throws IllegalArgumentException if there is no attribute with the given name
   */
  Object getAttributeValue(String name);

  /**
   * Sets the attribute to given <code>value</code>.
   * It is unspecified, if <code>value</code> is copied, before it is set, or not. 
   * @throws IllegalArgumentException if there is no attribute with the given name
   */
  void setAttributeValue(String name, Object value);

  /**
   * Returns <code>null</code> or the appropriate object of type ICMIModelClass according
   * to the relation role in <code>targetRoleName</code>, which must have the cardinality 1 or 0..1.<p>
   * If the role is of cardinality 0..1,
   * <code>null</code> can be returned.<p>
   * If there is no relation with an the given target rolename, an 
   * <code>java.lang.IllegalArgumentException</code> is thrown.
   */ 
  ICMIModelClass getRelatedModelObject(String targetRoleName);

  /**
   * Sets the relation role <code>targetRoleName</code> of an x:0..1 or x:1 relation to 
   * object <code>o</code> of type ICMIModelClass.</p>
   * 
   * If o is <code>null</code> and the cardinality of the relation role is 1,
   * a <code>java.lang.NullPointerException</code> exception is thrown.</p>
   * 
   * If <code>this</code> has no target relation role with the given <code>targetRoleName</code>,
   * an <code>java.lang.IllegalArgumentException</code> is thrown.</p>
   * 
   * If the model class or the model are read only, or if the model implementation doesn't
   * support this method, an <code>java.lang.UnsupportedOperationException</code> is thrown.</p>
   * 
   * If the method throws an exception for any of the above reasons, no changes have been
   * made of course.
   */ 
  void setRelatedModelObject(String targetRoleName, ICMIModelClass o);

  /**
   * Returns the appropriate collection of objects of type ICMIModelClass
   * according to the relation role in <code>targetRoleName</code>,
   * which must have the cardinality * or 1..*.</p>
   * 
   * The collection should be immutable.</p>
   * 
   * If the role is of cardinality 0..*, the collection could be empty, but not
   * <code>null</code>.</p>
   * 
   * If there is no relation with the given target role name, an 
   * <code>java.lang.IllegalArgumentException</code> is thrown.</p>
   * 
   * If the method throws an exception, no changes are made of course.
   */ 
  Collection getRelatedModelObjects(String targetRoleName);
  
  /**
   * sets the relation role <code>targetRoleName</code> of an x:* or x:1..* relation to the Collection
   * <code>col</code>, which contains exclusively objects of type ICMIModelClass.<p>
   * If <code>col</code> is <code>null</code>,
   * a <code>java.lang.NullPointerException</code> exception is thrown.<p>
   * If <code>col</code> is empty and the cardinality of the relation role is 1..*,
   * a <code>java.lang.NullPointerException</code> exception is thrown.<p>
   * If <code>this</code> has no target relation role with the given <code>targetRoleName</code>,
   * a <code>java.lang.IllegalArgumentException</code> is thrown.<p>
   * If the model class or the model are read only, or if the model implementation doesn't
   * support this method, an <code>java.lang.UnsupportedOperationException</code> is thrown.<p>
   * If the method throws an exception, no changes are made of course.
   */ 
  void setRelatedModelObjects(String targetRoleName, Collection col);

  /**
   * adds <code>o</code> to the relation role <code>targetRoleName</code> according to the <code>add</code>
   * method in the <code>java.util.Collection</code> interface.<p>
   * If <code>o</code> is <code>null</code>, a <code>java.lang.NullPointerException</code> exception is thrown
   * and no changes are made of course.<p>
   * If <code>this</code> has no target relation role with the given <code>targetRoleName</code> and cardinality 
   * * or 1..*, a <code>java.lang.IllegalArgumentException</code> is thrown.<p>
   * If the model class or the model are read only, or if the model implementation doesn't
   * support this method, an <code>java.lang.UnsupportedOperationException</code> is thrown.<p>
   * If the method throws an exception, no changes are made of course.
   */
  boolean addRelatedModelObject(String targetRoleName, ICMIModelClass o);
 
  /**
   * removes <code>o</code> from the relation role <code>targetRoleName</code> according to the
   * <code>remove</code> method in the <code>java.util.Collection</code> interface.<p>
   * If <code>o</code> is <code>null</code> or <code>o</code> was not in the collection,
   * the method silently returns.<p>
   * If <code>o</code> is the last element in the relation role, and the cardinality of the role
   * is 1..*, a <code>java.lang.IllegalArgumentException</code> is thrown.<p>
   * If <code>this</code> has no target relation role with the given <code>targetRoleName</code> and cardinality 
   * * or 1..*, a <code>java.lang.IllegalArgumentException</code> is thrown.<p>
   * If the model class or the model are read only, or if the model implementation doesn't
   * support this method, an <code>java.lang.UnsupportedOperationException</code> is thrown.<p>
   * If the method throws an exception, no changes are made of course.
   */ 
  boolean removeRelatedModelObject(String targetRoleName, ICMIModelClass o);
  
  /**
   * returns the metadata, i.e. an object of type ICMIModelClassInfo, corresponding
   * to this model class. Each implementation has to
   * support this method and must not return <code>null</code>.
   */
  ICMIModelClassInfo associatedModelClassInfo();
}
