/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 */

package com.sap.tc.cmi.exception;

import java.text.MessageFormat;
import java.util.Locale;
import java.util.TimeZone;

import com.sap.exception.BaseException;
import com.sap.localization.LocalizableText;
import com.sap.localization.LocalizationException;

/**
 * Base class for exceptions thrown by methods of the Common Model Interface. 
 * It extends {@link com.sap.exception.BaseException}. For further information see there.
 * This type can be called or extendeded by applications or frameworks using CMI.
 *
 * @SAPCMIPart 1
 * @author Frank Weigel
 * @version $Id: //tc/CommonModelInterface/630_VAL_REL/src/_cmi_api/java/com/sap/tc/cmi/exception/CMIException.java#3 $
 */
public class CMIException extends BaseException
{
  private Object[] parameters;

  /**
   * Creates an exception with a unparameterized text only. The <code>cause</code> and parameters
   * are set to <code>null</code>.
   */
  public CMIException(String messageText) {
    this(messageText, null, null);
  }

  /**
   * Creates an exception with the given (unparameterized) text and cause. 
   * The parameters are set to <code>null</code>.
   */
  public CMIException(String messageText, Throwable cause) {
    this(messageText, null, cause);
  }

  /**
   * Creates an exception with the given cause.
   */
  public CMIException(Throwable cause) {
    this(null, null, cause);
  }

  /**
   * Creates an exception with the text and text parameters.
   * @see java.util.MessageFormat
   */
  public CMIException(String messageText, Object[] parameters) {
    this(messageText, parameters, null);
  }

  /**
   * Creates an exception with the given text, parameters and cause. 
   * This is the default constructor for this class.
   */
  public CMIException(String messageText, Object[] parameters, Throwable cause)
  {
    super(new DummyLocalizableTextFormatter(messageText, parameters), cause);
    this.parameters = parameters;
  }

  // ---- methods

  /**
   * Returns the array of parameters that are part of the message text or null if there are no parameters.
   */
  public Object[] getParameters() {
    return this.parameters;
  }
  
  /**
   * Inner class for pretending localized texts, as long as there are none.
   * @deprecated This class is not part of the CMI API and will be removed in NW04 Stack09.
   */
  private static class DummyLocalizableTextFormatter implements LocalizableText {
    
    private DummyLocalizableTextFormatter(String messageText, Object[] parameters) {
      text = messageText == null ? null : MessageFormat.format(messageText, parameters);
    }
    
    /**
     * @see com.sap.localization.LocalizableText#finallyLocalize()
     */
    public void finallyLocalize() throws LocalizationException {
    }

    /**
     * @see com.sap.localization.LocalizableText#finallyLocalize(Locale, TimeZone)
     */
    public void finallyLocalize(Locale arg0, TimeZone arg1) throws LocalizationException {
    }

    /**
     * @see com.sap.localization.LocalizableText#finallyLocalize(Locale)
     */
    public void finallyLocalize(Locale arg0) throws LocalizationException {
    }

    /**
     * @see com.sap.localization.LocalizableText#finallyLocalize(TimeZone)
     */
    public void finallyLocalize(TimeZone arg0) throws LocalizationException {
    }

    /**
     * @see com.sap.localization.LocalizableText#format()
     */
    public String format() throws LocalizationException {
      return text;
    }

    /**
     * @see com.sap.localization.LocalizableText#format(Locale, TimeZone)
     */
    public String format(Locale arg0, TimeZone arg1) throws LocalizationException {
      return this.format();
    }

    /**
     * @see com.sap.localization.LocalizableText#format(Locale)
     */
    public String format(Locale arg0) throws LocalizationException {
      return this.format();
    }

    /**
     * @see com.sap.localization.LocalizableText#format(TimeZone)
     */
    public String format(TimeZone arg0) throws LocalizationException {
      return this.format();
    }

    /**
     * @see com.sap.localization.Localizable#getLocale()
     */
    public Locale getLocale() {
      return locale;
    }

    /**
     * @see com.sap.localization.Localizable#getTimeZone()
     */
    public TimeZone getTimeZone() {
      return timeZone;
    }

    /**
     * @see com.sap.localization.Localizable#setLocale(Locale)
     */
    public void setLocale(Locale arg0) {
      locale = arg0;
    }

    /**
     * @see com.sap.localization.Localizable#setTimeZone(TimeZone)
     */
    public void setTimeZone(TimeZone arg0) {
      timeZone = arg0;
    }

    private TimeZone timeZone;
    private Locale   locale;
    private final String text;
  }
}
