package com.sap.caf.rt.util;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;

import com.sap.caf.rt.bol.context.CAFContext;
import com.sap.tc.logging.Category;
import com.sap.tc.logging.Location;
import com.sap.util.monitor.jarm.ConfMonitor;
import com.sap.util.monitor.jarm.IMonitor;


/**
 *This class wraps the standard SAP Exception Framework and 
 * extends it with CAF specific logging, tracing and application
 * monitoring.  Use constructors for entering/exiting for
 * important methods that need logging and monitoring. The monitoring
 * functionality is called inside these methods with required monitoring level
 * 
 */

public final class CAFPublicLogger {

	/**	
	 * Trace level for messages that should be created always. 
	 * Corresponds to default JARM trace level 0
	 */
	public static final int LEVEL_BASIC = 0;

	/**
	 * Trace level for messages that should be created when single activity trace enabled
	 *Corresponds to JARM trace level 1
	 **/
	public static final int LEVEL_MEDIUM = 1;

	/**
	 * Trace level for messages that should be created when more  detailed analysis needed
	 * Corresponds to JARM trace level 2
	 */
	public static final int LEVEL_DETAIL = 2;

	/**
	 * This determines the location to which the trace files are written and 
	 * is related to the value in the log-configuration.xml file of CAF
	 */
	public static final Location LOC_CAF =
		Location.getLocation(CAFPublicLogger.class);

	private static Map locations = new HashMap();

	/**
	 * This determines the location to which the log files are written 
	 */
	public final static Category categoryCAF =
		Category.getCategory(Category.APPLICATIONS, "/Applications/CAF");
	/**
	 * @deprecated 
	 */	
	public final static Category categoryXAPPS =
		Category.getCategory(Category.APPLICATIONS, "/Applications/XAPPS");

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param application
	 */
	public static void entering(
		String user,
		String request,
		String methodName,
		String application) {
		Location.getLocation(application).entering(methodName);
		CAFContext.getMonitor(user, request).startComponent(
			ConfMonitor.COMPLEVELDETAIL,
			methodName);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param application
	 * @param traceLevel
	 */
	public static void entering(
		String user,
		String request,
		String methodName,
		String application,
		int traceLevel) {
		entering(
			user,
			request,
			methodName,
			Location.getLocation(application),
			traceLevel);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param loc
	 */
	public static void entering(
		String user,
		String request,
		String methodName,
		Location loc) {
		loc.entering(methodName);
		CAFContext.getMonitor(user, request).startComponent(
			ConfMonitor.COMPLEVELDETAIL,
			methodName);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param loc
	 * @param traceLevel
	 */
	public static void entering(
		String user,
		String request,
		String methodName,
		Location loc,
		int traceLevel) {
		loc.entering(methodName);

		if (traceLevel == 0)
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELBASIC,
				methodName);
		else if (traceLevel == 1)
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELMEDIUM,
				methodName);
		else
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELDETAIL,
				methodName);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param application
	 * @param params
	 */
	public static void entering(
		String user,
		String request,
		String methodName,
		String application,
		Object[] params) {
		Location.getLocation(application).entering(methodName, params);
		CAFContext.getMonitor(user, request).startComponent(
			ConfMonitor.COMPLEVELDETAIL,
			methodName);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param application
	 * @param params
	 * @param traceLevel
	 */
	public static void entering(
		String user,
		String request,
		String methodName,
		String application,
		Object[] params,
		int traceLevel) {
		Location.getLocation(application).entering(methodName, params);
		if (traceLevel == 0)
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELBASIC,
				methodName);
		else if (traceLevel == 1)
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELMEDIUM,
				methodName);
		else
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELDETAIL,
				methodName);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param loc
	 * @param params
	 */
	public static void entering(
		String user,
		String request,
		String methodName,
		Location loc,
		Object[] params) {
		loc.entering(methodName, params);
		CAFContext.getMonitor(user, request).startComponent(
			ConfMonitor.COMPLEVELDETAIL,
			methodName);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param loc
	 * @param params
	 * @param traceLevel
	 */
	public static void entering(
		String user,
		String request,
		String methodName,
		Location loc,
		Object[] params,
		int traceLevel) {
		loc.entering(methodName, params);
		if (traceLevel == 0)
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELBASIC,
				methodName);
		else if (traceLevel == 1)
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELMEDIUM,
				methodName);
		else
			CAFContext.getMonitor(user, request).startComponent(
				ConfMonitor.COMPLEVELDETAIL,
				methodName);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param application
	 */
	public static void exiting(
		String user,
		String request,
		String methodName,
		String application) {
		Location.getLocation(application).exiting();
		Location.getLocation(application).exiting();
		IMonitor mon = CAFContext.getMonitor(user, request);
		mon.endComponent(ConfMonitor.COMPLEVELDETAIL, methodName);
		mon.endRequest(request);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param loc
	 */
	public static void exiting(
		String user,
		String request,
		String methodName,
		Location loc) {
		loc.exiting();
		IMonitor mon = CAFContext.getMonitor(user, request);
		mon.endComponent(ConfMonitor.COMPLEVELDETAIL, methodName);
		mon.endRequest(request);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param loc
	 * @param res
	 */
	public static void exiting(
		String user,
		String request,
		String methodName,
		Location loc,
		Object res) {
		loc.exiting(res);
		IMonitor mon = CAFContext.getMonitor(user, request);
		mon.endComponent(ConfMonitor.COMPLEVELDETAIL, methodName);
		mon.endRequest(request);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param loc
	 * @param traceLevel
	 */
	public static void exiting(
		String user,
		String request,
		String methodName,
		Location loc,
		int traceLevel) {
		loc.exiting();
		IMonitor mon = CAFContext.getMonitor(user, request);
		if (traceLevel == 0)
			CAFContext.getMonitor(user, request).endComponent(
				ConfMonitor.COMPLEVELBASIC,
				methodName);
		else if (traceLevel == 1)
			CAFContext.getMonitor(user, request).endComponent(
				ConfMonitor.COMPLEVELMEDIUM,
				methodName);
		else
			CAFContext.getMonitor(user, request).endComponent(
				ConfMonitor.COMPLEVELDETAIL,
				methodName);
		mon.endRequest(request);
	}

	/**
	 * @param user
	 * @param request
	 * @param methodName
	 * @param loc
	 * @param res
	 * @param traceLevel
	 */
	public static void exiting(
		String user,
		String request,
		String methodName,
		Location loc,
		Object res,
		int traceLevel) {
		loc.exiting(res);
		IMonitor mon = CAFContext.getMonitor(user, request);
		if (traceLevel == 0)
			CAFContext.getMonitor(user, request).endComponent(
				ConfMonitor.COMPLEVELBASIC,
				methodName);
		else if (traceLevel == 1)
			CAFContext.getMonitor(user, request).endComponent(
				ConfMonitor.COMPLEVELMEDIUM,
				methodName);
		else
			CAFContext.getMonitor(user, request).endComponent(
				ConfMonitor.COMPLEVELDETAIL,
				methodName);
		mon.endRequest(request);
	}

	/**
	 * @param severity
	 * @param method
	 * @param throwable
	 *
	 * @deprecated use constructor of com.sap.tc.logging.Location.traceThrowableT
	 */
	public static void traceThrowable(
		int severity,
		String method,
		Throwable throwable) {
		StringWriter writer = new StringWriter();
		PrintWriter wrapper = new PrintWriter(writer);
		throwable.printStackTrace(wrapper);

		LOC_CAF.logT(severity, method, throwable.toString());
		LOC_CAF.logT(severity, method, writer.toString());
	}

	/**
	 * @param severity
	 * @param location
	 * @param method
	 * @param throwable
	 *
	 * @deprecated use constructor of com.sap.tc.logging.Location.traceThrowableT
	 **/
	public static void traceThrowable(
		int severity,
		Location location,
		String method,
		Throwable throwable) {
		StringWriter writer = new StringWriter();
		PrintWriter wrapper = new PrintWriter(writer);
		throwable.printStackTrace(wrapper);

		location.logT(severity, method, throwable.toString());
		location.logT(severity, method, writer.toString());

	}

	public static void traceThrowableT(
		int severity,
		Location location,
		String method,
		String message,
		Object[] params,
		Throwable throwable) {
		location.traceThrowableT(severity, method, message, params, throwable);
	}

	public static void traceThrowableT(
		int severity,
		Location location,
		String method,
		String message,
		Throwable throwable) {
		location.traceThrowableT(severity, method, message, throwable);
	}

	/**
	 * @param severity
	 * @param category
	 * @param method
	 * @param throwable
	 * 
	 * @deprecated use constructor of com.sap.tc.logging.Category.logThrowableT
	 */
	public static void logThrowable(
		int severity,
		Category category,
		String method,
		Throwable throwable) {
		StringWriter writer = new StringWriter();
		PrintWriter wrapper = new PrintWriter(writer);
		throwable.printStackTrace(wrapper);

		category.logT(severity, LOC_CAF, method, throwable.getMessage());
		LOC_CAF.logT(severity, method, throwable.toString());
		LOC_CAF.logT(severity, method, writer.toString());
	}

	/**
	 * @param severity
	 * @param category
	 * @param location
	 * @param method
	 * @param throwable
	 * 
	 * @deprecated use constructor of com.sap.tc.logging.Category.logThrowableT
	 */
	public static void logThrowable(
		int severity,
		Category category,
		Location location,
		String method,
		Throwable throwable) {
		StringWriter writer = new StringWriter();
		PrintWriter wrapper = new PrintWriter(writer);
		throwable.printStackTrace(wrapper);

		category.logT(severity, location, method, throwable.getMessage());
		location.logT(severity, method, throwable.toString());
		location.logT(severity, method, writer.toString());
	}

	/**
	 * @param severity
	 * @param category
	 * @param method
	 * @param message
	 * @param throwable
	 * 	
	 * @deprecated use constructor of com.sap.tc.logging.Category.logThrowableT
	 */
	public static void logThrowable(
		int severity,
		Category category,
		String method,
		String message,
		Throwable throwable) {
		StringWriter writer = new StringWriter();
		PrintWriter wrapper = new PrintWriter(writer);
		throwable.printStackTrace(wrapper);

		category.logT(
			severity,
			LOC_CAF,
			method,
			message + " " + throwable.getMessage());
		LOC_CAF.logT(severity, method, throwable.toString());
		LOC_CAF.logT(severity, method, writer.toString());
	}

	/**
	 * @param severity
	 * @param category
	 * @param location
	 * @param method
	 * @param message
	 * @param throwable
	 * 
	 * @deprecated use constructor of com.sap.tc.logging.Category.logThrowableT
	 */
	public static void logThrowable(
		int severity,
		Category category,
		Location location,
		String method,
		String message,
		Throwable throwable) {
		StringWriter writer = new StringWriter();
		PrintWriter wrapper = new PrintWriter(writer);
		throwable.printStackTrace(wrapper);

		category.logT(
			severity,
			location,
			method,
			message + " " + throwable.getMessage());
		location.logT(severity, method, throwable.toString());
		location.logT(severity, method, writer.toString());
	}

}
