/*
 * Created on 08.04.2004
 *
 */
package com.sap.caf.rt.ui.cool.metadata;

import com.sap.tc.cmi.metadata.CMICardinality;
import com.sap.tc.col.client.metadata.api.*;
import java.util.HashMap;
import java.util.Set;

/**
 * @author Sergey_Firsov
 * Describes the properties of a relation between two aspects
 *
 */
public class RelationDescriptor extends AbstractDescriptor
    implements IRelationDescriptor{
    private static final CMICardinality sCardinalities[];
    private static HashMap sCardinalityMap;
    static 
    {
        sCardinalities = (new CMICardinality[] {
            null, CMICardinality.CARDINALITY_ZERO_TO_ONE, CMICardinality.CARDINALITY_ONE, CMICardinality.CARDINALITY_MANY, CMICardinality.CARDINALITY_ONE_TO_MANY
        });
        sCardinalityMap = new HashMap(sCardinalities.length - 1);
        for(int i = 1; i < sCardinalities.length; i++)
            sCardinalityMap.put(sCardinalities[i], new Integer(i));

    }

    private HashMap mAttributes;
    private HashMap mRowAttributes;
    private AspectDescriptor targetAspectDescriptor;
    private AspectDescriptor sourceAspectDescriptor;
    private ServiceModuleDescriptor serviceModuleDescriptor;

    /**
	 * empty constructor
	 */
	 public RelationDescriptor(){
        super();
    }

    /**
	 * @param name - name of RelationDescriptor
	 * @param sourceAspectDescriptor - source RelationDescriptor
	 * @param targetAspectDescriptor - target RelationDescriptor
	 */
	 public RelationDescriptor(String name, AspectDescriptor sourceAspectDescriptor, AspectDescriptor targetAspectDescriptor)
    {
        super(name);
        mAttributes = new HashMap();
        mRowAttributes = new HashMap();
        if(targetAspectDescriptor == null)
            throw new RuntimeException("targetAspectDescriptor must not be null");
        this.targetAspectDescriptor = targetAspectDescriptor;
        if(sourceAspectDescriptor == null)
        {
            throw new RuntimeException("sourceAspectDescriptor must not be null");
        } else
        {
            this.sourceAspectDescriptor = sourceAspectDescriptor;
            sourceAspectDescriptor.addRelationDescriptor(this);
            return;
        }
    }

    /**
	 * @param name - name of RelationDescriptor
	 * @param cardinality - cardinality of RelationDescriptor
	 * @param sourceAspectDescriptor - source RelationDescriptor
	 * @param targetAspectDescriptor - target RelationDescriptor
	 */
	public RelationDescriptor(String name, CMICardinality cardinality, AspectDescriptor sourceAspectDescriptor, AspectDescriptor targetAspectDescriptor)
    {
        this(name, sourceAspectDescriptor, targetAspectDescriptor);
        setAttributeTargetCardinality(cardinality);
    }

    /**
	 * Returns the descriptor of the source aspect of this relation
	 */
	public IAspectDescriptor getSourceAspectDescriptor()
    {
        return sourceAspectDescriptor;
    }

    /**
	 * Returns the descriptor of the target aspect of this relation
	 */
	public IAspectDescriptor getTargetAspectDescriptor()
    {
        return targetAspectDescriptor;
    }

    /**
	 * Returns the descriptor of the service module in which this 
	 * relation is cached.
	 */
	public IServiceModuleDescriptor getServiceModuleDescriptor()
    {
        return serviceModuleDescriptor;
    }

    /**
    * Set specified serviceModuleDescriptor
	 * @param serviceModuleDescriptor
	 */
	public void setServiceModuleDescriptor(ServiceModuleDescriptor serviceModuleDescriptor)
    {
        this.serviceModuleDescriptor = serviceModuleDescriptor;
    }

    /**
    * add attribute value to the specified attribute name
	 * @param attributeName
	 * @param value
	 */
	public void addAttribute(String attributeName, String value)
    {
        mAttributes.put(attributeName, value);
    }

    /**
	 * Returns the value that is stored in this relation for the given attribute
	 */
	public String getAttributeStringValue(String attributeName)
    {
        return (String)mAttributes.get(attributeName);
    }

    /**
	 * Returns an array with the names of all the attributes which are set for this relation
	 */
	public String[] getAttributeNames()
    {
        String attributes[] = new String[mAttributes.size()];
        mAttributes.keySet().toArray(attributes);
        return attributes;
    }

    /**
	 * Returns an array with the names of all the attributes which 
	 * are set for this relation in the specified source aspect row.
	 */
	public String[] getAttributeNames(String rowKey)
    {
        Set attributeSet = mAttributes.keySet();
        HashMap rowMap = (HashMap)mRowAttributes.get(rowKey);
        if(rowMap != null)
        {
            Set dynamicAttributes = rowMap.keySet();
            attributeSet.addAll(dynamicAttributes);
        }
        String attributes[] = new String[attributeSet.size()];
        attributeSet.toArray(attributes);
        return attributes;
    }

    /**
	 * Returns the value that is stored in this relation for the given 
	 * attribute in the specified source aspect row.
	 */
	public String getAttributeStringValue(String rowKey, String attributeName)
    {
        HashMap rowMap = (HashMap)mRowAttributes.get(rowKey);
        if(rowMap != null && rowMap.containsKey(attributeName))
            return (String)rowMap.get(attributeName);
        else
            return getAttributeStringValue(attributeName);
    }

    /**
    * Clear Attributes for specified row
	 * @param rowKey
	 */
	void clearRowAttributes(String rowKey)
    {
        mRowAttributes.remove(rowKey);
    }

    /**
	 * Clear Attributes
	 */
	void clearRowAttributes()
    {
        mRowAttributes.clear();
    }

    /**
    * add attribute value to the specified attribute name for specified row
	 * @param rowKey
	 * @param attributeName
	 * @param value
	 */
	public void addAttribute(String rowKey, String attributeName, String value)
    {
        HashMap rowMap = (HashMap)mRowAttributes.get(rowKey);
        if(rowMap == null)
        {
            rowMap = new HashMap();
            mRowAttributes.put(rowKey, rowMap);
        }
        rowMap.put(attributeName, value);
    }

    /**
	 * add attribute value to the specified attribute name
	 */
	public String getAttributePagingMode()
    {
        String value = getAttributeStringValue("COL_PAGING_MODE");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_PAGING_MODE' is not provided by the backend!");
        switch((new Integer(value)).intValue())
        {
        case 2: // '\002'
            return "FORWARD";

        case 1: // '\001'
        default:
            return " ";
        }
    }

    /**
	 * Returns the cardinality of the target
	 */
	public CMICardinality getAttributeTargetCardinality()
    {
        String value = getAttributeStringValue("COL_TARGET_CARDINALITY");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_TARGET_CARDINALITY' is not provided for the relation '" + getName() + "' by the backend!");
        int cardvalue = (new Integer(value)).intValue();
        if(cardvalue >= 0 && cardvalue < sCardinalities.length)
            return sCardinalities[cardvalue];
        else
            throw new IllegalArgumentException("Illegal cardinality value " + cardvalue);
    }

    /**
    * set the cardinality of the target
	 * @param cardinality
	 */
	private void setAttributeTargetCardinality(CMICardinality cardinality)
    {
        addAttribute("COL_TARGET_CARDINALITY", ((Integer)sCardinalityMap.get(cardinality)).toString());
    }

    /**
	 * Returns the input options structure for this relation
	 */
	public IStructureDescriptor getInputOptions()
    {
        return serviceModuleDescriptor.getInputOptionsStructure();
    }

    /**
	 * Returns the sorting options structure for this relation
	 */
	public IStructureDescriptor getSortingOptions()
    {
        return serviceModuleDescriptor.getSortingOptionsStructure();
    }

    /**
	 * Returns the selections structure for this relation
	 */
	public IStructureDescriptor getSelections()
    {
        return serviceModuleDescriptor.getSelectionsStructure();
    }

}
