/*
 * Created on 08.04.2004
 *
 */
package com.sap.caf.rt.ui.cool.metadata;

import com.sap.tc.col.client.metadata.api.*;
import java.util.*;

/**
 * @author Sergey_Firsov
 * Describes the properties of an aspect
 *
 */
public class AspectDescriptor extends AbstractDescriptor
    implements IAspectDescriptor {
    private ServiceModuleDescriptor mServiceModuleDescriptor;
    private HashMap mAttributes;
    private HashMap mFieldAttributes;
    private HashMap mRowFieldAttributes;
    private HashMap mActionAttributes;
    private HashMap mRowActionAttributes;
    private HashMap mValueSets;
    private String description;
    private KeyAspectDescriptor keyDescriptor;
    private StructureDescriptor structure;
    private HashMap supportedActions;
    private HashMap mAspectGroups;
    private HashMap relations;

    /**
	 * empty constructor 
	 */
	public AspectDescriptor(){
	super();
    }

    /**
	 * @param serviceModuleDesc - ServiceModuleDescriptor for this AspectDescriptor
	 * @param name - name of AspectDescriptor
	 * @param description - description of AspectDescriptor
	 * @param structure - StructureDescriptor for this AspectDescriptor
	 * @param key - KeyAspectDescriptor for this AspectDescriptor
	 */
	public AspectDescriptor(ServiceModuleDescriptor serviceModuleDesc, String name, String description, StructureDescriptor structure, KeyAspectDescriptor key) {
        super(name);
        mAttributes = new HashMap();
        mFieldAttributes = new HashMap();
        mRowFieldAttributes = new HashMap();
        mActionAttributes = new HashMap();
        mRowActionAttributes = new HashMap();
        mValueSets = new HashMap();
        supportedActions = new HashMap();
        mAspectGroups = new HashMap();
        relations = new HashMap();
        setServiceModuleDescriptor(serviceModuleDesc);
        this.description = description;
        this.structure = structure;
        keyDescriptor = key;
    }

    /** 
	 * Returns the description of this aspect
	 */
	public String getDescription()
    {
        return description;
    }

    /**
     * add given relationDescriptor
	 * @param relationDescriptor
	 */
	public void addRelationDescriptor(RelationDescriptor relationDescriptor)
    {
        relations.put(relationDescriptor.getName(), relationDescriptor);
        if(mServiceModuleDescriptor != null && mServiceModuleDescriptor.getRelationDescriptor(relationDescriptor.getName()) != relationDescriptor)
            mServiceModuleDescriptor.addRelationDescriptor(relationDescriptor);
    }

    /**
	 * Returns the descriptor for the relation with the specified name.
	 */
	public IRelationDescriptor getRelationDescriptor(String name)
    {
        return (IRelationDescriptor)relations.get(name);
    }

    /**
	 * Returns an array with descriptors of all relations for which this aspect is the source aspect.
	 */
	public IRelationDescriptor[] getRelationDescriptors()
    {
        IRelationDescriptor relationDescriptors[] = new RelationDescriptor[relations.size()];
        relations.values().toArray(relationDescriptors);
        return relationDescriptors;
    }

    /**
    * add given aspectActionDescriptor
	 * @param aspectActionDescriptor
	 */
	public void addAspectActionDescriptor(AspectActionDescriptor aspectActionDescriptor)
    {
        if(aspectActionDescriptor != null)
            supportedActions.put(aspectActionDescriptor.getName(), aspectActionDescriptor);
    }

    /**
	 * Returns the descriptor of the aspect action with the specified name
	 */
	public IAspectActionDescriptor getAspectActionDescriptor(String name)
    {
        return (AspectActionDescriptor)supportedActions.get(name);
    }

    /**
	 * Returns an array with descriptors of all aspect actions
	 */
	public IAspectActionDescriptor[] getAspectActionDescriptors()
    {
        IAspectActionDescriptor actionDescriptors[] = new AspectActionDescriptor[supportedActions.size()];
        supportedActions.values().toArray(actionDescriptors);
        return actionDescriptors;
    }

    /**
    *  add given aspectGroupDescriptor
	 * @param anAspectGroupDescriptor
	 */
	public void addAspectGroupDescriptor(AspectGroupDescriptor anAspectGroupDescriptor)
    {
        if(anAspectGroupDescriptor != null)
            mAspectGroups.put(anAspectGroupDescriptor.getName(), anAspectGroupDescriptor);
    }

    /**
	 * Returns the descriptor of the aspect group with the specified name
	 */
	public IAspectGroupDescriptor getAspectGroupDescriptor(String name)
    {
        return (AspectGroupDescriptor)mAspectGroups.get(name);
    }

    /**
	 * Returns an array with descriptors of all aspect groups
	 */
	public IAspectGroupDescriptor[] getAspectGroupDescriptors()
    {
        IAspectGroupDescriptor groupDescriptors[] = new AspectGroupDescriptor[mAspectGroups.size()];
        mAspectGroups.values().toArray(groupDescriptors);
        return groupDescriptors;
    }

    /**
    * add attribute value to the specified attribute name
	 * @param attributeName
	 * @param value
	 */
	 public void addAttribute(String attributeName, String value)
    {
        mAttributes.put(attributeName, value);
    }

    /**
	 * Returns the value that is stored in the aspect for the given attribute
	 */
	 public String getAttributeStringValue(String attributeName)
    {
        return (String)mAttributes.get(attributeName);
    }

    /**
	 * Returns an array with the names of all the attributes which are set for this aspect
	 */
	 public String[] getAttributeNames()
    {
        String attributes[] = new String[mAttributes.size()];
        mAttributes.keySet().toArray(attributes);
        return attributes;
    }

    /**
	 * Returns whether Update/Insert/Delete or Action without COL_AFFECTS_NOTHING attribute being set is supported in this aspect
	 */
	 public boolean getAttributeNoChange()
    {
        String value = getAttributeStringValue("COL_NO_CHANGE");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_NO_CHANGE' is not provided by the backend!");
        else
            return value.equalsIgnoreCase("X");
    }

    /**
	 * Returns whether externally provided keys can be provided at newly created objects
	 */
	 public boolean getAttributeExternalKeys()
    {
        String value = getAttributeStringValue("COL_EXTERNAL_KEYS");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_EXTERNAL_KEYS' is not provided by the backend!");
        else
            return value.equalsIgnoreCase("X");
    }

    /**
	 * Returns whether field wise operations are available in this aspect
	 */
	 public boolean getAttributeFieldwiseOperation()
    {
        String value = getAttributeStringValue("COL_FIELDWISE_OP");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_FIELDWISE_OP' is not provided by the backend!");
        else
            return value.equalsIgnoreCase("X");
    }

    /**
	 *  Returns whether the exclusive lock mode is supported
	 */
	 public boolean getAttributeLockmodeExclusive()
    {
        String value = getAttributeStringValue("COL_LOCKMODE_E");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_LOCKMODE_E' is not provided by the backend!");
        else
            return value.equalsIgnoreCase("X");
    }

    /**
	 *  Returns whether the shared promotable lock mode is supported
	 */
	public boolean getAttributeLockmodeSharedPromotable()
    {
        String value = getAttributeStringValue("COL_LOCKMODE_SP");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_LOCKMODE_SP' is not provided by the backend!");
        else
            return value.equalsIgnoreCase("X");
    }

    /**
	 * Returns whether the shared lock mode is supported
	 */
	public boolean getAttributeLockmodeShared()
    {
        String value = getAttributeStringValue("COL_LOCKMODE_S");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_LOCKMODE_S' is not provided by the backend!");
        else
            return value.equalsIgnoreCase("X");
    }

    /**
	 * Returns whether Update/Insert/Actions can act on multiple instances
	 */
	public boolean getAttributeMultiInstanceSupport()
    {
        String value = getAttributeStringValue("COL_MULTI_INSTANCE_SUPPORT");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_MULTI_INSTANCE_SUPPORT' is not provided by the backend!");
        else
            return value.equalsIgnoreCase("X");
    }

    /**
	 * Returns whether row wise operations are available in this aspect
	 */
	public boolean getAttributeRowwiseOperation()
    {
        String value = getAttributeStringValue("COL_ROWWISE_OP");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_ROWWISE_OP' is not provided by the backend!");
        else
            return value.equalsIgnoreCase("X");
    }

    /**
	 * Returns the name of the relation to the aspect of which a record 
	 * can be specified as related to records which are inserted in this aspect
	 */
	public String getAttributeInsertRelation()
    {
        String value = getAttributeStringValue("COL_INSERT_RELATION");
        if(value == null)
            value = "";
        return value;
    }

    /**
	 * Returns whether the insert operation of this aspect requires the 
	 * object to which the inserted object(s) are related to be specified
	 */
	public boolean getAttributeInsertRelationRequired()
    {
        String value = getAttributeStringValue("COL_INSERT_RELATION_REQUIRED");
        if(value == null)
            value = " ";
        return value.equalsIgnoreCase("X");
    }

    /**
    * set attribute value to the specified attribute name for the 
    * specified field name
	 * @param fieldName
	 * @param attributeName
	 * @param value
	 */
	public void addFieldAttribute(String fieldName, String attributeName, String value)
    {
        addNestedAttribute(mFieldAttributes, fieldName, attributeName, value);
    }

    /**
	 * Returns the value that is stored in the aspect for the given 
	 * attribute of the given field.
	 */
	public String getFieldAttributeStringValue(String fieldName, String attributeName)
    {
        return getNestedAttributeStringValue(mFieldAttributes, fieldName, attributeName);
    }

    /**
	 * Returns an array with the names of all the attributes for a specified field
	 */
	public String[] getFieldAttributeNames(String fieldName)
    {
        if(getStructure().getFieldDescriptor(fieldName) == null)
        {
            throw new IllegalArgumentException("No such field '" + fieldName + "' in aspect '" + getName() + "'");
        } else
        {
            Set attributeSet = getNestedAttributeNames(mFieldAttributes, fieldName);
            String attributes[] = new String[attributeSet.size()];
            attributeSet.toArray(attributes);
            return attributes;
        }
    }

    /**
	 * Returns an array with the names of all the attributes which are 
	 * set for a structure field of given aspect row.
	 */
	public String[] getFieldAttributeNames(String rowKey, String fieldName)
    {
        Set attributeSet = getNestedAttributeNames(mFieldAttributes, fieldName);
        HashMap rowMap = (HashMap)mRowFieldAttributes.get(rowKey);
        if(rowMap != null)
        {
            Set dynamicAttributes = getNestedAttributeNames(rowMap, fieldName);
            attributeSet.addAll(dynamicAttributes);
        }
        String attributes[] = new String[attributeSet.size()];
        attributeSet.toArray(attributes);
        return attributes;
    }

    /**
	 * Returns the value that is stored for the given attribute for the given aspect row
	 */
	public String getFieldAttributeStringValue(String rowKey, String fieldName, String attributeName)
    {
        HashMap rowMap = (HashMap)mRowFieldAttributes.get(rowKey);
        if(rowMap != null)
        {
            String value = getNestedAttributeStringValue(rowMap, fieldName, attributeName);
            if(value != null)
                return value;
        }
        return getNestedAttributeStringValue(mFieldAttributes, fieldName, attributeName);
    }

    /**
	 * Clears all dynamically set attribute values for the given aspect row. 
	 */
	public void clearRowAttributes(String rowKey)
    {
        mRowFieldAttributes.remove(rowKey);
        mRowActionAttributes.remove(rowKey);
        RelationDescriptor rel;
        for(Iterator relIt = relations.values().iterator(); relIt.hasNext(); rel.clearRowAttributes(rowKey))
            rel = (RelationDescriptor)relIt.next();

    }

    /**
	 * Clears all dynamically set attribute values that are currently 
	 * cached in this aspect descriptor for any rows.
	 */
	public void clearRowAttributes()
    {
        mRowFieldAttributes.clear();
        mRowActionAttributes.clear();
        RelationDescriptor rel;
        for(Iterator relIt = relations.values().iterator(); relIt.hasNext(); rel.clearRowAttributes())
            rel = (RelationDescriptor)relIt.next();

    }

    /**
    * set attribute value to the specified attribute name for the 
    * specified field name for the given aspect row
	 * @param rowKey
	 * @param fieldName
	 * @param attributeName
	 * @param value
	 */
	public void addFieldAttribute(String rowKey, String fieldName, String attributeName, String value)
    {
        HashMap rowMap = (HashMap)mRowFieldAttributes.get(rowKey);
        if(rowMap == null)
        {
            rowMap = new HashMap();
            mRowFieldAttributes.put(rowKey, rowMap);
        }
        addNestedAttribute(rowMap, fieldName, attributeName, value);
    }

    /**
	 * @param map
	 * @param keyName
	 * @param attributeName
	 * @return
	 */
	private String getNestedAttributeStringValue(HashMap map, String keyName, String attributeName)
    {
        HashMap innerMap = (HashMap)map.get(keyName);
        if(innerMap != null)
            return (String)innerMap.get(attributeName);
        else
            return null;
    }

    /**
	 * @param map
	 * @param keyName
	 * @return
	 */
	private Set getNestedAttributeNames(HashMap map, String keyName)
    {
        HashMap innerMap = (HashMap)map.get(keyName);
        if(innerMap != null)
            return innerMap.keySet();
        else
            return new HashSet();
    }

    /**
	 * @param map
	 * @param keyName
	 * @param attributeName
	 * @param value
	 */
	private void addNestedAttribute(HashMap map, String keyName, String attributeName, String value)
    {
        HashMap innerMap = (HashMap)map.get(keyName);
        if(innerMap == null)
        {
            innerMap = new HashMap();
            map.put(keyName, innerMap);
        }
        innerMap.put(attributeName, value);
    }

    /**
	 * Returns whether this aspect is a key aspect
	 */
	public boolean isKeyAspect()
    {
        return false;
    }

    /**
	 * Returns the structure desciptor for aspect rows of this aspect.
	 */
	public IStructureDescriptor getStructure()
    {
        return structure;
    }

    /**
	 * Returns the service module descriptor of the service module for 
	 * which this aspect is cached in the repository.
	 */
	public IServiceModuleDescriptor getServiceModuleDescriptor()
    {
        return mServiceModuleDescriptor;
    }

    /**
    * Set the service module descriptor 
	 * @param descriptor
	 */
	private void setServiceModuleDescriptor(ServiceModuleDescriptor descriptor)
    {
        if(descriptor != null)
        {
            mServiceModuleDescriptor = descriptor;
            mServiceModuleDescriptor.addAspectDescriptor(this);
            RelationDescriptor rel;
            for(Iterator it = relations.values().iterator(); it.hasNext(); mServiceModuleDescriptor.addRelationDescriptor(rel))
                rel = (RelationDescriptor)it.next();

        }
    }

    /**
	 * Returns the key aspect descriptor for this aspect
	 */
	public IKeyAspectDescriptor getKeyDescriptor()
    {
        return keyDescriptor;
    }

    /**
    * Set the key descriptor
	 * @param keyDescriptor
	 */
	public void setKeyDescriptor(KeyAspectDescriptor keyDescriptor)
    {
        this.keyDescriptor = keyDescriptor;
    }

    /**
	 * returns a string representation of this AspectDescriptor 
	 */
	public String toString()
    {
        String nl = System.getProperty("line.separator");
        StringBuffer buf = new StringBuffer();
        buf.append("  <AspectDescriptor name=\"").append(getName()).append("\"");
        buf.append(" key=\"").append(keyDescriptor.getName()).append("\">").append(nl);
        buf.append(structure.toString());
        buf.append("  </AspectDescriptor>").append(nl);
        return buf.toString();
    }

    /**
	 * Returns which side effects are to be excpected when performing 
	 * the given aspect action on the given aspect row.
	 */
	public int getActionAttributeAffects(String rowKey, String actionName)
    {
        String value = getActionAttributeStringValue(rowKey, actionName, "COL_AFFECTS");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_AFFECTS' for the action '" + actionName + "' is not provided by the backend!");
        else
            return (new Integer(value)).intValue();
    }

    /**
	 * Returns which side effects are to be excpected when performing the 
	 * given aspect action
	 */
	public int getActionAttributeAffects(String actionName)
    {
        String value = getActionAttributeStringValue(actionName, "COL_AFFECTS");
        if(value == null)
            throw new UnsupportedOperationException("The system property 'COL_AFFECTS' for the action '" + actionName + "' is not provided by the backend!");
        else
            return (new Integer(value)).intValue();
    }

    /**
	 * Returns the name of the relation whose target aspect the action 
	 * can affect for the given aspect action.
	 */
	public String getActionAttributeRelation(String actionName)
    {
        String value = getActionAttributeStringValue(actionName, "COL_ACTION_RELATION");
        return value != null ? value : "";
    }

    /**
	 * Returns an array with the names of all the attributes which are 
	 * set for an action of the given aspect row.
	 */
	public String[] getActionAttributeNames(String rowKey, String actionName)
    {
        Set attributeSet = getNestedAttributeNames(mActionAttributes, actionName);
        HashMap rowMap = (HashMap)mRowActionAttributes.get(rowKey);
        if(rowMap != null)
        {
            Set dynamicAttributes = getNestedAttributeNames(rowMap, actionName);
            attributeSet.addAll(dynamicAttributes);
        }
        String attributes[] = new String[attributeSet.size()];
        attributeSet.toArray(attributes);
        return attributes;
    }

    /**
	 * Returns an array with the names of all the attributes for a specified action
	 */
	public String[] getActionAttributeNames(String actionName)
    {
        Set attributeSet = getNestedAttributeNames(mActionAttributes, actionName);
        String attributes[] = new String[attributeSet.size()];
        attributeSet.toArray(attributes);
        return attributes;
    }

    /**
	 * Returns the value that is stored for the given attribute for the given aspect row
	 */
	public String getActionAttributeStringValue(String rowKey, String actionName, String attributeName)
    {
        HashMap rowMap = (HashMap)mRowActionAttributes.get(rowKey);
        if(rowMap != null)
        {
            String value = getNestedAttributeStringValue(rowMap, actionName, attributeName);
            if(value != null)
                return value;
        }
        return getNestedAttributeStringValue(mActionAttributes, actionName, attributeName);
    }

    /**
	 * Returns the value that is stored in the aspect for the given attribute of the given action
	 */
	public String getActionAttributeStringValue(String actionName, String attributeName)
    {
        return getNestedAttributeStringValue(mActionAttributes, actionName, attributeName);
    }

    /**
    * set actoin attribute for given action name
	 * @param actionName
	 * @param attributeName
	 * @param value
	 */
	public void addActionAttribute(String actionName, String attributeName, String value)
    {
        addNestedAttribute(mActionAttributes, actionName, attributeName, value);
    }

    /**
    * set actoin attribute for given action name and given aspect row
	 * @param rowKey
	 * @param actionName
	 * @param attributeName
	 * @param value
	 */
	public void addActionAttribute(String rowKey, String actionName, String attributeName, String value)
    {
        HashMap rowMap = (HashMap)mRowActionAttributes.get(rowKey);
        if(rowMap == null)
        {
            rowMap = new HashMap();
            mRowActionAttributes.put(rowKey, rowMap);
        }
        addNestedAttribute(rowMap, actionName, attributeName, value);
    }

    /**
	 * Returns the value set descriptor for the given field of the aspect structure
	 */
	public IValueSetDescriptor getValueSetDescriptor(String fieldName)
    {
        return (IValueSetDescriptor)mValueSets.get(fieldName);
    }

    /**
    * Set the value set descriptor with given fieldNmae
	 * @param fieldName
	 * @param valueSetDescriptor
	 */
	public void setValueSetDescriptor(String fieldName, IValueSetDescriptor valueSetDescriptor)
    {
        mValueSets.put(fieldName, valueSetDescriptor);
    }

}
