/*
 * Created on 14.04.2004
 */
package com.sap.caf.rt.services.serviceaccess;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Locale;

import javax.jdo.JDOException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.RollbackException;
import javax.transaction.Status;
import javax.transaction.UserTransaction;

import com.sap.caf.rt.exception.DataAccessException;
import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.srv.IDataContainerBean;
import com.sap.caf.rt.ui.cool.metadata.ServiceModuleDescriptor;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * 
 */
public class CAFServiceAccessBeanImpl implements ICAFServiceAccess {
	private final static String JARM_REQUEST = "CAF:RT:oal";
	private static final String APPLICATION = CAFServiceAccessBean.class.getName();
	private static final Location logger = Location.getLocation(APPLICATION);
	private static final String jARMRequest = JARM_REQUEST+APPLICATION;
	
	private MetamodelHelper mmHelper;
	private Map serviceWrappers;
	private Locale mLocale;
	private static int activeTransactionsCount;

	/**
	 * 
	 */
	public CAFServiceAccessBeanImpl() {
		super();
		mLocale = Locale.getDefault();
	}

//-----------  Implementation of IServiceAccess
	/*
	 * @see com.sap.caf.rt.srv.accessservice.IServiceAccess#readDataObject(java.lang.String, java.lang.String, java.util.Collection)
	 */
	public Collection readDataObject(
		String serviceName,
		String aspectName,
		Collection keys) throws ServiceException {
			final String method = "readDataObject(String, String, Collection)";
			CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);
		
			try{
				startTransaction();
				return getServiceWrapper(serviceName).readDataObject(aspectName, keys);
			}catch(Exception e){
				try {
					rollback();
				} catch (ServiceException th) {
					CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								logger,
								method,
								"Error in " + method,
								th);
				}
				throw handleException(e);
			}
			finally {
				CAFPublicLogger.exiting(null, jARMRequest, method, logger, 1);
			}
	}

	/*
	 * @see com.sap.caf.rt.srv.accessservice.IServiceAccess#readRelatedDataObject(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public IDataContainerBean readRelatedDataObject(
		String serviceName,
		String aspectName,
		String parentServiceName,
		String parentAspect,
		String parentAspectKey) throws ServiceException {
			//return getServiceWrapper(serviceName).readRelatedDataObject(aspectName,serviceName,)
			final String method = "readRelatedDataObject(String, String, String, String, String)";
			CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);
		
			try{			
				return null;
			}catch(Exception e){
				try {
					rollback();
				} catch (ServiceException th) {
					CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								logger,
								method,
								"Error in " + method,
								th);
				}
				throw handleException(e);
			}
			finally {
				CAFPublicLogger.exiting(null, jARMRequest, method, logger, 1);
			}
			
	}

	/**
	 * @see com.sap.caf.rt.srv.accessservice.IServiceAccess#updateDataObject(java.lang.String, java.lang.String, com.sap.caf.rt.srv.IDataContainerBean)
	 */
	public IDataContainerBean updateDataObject (
		String serviceName,
		String aspectName,
		IDataContainerBean data)throws ServiceException {
			final String method = "updateDataObject(String, String, IDataContainerBean)";
			CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);
			
			try{
				startTransaction();
				return getServiceWrapper(serviceName).updateDataObject(aspectName, data);
			}catch(Exception e){
				try {
					setRollbackOnly();
				} catch (ServiceException th) {
					CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								logger,
								method,
								"Error in " + method,
								th);
				}
				throw handleException(e);
			}
			finally {
				CAFPublicLogger.exiting(null, jARMRequest, method, logger, 1);
			}

	}

	/**
	 * @see com.sap.caf.rt.srv.accessservice.IServiceAccess#createDataObject(java.lang.String, java.lang.String, com.sap.caf.rt.srv.IDataContainerBean)
	 */
	public IDataContainerBean createDataObject (
		String serviceName,
		String aspectName,
		IDataContainerBean data) throws ServiceException{
			final String method = "createDataObject(String, String, IDataContainerBean)";
			CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);
		
			try{
				startTransaction();			
				return getServiceWrapper(serviceName).createDataContainerBean(aspectName, data);
			}catch(Exception e){
				try {
					setRollbackOnly();
				} catch (ServiceException th) {
					CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								logger,
								method,
								"Error in " + method,
								th);
				}				
				throw handleException(e);
			}
			finally {
				CAFPublicLogger.exiting(null, jARMRequest, method, logger, 1);
			}

	}

	/*
	 * @see com.sap.caf.rt.srv.accessservice.IServiceAccess#deleteDataObject(java.lang.String, java.lang.String, java.util.Collection)
	 */
	public boolean deleteDataObject(
		String serviceName,
		String aspectName,
		Collection keys) throws ServiceException {
			final String method = "deleteDataObject(String, String, Collection)";
			CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);
		
			try{
				startTransaction();
				return getServiceWrapper(serviceName).deleteDataObject(aspectName, keys);
			}catch(Exception e){
				try {
					setRollbackOnly();
				} catch (ServiceException th) {
					CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								logger,
								method,
								"Error in " + method,
								th);
				}				
				throw handleException(e);
			}
			finally {
				CAFPublicLogger.exiting(null, jARMRequest, method, logger, 1);
			}

	}

	/**
	 * @see com.sap.caf.rt.srv.accessservice.IServiceAccess#findByDataObject(java.lang.String, java.lang.String, java.util.Collection)
	 */
	public Collection findByDataObject (
		String serviceName,
		String queryName,
		Collection parameters) throws ServiceException{
			final String method = "findByDataObject(String, String, Collection)";
			CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		
			try{
				startTransaction();
				return getServiceWrapper(serviceName).findByDataObject(queryName, parameters);
			}catch(Exception e){
				try {
					rollback();
				} catch (ServiceException th) {
					CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								logger,
								method,
								"Error in " + method,
								th);
				}				
				throw handleException(e);
			}
			finally {
				CAFPublicLogger.exiting(null, jARMRequest, method, logger, 1);
			}

	}

	/*
	 * @see com.sap.caf.rt.srv.accessservice.IServiceAccess#invokeCustom(java.lang.String, java.lang.String, java.lang.Object)
	 */
	public Object invokeCustom(
		String serviceName,
		String operationName,
		Object parameters,
		Collection keys) throws ServiceException {
			final String method = "invokeCustom(String, String, Collection)";
			CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
			try{
				// start transaction
				startTransaction();
				Object res = getServiceWrapper(serviceName).invokeCustom(operationName, parameters, keys);
				// commit after invoke action
				commit();
				return res;
			}catch(Exception e){
				try {
					rollback();
				} catch (ServiceException th) {
					CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								logger,
								method,
								"Error in " + method,
								th);
				}				
				throw handleException(e);
			}
			finally {
				CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
			}
	
		}

	/*
	 * @see com.sap.caf.rt.srv.accessservice.IServiceAccess#recalculate(java.lang.String, com.sap.caf.rt.srv.IDataContainerBean)
	 */
	public IDataContainerBean recalculate (
		String serviceName,
		IDataContainerBean data) throws ServiceException{
			final String method = "recalculate(String, IDataContainerBean)";
			CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		
			try{
				startTransaction();			
				return getServiceWrapper(serviceName).recalculate(data);
			}catch(Exception e){
				try {
					rollback();
				} catch (ServiceException th) {
					CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								logger,
								method,
								"Error in " + method,
								th);
				}				
				throw handleException(e);
			}
			finally {
				CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
			}
			
	}

//---------  Implementation of ITransactionHandler
	/**
	 * Commit transaction
	 * @see com.sap.caf.rt.srv.accessservice.ITransactionHandler#commit()
	 */
	public void commit() throws ServiceException{
		final String method = "commit()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		
		try {
			UserTransaction t = getTransaction();
			if (t.getStatus() == Status.STATUS_MARKED_ROLLBACK) {
				try {
					t.rollback();
				} finally {
					activeTransactionsCount--;
					logger.infoT("Rolled back. Active Transactions: " + activeTransactionsCount);
				}
			} else if (isTransactionAlive(t)) {
				try {
					t.commit();
				} catch (Exception ex) {
					if (isTransactionAlive(t)) {
						t.setRollbackOnly();
					}
					throw handleException(ex);
				} finally {
					activeTransactionsCount--;
					logger.infoT("Commited. Active Transactions: " + activeTransactionsCount);
				}
			}
		} catch (Exception e) {
			//$JL-EXC$
			throw handleException(e);
		}
		finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		}

	}

	/**
	 * Rollback transaction
	 */
	public void rollback()throws ServiceException {
		final String method = "rollback()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		
		try {
			UserTransaction t = getTransaction();
			if (isTransactionAlive(t)) {
				try {
					t.rollback();
				} finally {
					activeTransactionsCount--;
					logger.infoT("Rolled back. Active Transactions: " + activeTransactionsCount);
				}
			}
		} catch (Exception e) {
			//$JL-EXC$
			throw handleException(e);
		}
		finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		}
	}


	public void setRollbackOnly() throws ServiceException{
		try{
			UserTransaction t = getTransaction();
			if (isTransactionAlive(t)) {
				t.setRollbackOnly();
			}
		}catch(Exception e){
			handleException(e);
		}

	}

	/** 
	 * Start transaction 
	 */
	public void startTransaction() throws ServiceException{
		final String method = "startTransaction()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		
		try {
			UserTransaction t = getTransaction();
			// If there is no active transaction start new one.
			// Transactions that are marked for rollback only are treated as active ones.
			if (!isTransactionAlive(t)) {
				t.begin();
				activeTransactionsCount++;
				logger.infoT("Started. Active Transactions: " + activeTransactionsCount);
			}
		} catch (Exception e) {
			//$JL-EXC$
			throw handleException(e);
		}
		finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		}

	}

	/**
	 * Lookup UserTransaction and return it;
	 * @return UserTransaction
	 * @throws NamingException
	 */
	static UserTransaction getTransaction() throws NamingException {
		final String method = "getTransaction()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		
		InitialContext initContext = new InitialContext();
		CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		return (UserTransaction) initContext.lookup("java:comp/UserTransaction");
	}

	/**
	 * Return <code>true</code> if transaction alive
	 * @param tr
	 * @return
	 * @throws Exception
	 */
	static boolean isTransactionAlive(UserTransaction tr) throws Exception {
		// Transaction that is marked for roolback is still active one. So, it is excluded from the check.
		final String method = "isTransactionAlive(UserTransaction)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		
		int status = tr.getStatus();
		CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		return status != Status.STATUS_NO_TRANSACTION
			&& status != Status.STATUS_ROLLEDBACK
			&& status != Status.STATUS_UNKNOWN
			&& status != Status.STATUS_COMMITTED;
	}
	
	/**
	 * 
	 * @return
	 * @throws Exception
	 */
	public boolean isTransactionAlive() throws ServiceException {
		final String method = "isTransactionAlive()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		try {
			UserTransaction ut = getTransaction();
			return isTransactionAlive(ut);
		} catch (Exception e) {
			//$JL-EXC$
			throw handleException(e);
		} finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		}
	}

	/**
	 * Create instance of <code>ServiceException</code> by given exception
	 * @param th
	 * @return ServiceException
	 */
	private ServiceException handleException(Throwable th) {
		final String method = "handleException(Throwable)";
		if (th instanceof RollbackException) {
			Throwable t = th.getCause();
			// Workaround for entity bean ejbStore problem
			// Original exception is being lost in container
			// and there is no way to recover it
			if (t == null) {
				return new ServiceException("TRANSACTION_CANT_BE_COMMITED", null, th);
			} else {
				th = t;
				if(th instanceof JDOException) {
					th = new ServiceException("TRANSACTION_CANT_BE_COMMITED", null, th);
				}							
			}
		}
		CAFPublicLogger.traceThrowableT(
			Severity.DEBUG,
			logger,
			method,
			"Error in " + method,
			th);
		logger.throwing(method, th);
		return (th instanceof ServiceException) 
				? (ServiceException) th
				: new ServiceException(th);
	}

//--------------  Implementation of IServiceModelProvider
	/**
	 * @see com.sap.caf.rt.ui.cool.metadata.IServiceModelProvider#getServiceNames()
	 */
	public Collection getServiceNames() throws  ServiceException{
		return mmHelper.getServiceModuleNames();
	}

	/**
	 * @see com.sap.caf.rt.ui.cool.metadata.IServiceModelProvider#getServiceModuleDescriptor(java.lang.String)
	 */
	public ServiceModuleDescriptor getServiceModuleDescriptor(String serviceModuleName) throws  ServiceException{
		return mmHelper.getServiceModuleDescriptor(serviceModuleName);
	}

	public Locale getLocale(){
		return mLocale;
	}



	/**
	 * Return service wrapper class
	 * @param serviceModuleName identificator of service
	 * @return ServiceWrapper instance
	 */
	public ServiceWrapper getServiceWrapper(String serviceModuleName)throws ServiceException{
		final String method = "getServiceWrapper(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		
		ServiceWrapper wrapper = (ServiceWrapper)serviceWrappers.get(serviceModuleName);
		ServiceModuleDescriptor descriptor = mmHelper.getServiceModuleDescriptor(serviceModuleName);
		if((wrapper == null) && (descriptor != null)){
			wrapper = new ServiceWrapper(this, descriptor, mmHelper.getServiceJNDI_KEY(serviceModuleName));
			serviceWrappers.put(serviceModuleName, wrapper);
		}else{
			//TODO throw Exception
		}
		CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		return wrapper;
	}

	/**
	 * initialization of object;
	 *
	 */
	protected void init(Map prop) throws DataAccessException {
		boolean isNeedMetamodel = true;
		if(prop != null){
			Object obj = prop.get("LOCALE");
			if(obj instanceof Locale){
				mLocale = (Locale)obj;
			}
			
			Object needMetamodel = prop.get(ICAFServiceAccess.IS_NEED_METAMODEL);
			if(needMetamodel!=null && needMetamodel instanceof Boolean){
				isNeedMetamodel = ((Boolean)needMetamodel).booleanValue();
			}
		}
		
		if(isNeedMetamodel) {
			mmHelper = MetamodelHelperFactory.getMetamodelHelper(mLocale);
		}
		
		activeTransactionsCount = 0;
		serviceWrappers = new HashMap();		
	}
	
	public MetamodelHelper getMetamodelHelper() {
		return mmHelper;
	}

}
