package com.sap.caf.rt.services.publish;

import javax.ejb.CreateException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import com.sap.caf.rt.bol.context.CAFContext;
import com.sap.caf.rt.exception.BEException;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * PublishingBean implements the publishing service of caf which is called by any business object if it was
 * created, changed or removed. The service uses jms to send the message. The publisher/subscriber pattern is
 * used to offer any application to be notified if a change had taken place.
 * @ejbHome <{com.sap.caf.rt.services.PublishingHome}>
 * @ejbLocal <{com.sap.caf.rt.services.publish.PublishingLocal}>
 * @ejbLocalHome <{com.sap.caf.rt.services.publish.PublishingLocalHome}>
 * @ejbRemote <{com.sap.caf.rt.services.Publishing}>
 * @stateless 
 */
public class PublishingBean implements SessionBean {
  private static final Location location = Location.getLocation(PublishingBean.class);
  private final static String JARM_REQUEST = "CAF:RT:oal";
  private final static String CHANGE = "BUSINESS_OBJECT_CHANGED";
  private final static String REMOVE = "BUSINESS_OBJECT_REMOVED";
  private final static String CREATE = "BUSINESS_OBJECT_CREATED";
  private final static String[] msgTemplate = { "OBJTYPE", "KEY", "RID", "USEDBY" };
  private String CF_NAME = CAFContext.JNDI_NAME_JMS;
  private final static String XAPPS_TOPIC = CAFContext.JNDI_NAME_JMS_TOPIC;
  private Topic topic = null;
  private TopicSession session = null;
  private TopicPublisher pub = null;
  private TopicConnectionFactory cf = null;
  private TopicConnection conn = null;
  private Context context = null;

  public void ejbRemove() {
    String method = "ejbRemove()";
    CAFPublicLogger.entering(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);

    try {
      pub.close();
      session.close();
      conn.close();
    } catch (JMSException ex) {
      CAFPublicLogger.traceThrowableT(Severity.DEBUG, location, method, "JMSException in connection close", ex);
    } finally {
      CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
    }
  }

  public void ejbActivate() {
    String method = "ejbActivate()";
    CAFPublicLogger.entering(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);

    try {
      context = new InitialContext();
      cf = (TopicConnectionFactory) context.lookup(CF_NAME);
      conn = cf.createTopicConnection();
      topic = (Topic) context.lookup(XAPPS_TOPIC);

      session = conn.createTopicSession(false, Session.AUTO_ACKNOWLEDGE);
      pub = session.createPublisher(topic);
    } catch (NamingException e) {
      Object[] args = { CF_NAME, XAPPS_TOPIC };
      CAFPublicLogger.categoryCAF.logThrowableT(
        Severity.ERROR,
        location,
        method,
        "Error in JNDI lookup of {0} using topic {1}",
        args,
        e);
    } catch (JMSException e) {
      CAFPublicLogger.traceThrowableT(Severity.DEBUG, location, method, "JMSException in ejbActivate", e);
    } finally {
      CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
    }

  }

  public void ejbPassivate() {
    String method = "ejbPassivate()";
    CAFPublicLogger.entering(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);

    try {
      pub.close();
      session.close();
      conn.close();
    } catch (JMSException ex) {
      CAFPublicLogger.traceThrowableT(Severity.DEBUG, location, method, "JMSException in connection close", ex);
    } finally {
      CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
    }
  }

  public void setSessionContext(SessionContext context) {
    myContext = context;
  }

  private SessionContext myContext;
  /**
   * Create Method.
   */
  public void ejbCreate() throws CreateException, BEException {

    String method = "ejbCreate()";
    CAFPublicLogger.entering(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);

    try {
      context = new InitialContext();
      cf = (TopicConnectionFactory) context.lookup(CF_NAME);
      conn = cf.createTopicConnection();
      topic = (Topic) context.lookup(XAPPS_TOPIC);

      session = conn.createTopicSession(false, Session.AUTO_ACKNOWLEDGE);
      pub = session.createPublisher(topic);
    } catch (NamingException e) {
      Object[] args = { CF_NAME, XAPPS_TOPIC };
      CAFPublicLogger.categoryCAF.logThrowableT(
        Severity.ERROR,
        location,
        method,
        "Error in JNDI lookup of {0} using topic {1}",
        args,
        e);
      throw new BEException(e);
    } catch (JMSException e) {
      Object[] args = { CF_NAME, XAPPS_TOPIC };
      CAFPublicLogger.traceThrowableT(
        Severity.DEBUG,
        location,
        method,
        "JMSException in ejbCreate using JNDI name : {0} topic : {1}",
        args,
        e);
      throw new BEException(e);
    } finally {
      CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
    }
  }

  public void publishCreated(Object[] args) throws BEException {
    publish(args, CREATE);
  }

  public void publishChanged(Object[] args) throws BEException {
    publish(args, CHANGE);
  }

  public void publishRemoved(Object[] args) throws BEException {
    publish(args, REMOVE);
  }

  private void publish(Object[] args, String action) throws BEException {
    String method = "publish(Object[], action)";
    CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[] { args, action }, CAFPublicLogger.LEVEL_MEDIUM);

    try {
      Message msg = session.createMessage();
      msg.setStringProperty("ACTION", action);

      for (int i = 0; i < msgTemplate.length; i++) {
        msg.setStringProperty(msgTemplate[i], args[i].toString());
      }

      pub.publish(msg);
    } catch (JMSException e) {
      Object[] arg = { args, action };
      CAFPublicLogger.traceThrowableT(
        Severity.DEBUG,
        location,
        method,
        "JMSException in publish using arguments : {0} action : {1}",
        arg,
        e);
      throw new BEException("CAF_INVALID_INPUT", null, e);
    } finally {
      CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
    }

  }
}
