package com.sap.caf.rt.services.notify.subscr.persist.xml;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.sap.caf.rt.services.notify.res.IResourceKeys;
import com.sap.caf.rt.services.notify.subscr.persist.IPersistable;
import com.sap.caf.rt.services.notify.subscr.persist.PersistException;
import com.sap.caf.rt.services.notify.subscr.persist.PersistUtils;

/**
 * @author viachaslau_kudzinau@epam.com
 */
public final class XMLUtils implements IXMLNames {

	public static Object loadObject(IXMLPersistManager manager, Node root, Map nodes, String nodeName, 
		ClassLoader classLoader, Class classType, boolean required) throws PersistException 
	{
		Node node = XMLUtils.getNode(root, nodes, nodeName, required);
		return node!=null? loadObject(manager, node, classLoader, classType, required): null;
	}
	 
	public static Object loadObject(IXMLPersistManager manager, Node node, ClassLoader classLoader, Class classType, boolean required) throws PersistException {
		Object obj = createObject(node, classLoader, classType);
		if (obj instanceof IPersistable) {
			manager.load(node, (IPersistable)obj);
		}
		return obj;
	}
	 
	public static Object loadObject(IXMLPersistManager manager, Node node, 
		ClassLoader classLoader, Class classType) throws PersistException 
	{
		return loadObject(manager, node, classLoader, classType, true);
	}
	
	public static Object createObject(Node node, 
		ClassLoader classLoader, Class classType) throws PersistException 
	{
		return createObject(node, classLoader, classType, true);
	}

	public static Object createObject(Node node, ClassLoader classLoader, Class classType, boolean required) throws PersistException {
		String className = XMLUtils.getStrNodeAttrValue(node, ATTR_CLASS, required);
		if (className==null) {
			return null;
		}
		return PersistUtils.newObject(className, classLoader, classType);
	}

	public static Object createObject(Node root, Map nodes, String nodeName, 
		ClassLoader classLoader, Class classType, boolean required) throws PersistException 
	{
		Node node = XMLUtils.getNode(root, nodes, nodeName, required);
		return node!=null? createObject(node, classLoader, classType, required): null;
	}

	public static String getStrNodeAttrValue(Node root, Map nodes, String nodeName, boolean required) 
		throws MissedNodeException, MissedValueException, MissedAttrException 
	{
		Node node = XMLUtils.getNode(root, nodes, nodeName, required);
		return node!=null? getStrNodeAttrValue(node, ATTR_CLASS, required): null;
	}

	public static Map getNodesAsMap(Node root) {
		Map nodes = new LinkedHashMap();
		NodeList children = root.getChildNodes();
		Node node;
		for (int i=0; i<children.getLength(); i++) {
			node = children.item(i);
			if (node.getNodeType()==Node.ELEMENT_NODE) {
				nodes.put(node.getNodeName(), node);
			}
		}
		return nodes;
	}

	public static Node getChildNode(Node root, String nodeName) throws MissedNodeException {
		return getChildNode(root, nodeName, false);
	}
	
	public static Node getChildNode(Node root, String nodeName, boolean required) throws MissedNodeException {
		NodeList children = root.getChildNodes();
		Node node;
		for (int i=0; i<children.getLength(); i++) {
			node = children.item(i);
			if (node.getNodeType()==Node.ELEMENT_NODE && nodeName!=null && nodeName.equals(node.getNodeName())) {
				return node;
			}
		}
		if (required) {
			throw new MissedNodeException(root, nodeName);
		}
		return null;
	}
	
	
	public static List getNodesAsList(Node root, String nodeName) {
		List nodes = new ArrayList();
		NodeList children = root.getChildNodes();
		Node node;
		for (int i=0; i<children.getLength(); i++) {
			node = children.item(i);
			if (node.getNodeType()==Node.ELEMENT_NODE && nodeName!=null && nodeName.equals(node.getNodeName())) {
				nodes.add(node);
			}
		}
		return nodes;
	}
	
	public static String getStrChildNodeValue(Node root, String childName, boolean required) 
		throws MissedNodeException, MissedValueException 
	{
		NodeList children = root.getChildNodes();
		Node node;
		for (int i=0; i<children.getLength(); i++) {
			node = children.item(i);
			if (node.getNodeType()==Node.ELEMENT_NODE && childName.equals(node.getNodeName())) {
				return getStrNodeValue(node, required);
			}
		}
		if (required) {
			throw new MissedNodeException(root, childName);
		}
		return null;
	}

	public static String getStrNodeAttrValue(Node root, String attrName, boolean required) 
		throws MissedValueException, MissedAttrException 
	{
		NamedNodeMap attrs = root.getAttributes();
		Node node = attrs.getNamedItem(attrName);
		if (node==null) {
			if (required) {
				throw new MissedAttrException(root, attrName);
			}
			return null;
		}
		String value = node.getNodeValue();
		if (required && (value==null || value.length()==0)) {
			throw new MissedValueException(node);
		}
		return (value!=null? value.trim(): null);
	}
	
	public static String getStrChildNodeAttrValue(Node root, Map nodes, String childNodeName, 
		String attrName, boolean required) throws MissedNodeException, MissedValueException, MissedAttrException 
	{
		Node node = findNode(childNodeName, nodes);
		if (node==null) {
			if (required) {
				throw new MissedNodeException(root, childNodeName);
			}
			return null;
		}
		return getStrNodeAttrValue(node, attrName, required);
	} 
	
	public static Node getNode(Node root, Map nodes, String nodeName, boolean required) throws MissedNodeException {
		Node node = (Node)nodes.get(nodeName);
		if (required && node==null) {
			throw new MissedNodeException(root, nodeName);
		}
		return node;
	}

	public static Node findNode(String nodeName, Map nodes) {
		return (Node)nodes.get(nodeName);
	}

	public static String getStrChildNodeValue(Node root, Map nodes, String nodeName, boolean required) 
		throws MissedNodeException, MissedValueException 
	{
		Node node = getNode(root, nodes, nodeName, required);
		String value = getStrNodeValue(node, required);
		if (required && (value==null || value.length()==0)) {
			throw new MissedValueException(node);
		}
		return value;
	}

	public static String getStrNodeValue(Node node, boolean required) throws MissedValueException {
		String value = null;
		if (node.getNodeType()==Node.ATTRIBUTE_NODE) {
			value = node.getNodeValue();
		}
		else {
			Node childNode = node.getFirstChild();
			if (childNode==null || childNode.getNodeType()!=Node.TEXT_NODE) {
				throw new MissedValueException(node);
			}
			value = childNode.getNodeValue();
		}
		if (required && (value==null || value.length()==0)) {
			throw new MissedValueException(node);
		}
		return (value!=null? value.trim(): null);
	}

	public static boolean getBoolNodeValue(Node root, Map nodes, String nodeName, boolean required) 
		throws PersistException 
	{
		String value = XMLUtils.getStrChildNodeValue(root, nodes, nodeName, required);
		if (value==null) {
			return false;
		}
		else if ("true".equals(value)) {
			return true;
		}
		else if ("false".equals(value)) {
			return false;
		}
		throw new PersistException(IResourceKeys.WRONG_BOOL_NODE_VALUE, new String[] {nodeName});
	}

	public static Map getAttributes(Node root) {
		NamedNodeMap attrs = root.getAttributes();
		Map map = new LinkedHashMap();
		Node node;
		for(int i=0; i<attrs.getLength(); i++) {
			node = attrs.item(i);
			map.put(node.getNodeName(), node);
		}
		return map;
	}

}
