package com.sap.caf.rt.services.notify.subscr.persist.subscr;

import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.w3c.dom.Node;

import com.sap.caf.rt.services.notify.common.Log;
import com.sap.caf.rt.services.notify.res.IResourceKeys;
import com.sap.caf.rt.services.notify.subscr.ISubscrFilter;
import com.sap.caf.rt.services.notify.subscr.ISubscrRecipients;
import com.sap.caf.rt.services.notify.subscr.ISubscrTemplate;
import com.sap.caf.rt.services.notify.subscr.Subscription;
import com.sap.caf.rt.services.notify.subscr.app.IApplication;
import com.sap.caf.rt.services.notify.subscr.persist.IPersistEntity;
import com.sap.caf.rt.services.notify.subscr.persist.IPersistManager;
import com.sap.caf.rt.services.notify.subscr.persist.PersistException;
import com.sap.caf.rt.services.notify.subscr.persist.PersistUtils;
import com.sap.caf.rt.services.notify.subscr.persist.sql.ISQLNames;
import com.sap.caf.rt.services.notify.subscr.persist.sql.ISQLPersistEntity;
import com.sap.caf.rt.services.notify.subscr.persist.sql.ISQLPersistManager;
import com.sap.caf.rt.services.notify.subscr.persist.sql.SQLUtils;
import com.sap.caf.rt.services.notify.subscr.persist.state.IPersistState;
import com.sap.caf.rt.services.notify.subscr.persist.state.PersistState;
import com.sap.caf.rt.services.notify.subscr.persist.xml.IXMLNames;
import com.sap.caf.rt.services.notify.subscr.persist.xml.IXMLPersistManager;
import com.sap.caf.rt.services.notify.subscr.persist.xml.XMLUtils;

public class PersistSubscription extends Subscription implements IPersistSubscription {

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistEntity#getKey()
	 */
	public Object getKey() {
		return m_key;
	}

	public void setKey(Object key) {
		m_key = key;
	}

	public void load(IPersistManager manager) throws PersistException {
		if (manager instanceof IXMLPersistManager) {
			load((IXMLPersistManager) manager);
		} else if (manager instanceof ISQLPersistManager) {
			load((ISQLPersistManager) manager);
		}
	}

	protected void load(ISQLPersistManager manager) throws PersistException {
		Map values = SQLUtils.loadObjValues(manager, ISQLNames.SELECT_SUBSCR, ISQLNames.SELECT_SUBSCR_TYPES, this);
		setId((String) values.get(ISQLNames.SUBSCR_ID));
		setGroupId((String) values.get(ISQLNames.SUBSCR_GRP_ID));
		setEnabled(SQLUtils.INT_1.equals((Integer) values.get(ISQLNames.SUBSCR_ENABLED)));
		setDescription((String) values.get(ISQLNames.SUBSCR_DESCR));
		setFilter(loadFilter(manager));
		setTemplate(loadTemplate(manager));
		setRecipients(loadRecipients(manager));
	}

	private ISubscrRecipients loadRecipients(ISQLPersistManager manager) throws PersistException {
		List rows = SQLUtils.execSelect(manager, ISQLNames.SELECT_RECIP_KEY, ISQLNames.SELECT_RECIP_KEY_TYPES, new Object[] { getKey()});
		if (rows == null || rows.size() == 0) {
			throw new PersistException(IResourceKeys.CANT_LOAD_INCL_SUBSCR_RECIP, new Object[] { getKey()});
		}
		Map row = (Map) rows.get(0);
		String className = (String) row.get(ISQLNames.FLT_CLASS_NAME);
		IPersistSubscrRecipients recip = (IPersistSubscrRecipients) newObject(className, IPersistSubscrRecipients.class, PersistSubscrRecipients.class);
		recip.setSubscription(this);
		recip.setKey(row.get(ISQLNames.RECIP_KEY));
		manager.load(recip);
		return recip;
	}

	protected ISubscrTemplate loadTemplate(ISQLPersistManager manager) throws PersistException {
		List rows = SQLUtils.execSelect(manager, ISQLNames.SELECT_TEMPL_KEY, ISQLNames.SELECT_TEMPL_KEY_TYPES, new Object[] { getKey()});
		if (rows == null || rows.size() == 0) {
			throw new PersistException(IResourceKeys.CANT_LOAD_SUBSCR_TEMPL, new Object[] { getKey()});
		}
		Map row = (Map) rows.get(0);
		String className = (String) row.get(ISQLNames.FLT_CLASS_NAME);
		IPersistSubscrTemplate templ = (IPersistSubscrTemplate) newObject(className, IPersistSubscrTemplate.class, PersistSubscrTemplate.class);
		templ.setSubscription(this);
		templ.setKey(row.get(ISQLNames.TEMPL_KEY));
		manager.load(templ);
		return templ;
	}

	protected ISubscrFilter loadFilter(ISQLPersistManager manager) throws PersistException {
		List rows = SQLUtils.execSelect(manager, ISQLNames.SELECT_FLT_KEYS, ISQLNames.SELECT_FLT_KEYS_TYPES, new Object[] { getKey()});
		if (rows == null || rows.size() == 0) {
			throw new PersistException(IResourceKeys.CANT_LOAD_SUBSCR_FLT, new Object[] { getKey()});
		}
		Map row = (Map) rows.get(0);
		String className = (String) row.get(ISQLNames.FLT_CLASS_NAME);
		IPersistSubscrFilter filter = (IPersistSubscrFilter) newObject(className, IPersistSubscrFilter.class, PersistSubscrFilter.class);
		filter.setSubscription(this);
		filter.setKey(row.get(ISQLNames.FLT_KEY));
		manager.load(filter);
		return filter;
	}

	protected void load(IXMLPersistManager manager) throws PersistException {
		Node root = manager.getCurrentNode();
		Map nodes = XMLUtils.getNodesAsMap(root);
		Map attrs = XMLUtils.getAttributes(root);
		ClassLoader classLoader = getClassLoader();
		Properties props = getApplication().getProperties();
		String resName = ((IPersistApplication) getApplication()).getResourceName();
		setId(XMLUtils.getStrChildNodeValue(root, attrs, IXMLNames.ATTR_ID, true));
		setEnabled(PersistUtils.resolveBoolPropRef(XMLUtils.getStrChildNodeValue(root, attrs, IXMLNames.ATTR_ACTIVE, true), props));
		setDescription(PersistUtils.resolveResRef(XMLUtils.getStrChildNodeValue(root, nodes, IXMLNames.TAG_DESCR, true), resName, classLoader));
		setGroupId(XMLUtils.getStrChildNodeValue(root, nodes, IXMLNames.TAG_GROUP, true));
		// load dependent objects
		setFilter(loadFilter(manager, root, nodes));
		setTemplate(loadTemplate(manager, root, nodes));
		setRecipients(loadRecipients(manager, root, nodes));
		//		setExcludedRecipients(loadExclRecipients(manager, root, nodes));
	}

	protected IPersistSubscrFilter loadFilter(IXMLPersistManager manager, Node root, Map nodes) throws PersistException {
		Node node = XMLUtils.getNode(root, nodes, IXMLNames.TAG_FILTER, true);
		String className = XMLUtils.getStrNodeAttrValue(node, IXMLNames.ATTR_CLASS, true);
		IPersistSubscrFilter filter = (IPersistSubscrFilter) PersistUtils.newObject(className, getClassLoader(), IPersistSubscrFilter.class);
		filter.setSubscription(this);
		manager.load(node, filter);
		return filter;
	}

	protected IPersistSubscrTemplate loadTemplate(IXMLPersistManager manager, Node root, Map nodes) throws PersistException {
		Node node = XMLUtils.getNode(root, nodes, IXMLNames.TAG_TEMPLATE, true);
		String className = XMLUtils.getStrNodeAttrValue(node, IXMLNames.ATTR_CLASS, true);
		IPersistSubscrTemplate template = (IPersistSubscrTemplate) PersistUtils.newObject(className, getClassLoader(), IPersistSubscrTemplate.class);
		template.setSubscription(this);
		template.setId(getId());
		manager.load(node, template);
		return template;
	}

	protected IPersistSubscrRecipients loadRecipients(IXMLPersistManager manager, Node root, Map nodes) throws PersistException {
		Node node = XMLUtils.getNode(root, nodes, IXMLNames.TAG_RECIPIENTS, true);
		String className = XMLUtils.getStrNodeAttrValue(node, IXMLNames.ATTR_CLASS, true);
		IPersistSubscrRecipients recip = (IPersistSubscrRecipients) PersistUtils.newObject(className, getClassLoader(), IPersistSubscrRecipients.class);
		recip.setSubscription(this);
		manager.load(node, recip);
		return recip;
	}

	protected IPersistSubscrRecipients loadExclRecipients(IXMLPersistManager manager, Node root, Map nodes) throws PersistException {
		Node node = XMLUtils.getNode(root, nodes, IXMLNames.TAG_EXCL_RECIPIENTS, false);
		if (node == null) {
			return null;
		}
		String className = XMLUtils.getStrNodeAttrValue(node, IXMLNames.ATTR_CLASS, false);
		if (className == null) {
			return null;
		}
		IPersistSubscrRecipients recip = (IPersistSubscrRecipients) PersistUtils.newObject(className, getClassLoader(), IPersistSubscrRecipients.class);
		recip.setSubscription(this);
		manager.load(node, recip);
		return recip;
	}

	public void store(IPersistManager manager) throws PersistException {
		if (!isActive()) {
			throw new PersistException(IResourceKeys.CANT_STORE_INACTIVE, new Object[] { getKey()});
		}
		if (manager instanceof ISQLPersistManager) {
			store((ISQLPersistManager) manager);
		}
	}

	protected void store(ISQLPersistManager manager) throws PersistException {
		SQLUtils.execUpdate(manager, ISQLNames.UPDATE_SUBSCR, ISQLNames.UPDATE_SUBSCR_TYPES, new Object[] {((ISQLPersistEntity) getApplication()).getKey(), // app_key
			getId(), // id
			getGroupId(), // group_id
			isEnabled() ? SQLUtils.INT_1 : SQLUtils.INT_0, // is_enabled
			getDescription(), // descr 
			getKey(), // subscr_key
		});
		// update dependent entities
		manager.store((IPersistEntity) getFilter());
		manager.store((IPersistEntity) getTemplate());
		manager.store((IPersistEntity) getRecipients());
	}

	public void delete(IPersistManager manager) throws PersistException {
		if (manager instanceof ISQLPersistManager) {
			ISQLPersistManager sqlManager = (ISQLPersistManager) manager;
			Object key = getKey();
			PersistSubscrPrefsFactory prefsFactory = (PersistSubscrPrefsFactory) PersistSubscrPrefsFactory.getInstance();
			prefsFactory.deleteSubscrPrefs(key);
			// delete dependent entities
			manager.delete((IPersistEntity) getFilter());
			manager.delete((IPersistEntity) getTemplate());
			manager.delete((IPersistEntity) getRecipients());
			// delete subscr
			SQLUtils.execUpdate(sqlManager, ISQLNames.DELETE_SUBSCR, ISQLNames.DELETE_SUBSCR_TYPES, new Object[] { key // subscr_key
			});
		}
	}

	public void create(IPersistManager manager) throws PersistException {
		if (manager instanceof ISQLPersistManager) {
			ISQLPersistManager sqlManager = (ISQLPersistManager) manager;
			setKey(SQLUtils.createKey());
			SQLUtils.execUpdate(sqlManager, ISQLNames.INSERT_SUBSCR, ISQLNames.INSERT_SUBSCR_TYPES, new Object[] { getKey(), // subscr_key
				 ((ISQLPersistEntity) getApplication()).getKey(), // app_key
				getId(), // id
				getGroupId(), // group_id
				isEnabled() ? SQLUtils.INT_1 : SQLUtils.INT_0, // is_enabled
				getDescription(), // descr 
			});
			// create dependent
			manager.create((IPersistEntity) getFilter());
			manager.create((IPersistEntity) getTemplate());
			manager.create((IPersistEntity) getRecipients());
		}
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistable#getStateMachine()
	 */
	public IPersistState getState() {
		return m_state;
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistable#getClassLoader()
	 */
	public ClassLoader getClassLoader() {
		return getApplication().getClassLoader();
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscription#setDescription(java.lang.String)
	 */
	public void setDescription(String descr) {
		super.setDescription(descr);
		getState().setUpdated(true);
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscription#setEnabled(boolean)
	 */
	public void setEnabled(boolean enabled) {
		super.setEnabled(enabled);
		getState().setUpdated(true);
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscription#setGroupId(java.lang.String)
	 */
	public void setGroupId(String groupId) {
		super.setGroupId(groupId);
		getState().setUpdated(true);
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscription#setId(java.lang.String)
	 */
	public void setId(String id) {
		super.setId(id);
		getState().setUpdated(true);
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.subscr.IPersistSubscription#isActive()
	 */
	public boolean isActive() {
		if (!m_bActive) {
			return false;
		}
		IApplication app = getApplication();
		boolean isAppActive = app != null ? app.isActive() : false;
		return isAppActive;
	}

	public void setActive(boolean active) {
		m_bActive = active;
	}

	protected Object newObject(String className, Class classType, Class defClassType) throws PersistException {
		ClassLoader classLoader = getClassLoader();
		try {
			return PersistUtils.newObject(className, classLoader, classType);
		} catch (PersistException e) {
			Log.fatal(IResourceKeys.CANT_LOAD_CLASS, new Object[] { className });
			if (Log.isDebugEnabled()) {
				Log.debug("Can not load class", e);
			}
		}
		// custom class was not loaded, disactivate subscr and load def class
		setActive(false);
		return PersistUtils.newObject(defClassType.getName(), classLoader, defClassType);
	}

	private boolean m_bActive = true;
	private IPersistState m_state = new PersistState();
	private Object m_key;

}
