package com.sap.caf.rt.services.notify.subscr.persist.subscr;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Map.Entry;

import org.w3c.dom.Node;

import com.sap.caf.rt.services.notify.subscr.SubscrTemplate;
import com.sap.caf.rt.services.notify.subscr.persist.sql.ISQLNames;
import com.sap.caf.rt.services.notify.subscr.persist.sql.ISQLPersistEntity;
import com.sap.caf.rt.services.notify.subscr.persist.sql.ISQLPersistManager;
import com.sap.caf.rt.services.notify.subscr.persist.sql.SQLUtils;
import com.sap.caf.rt.services.notify.subscr.persist.state.IPersistState;
import com.sap.caf.rt.services.notify.subscr.persist.state.PersistState;
import com.sap.caf.rt.services.notify.subscr.persist.xml.IXMLNames;
import com.sap.caf.rt.services.notify.subscr.persist.xml.IXMLPersistManager;
import com.sap.caf.rt.services.notify.subscr.persist.xml.MissedAttrException;
import com.sap.caf.rt.services.notify.subscr.persist.xml.MissedValueException;
import com.sap.caf.rt.services.notify.subscr.persist.xml.XMLUtils;
import com.sap.caf.rt.services.notify.subscr.persist.IPersistManager;
import com.sap.caf.rt.services.notify.subscr.persist.PersistException;
import com.sap.caf.rt.services.notify.subscr.persist.PersistUtils;

/**
 * @author viachaslau_kudzinau@epam.com
 */
public class PersistSubscrTemplate extends SubscrTemplate implements IPersistSubscrTemplate {

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistable#load(com.sap.caf.rt.services.notify.subscr.persist.IPersistManager)
	 */
	public void load(IPersistManager manager) throws PersistException {
		if (manager instanceof IXMLPersistManager) {
			load((IXMLPersistManager)manager);
		}
		else if (manager instanceof ISQLPersistManager) {
			load((ISQLPersistManager)manager);
		}
	}
	
	protected void load(ISQLPersistManager manager) throws PersistException {
		Map values = SQLUtils.loadObjValues(manager, ISQLNames.SELECT_TEMPL, ISQLNames.SELECT_TEMPL_TYPES, this);
		setId((String)values.get(ISQLNames.TEMPL_ID));
		setSubject((String)values.get(ISQLNames.TEMPL_SUBJ));
		setProperties(loadProperties(manager));
	}

	protected Properties loadProperties(ISQLPersistManager manager) throws PersistException {
		Map map = SQLUtils.loadMap(manager, 
				ISQLNames.SELECT_TEMPLPROP, ISQLNames.SELECT_TEMPLPROP_TYPES, new Object[] {getKey()});
		Properties props = new Properties();  
		props.putAll(map);
		return props;
	}

	protected void load(IXMLPersistManager manager) throws PersistException {
		Node root = manager.getCurrentNode();
		setSubject(XMLUtils.getStrChildNodeValue(root, IXMLNames.TAG_SUBJECT, true));
		loadProperties(XMLUtils.getNodesAsList(
			XMLUtils.getChildNode(root, IXMLNames.TAG_PROPERTIES, true), IXMLNames.TAG_PROPERTY));		
	}

	
	protected void loadProperties(List nodes) throws MissedValueException, MissedAttrException, PersistException {
		if (nodes!=null) {
			Node node;
			IPersistApplication app = (IPersistApplication)getSubscription().getApplication();
			for(int i=0; i<nodes.size(); i++) {
				node = (Node)nodes.get(i);
				addProperty(
						XMLUtils.getStrNodeAttrValue(node, IXMLNames.ATTR_NAME, true),
						PersistUtils.resolveRef(XMLUtils.getStrNodeValue(node, true), app) 
					);
			}
		}
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistEntity#create()
	 */
	public void create(IPersistManager manager) throws PersistException {
		if (manager instanceof ISQLPersistManager) {
			ISQLPersistManager sqlManager = (ISQLPersistManager)manager; 
			setKey(SQLUtils.createKey());
			SQLUtils.execUpdate(sqlManager, ISQLNames.INSERT_TEMPL,
				ISQLNames.INSERT_TEMPL_TYPES, 
				new Object[] {
					getKey(), // templ_key
					((ISQLPersistEntity)getSubscription()).getKey(), // subscr_key
					getId(), // id
					getClass().getName(), // class_name
					getSubject() // subject
				});
			createProperties(sqlManager);
		}
	}

	protected void createProperties(ISQLPersistManager manager) throws PersistException {
		Properties props = getProperties();
		Map.Entry prop;
		for(Iterator itr = props.entrySet().iterator(); itr.hasNext(); ) {
			prop = (Map.Entry)itr.next();
			createProperty(manager, prop);
		}
	}

	protected void createProperty(ISQLPersistManager manager, Map.Entry property) throws PersistException {
		SQLUtils.execUpdate(manager, ISQLNames.INSERT_TEMPLPROP,
			ISQLNames.INSERT_TEMPLPROP_TYPES, 
			new Object[] {
				getKey(), // templ_key
				property.getKey(), // name
				property.getValue() // val
			});
	}
	
	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistEntity#delete()
	 */
	public void delete(IPersistManager manager) throws PersistException {
		if (manager instanceof ISQLPersistManager) {
			ISQLPersistManager sqlManager = (ISQLPersistManager)manager;
			deleteProperties(sqlManager); 
			SQLUtils.execUpdate(sqlManager, ISQLNames.DELETE_TEMPL,
				ISQLNames.DELETE_TEMPL_TYPES, 
				new Object[] {
					getKey(), // templ_key
					((ISQLPersistEntity)getSubscription()).getKey() // subscr_key
				});
		}
	}


	protected void deleteProperties(ISQLPersistManager manager) throws PersistException {
		SQLUtils.execUpdate(manager, ISQLNames.DELETE_TEMPLPROP_TEMPL,
			ISQLNames.DELETE_TEMPLPROP_TEMPL_TYPES, 
			new Object[] {
				getKey(), // templ_key
			});
	}

	protected void deleteProperty(ISQLPersistManager manager, Map.Entry property) throws PersistException {
		SQLUtils.execUpdate(manager, ISQLNames.DELETE_TEMPLPROP,
			ISQLNames.DELETE_TEMPLPROP_TYPES, 
			new Object[] {
				getKey(), // templ_key
				property.getKey() // name
			});
	}
	
	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistable#store(com.sap.caf.rt.services.notify.subscr.persist.IPersistManager)
	 */
	public void store(IPersistManager manager) throws PersistException {
		if (manager instanceof ISQLPersistManager) {
			ISQLPersistManager sqlManager = (ISQLPersistManager)manager; 
			SQLUtils.execUpdate(sqlManager, ISQLNames.UPDATE_TEMPL,
				ISQLNames.UPDATE_TEMPL_TYPES, 
				new Object[] {
					((ISQLPersistEntity)getSubscription()).getKey(), // subscr_key
					getId(), // id
					getClass().getName(), // class_name
					getSubject(), // subject
					getKey(), // templ_key
				});
			storeProperties(sqlManager);
		}
	}
	
	protected void storeProperties(ISQLPersistManager manager) throws PersistException {
		Map oldProps = loadProperties(manager);
		Map newProps = getProperties();
		if (!newProps.isEmpty()) {
			Map.Entry newProp;
			Object newKey;
			for(Iterator itr = newProps.entrySet().iterator(); itr.hasNext(); ) {
				newProp = (Map.Entry)itr.next();
				newKey = newProp.getKey();
				if (oldProps.containsKey(newKey)) {
					if (!oldProps.get(newKey).equals(newProp.getValue())) {
						updateProperty(manager, newProp);
					}
					oldProps.remove(newKey);
				}
				else {
					createProperty(manager, newProp);
				}
			}
		}
		if (!oldProps.isEmpty()) {
			Map.Entry oldProp;
			for(Iterator itr = oldProps.entrySet().iterator(); itr.hasNext(); ) {
				oldProp = (Map.Entry)itr.next();
				deleteProperty(manager, oldProp);
			}
		}
	}

	protected void updateProperty(ISQLPersistManager manager, Map.Entry property) throws PersistException {
		SQLUtils.execUpdate(manager, ISQLNames.UPDATE_TEMPLPROP,
			ISQLNames.UPDATE_TEMPLPROP_TYPES, 
			new Object[] {
				property.getValue(), // val
				getKey(), // templ_key
				property.getKey(), // name
			});
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscrTemplate#getLocalizedSubject()
	 */
	public synchronized String getLocalizedSubject() {
		String subj = getSubject();
		if (m_sLocSubj==null && subj!=null && !subj.equals(m_sLocSubj)) {
			m_sLocSubj=subj;
			if (subj!=null && subj.length()!=0) {
				String resKey = PersistUtils.getReferenceKey(subj);
				if (resKey!=null) {
					String resName = getResourceName();
					m_sLocSubj =  resName!=null? PersistUtils.getLocalizedMessage(resName, resKey, null, getClassLoader()): subj;
				}
			}
		}
		return m_sLocSubj;
	}

	private String getResourceName() {
		return ((IPersistApplication)getSubscription().getApplication()).getResourceName();
	}
	
	protected ClassLoader getClassLoader() {
		return ((IPersistSubscription)getSubscription()).getApplication().getClassLoader();
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistable#getStateMachine()
	 */
	public IPersistState getState() {
		return m_state;
	}
	
	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.sql.ISQLPersistEntity#getKey()
	 */
	public Object getKey() {
		return m_key;
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.sql.ISQLPersistEntity#setKey(java.lang.Object)
	 */
	public void setKey(Object key) {
		m_key = key;
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscrTemplate#setSubject(java.lang.String)
	 */
	public void setSubject(String subject) {
		super.setSubject(subject);
		m_sLocSubj = null;
		getState().setUpdated(true);
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscrTemplate#setId(java.lang.String)
	 */
	public void setId(String id) {
		super.setId(id);
		getState().setUpdated(true);
	}
	
	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscrTemplate#addProperty(java.lang.String, java.lang.String)
	 */
	public void addProperty(String name, String value) {
		super.addProperty(name, value);
		getState().setUpdated(true);
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscrTemplate#removeProperty(java.lang.String)
	 */
	public void removeProperty(String name) {
		super.removeProperty(name);
		getState().setUpdated(true);
	}
	
	/**
	 * @see com.sap.caf.rt.services.notify.subscr.ISubscrTemplate#setProperties(java.util.Properties)
	 */
	public void setProperties(Properties props) {
		super.setProperties(props);
		getState().setUpdated(true);
	}

	private Object m_key;
	private IPersistState m_state = new PersistState();
	private String m_sLocSubj;

}
