package com.sap.caf.rt.services.notify.subscr.persist.sql;

import java.sql.Connection;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import com.sap.caf.rt.services.notify.common.JndiUtils;
import com.sap.caf.rt.services.notify.common.Log;
import com.sap.caf.rt.services.notify.res.IResourceKeys;
import com.sap.caf.rt.services.notify.subscr.persist.PersistException;
import com.sap.caf.rt.services.notify.subscr.persist.PersistManager;
import com.sap.engine.frame.core.thread.ThreadContext;

/**
 * @author viachaslau_kudzinau@epam.com
 */
public class SQLPersistManager extends PersistManager implements ISQLPersistManager {

	protected synchronized TransactionManager getTransactionManager() throws PersistException {
		if (m_transMan == null) {
			try {
				InitialContext context = getJndiContext();
				m_transMan = (TransactionManager) context.lookup("ts");
			} catch (NamingException e) {
				Log.fatal(e);
				throw new PersistException(IResourceKeys.CANT_GET_TR_MANAGER, e);
			}
		}
		return m_transMan;
	}

	protected synchronized InitialContext getJndiContext() throws NamingException {
		if (m_jndiContext == null) {
			m_jndiContext = new InitialContext();
		}
		return m_jndiContext;
	}

	public synchronized Connection getConnection() throws PersistException {
		try {
			DataSource dataSource = (DataSource) getJndiContext().lookup(DATASOURCE_NAME);
			return dataSource.getConnection();
		} catch (Exception e) {
			Log.fatal(e);
			throw new PersistException(IResourceKeys.CANT_GET_DB_CONNECT, e);
		}
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistManager#begin()
	 */
	public void begin() throws PersistException {
		if (!hasThreadContext()) {
			return;
		}
		TransactionManager tm = getTransactionManager();
		suspend(tm);
		try {
			tm.begin();
		} catch (Exception e) {
			Log.fatal(e);
			resume(tm);
			throw new PersistException(IResourceKeys.CANT_BEGIN_TRANS, e);
		}
	}

	protected void resume(TransactionManager tm) throws PersistException {
		Transaction oldTrans = getOldTransaction();
		if (oldTrans == null) {
			return;
		}
		try {
			tm.resume(oldTrans);
		} catch (Exception e) {
			Log.fatal(e);
			throw new PersistException(IResourceKeys.CANT_RESUME_OLD_TR, e);
		} finally {
			m_oldTrans = null;
		}
	}

	protected void suspend(TransactionManager tm) throws PersistException {
		if (getOldTransaction() != null) {
			throw new PersistException(IResourceKeys.OLD_TR_ACTIVE);
		}
		try {
			setOldTransaction(tm.suspend());
		} catch (SystemException e) {
			Log.fatal(e);
			throw new PersistException(IResourceKeys.CANT_SUSPEND_OLD_TR, e);
		}
	}

	protected synchronized Transaction getOldTransaction() {
		return m_oldTrans;
	}

	protected void setOldTransaction(Transaction trans) {
		m_oldTrans = trans;
	}
	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistManager#commit()
	 */
	public void commit() throws PersistException {
		if (!hasThreadContext()) {
			return;
		}
		TransactionManager tm = getTransactionManager();
		try {
			tm.commit();
		} catch (Exception e) {
			Log.fatal(e);
			throw new PersistException(IResourceKeys.CANT_COMMIT_TRANS, e);
		} finally {
			resume(tm);
		}
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.persist.IPersistManager#revert()
	 */
	public void revert() throws PersistException {
		if (!hasThreadContext()) {
			return;
		}
		TransactionManager tm = getTransactionManager();
		try {
			tm.rollback();
		} catch (Exception e) {
			Log.fatal(e);
			throw new PersistException(IResourceKeys.CANT_ROLLBACK_TRANS, e);
		} finally {
			resume(tm);
		}
	}

	protected boolean hasThreadContext() throws PersistException {
		try {
			ThreadContext threadContext = JndiUtils.getThreadContext(getClass().getClassLoader());
			return (threadContext != null && !threadContext.isSystem());
		} catch (Exception e) {
			Log.fatal(e);
			throw new PersistException(e);
		}
	}

	private Transaction m_oldTrans;
	private TransactionManager m_transMan;
	private InitialContext m_jndiContext;
	private final static String DATASOURCE_NAME = "jdbc/SAP/CAF_RT";
}
