package com.sap.caf.rt.services.notify.subscr.app;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import com.sap.caf.rt.services.notify.common.Log;
import com.sap.caf.rt.services.notify.res.IResourceKeys;
import com.sap.caf.rt.services.notify.subscr.SubscrListFactory;
import com.sap.caf.rt.services.notify.subscr.ISubscrList;
import com.sap.caf.rt.services.notify.subscr.SubscrException;
import com.sap.caf.rt.services.notify.subscr.persist.PersistException;
import com.sap.caf.rt.services.notify.subscr.persist.subscr.IPersistApplication;
import com.sap.caf.rt.services.notify.subscr.persist.xml.IXMLPersistManager;
import com.sap.caf.rt.services.notify.subscr.persist.xml.XMLPersistManager;

/**
 * @author viachaslau_kudzinau@epam.com
 */
public class AppSubscrListFactory extends SubscrListFactory {

	protected AppSubscrListFactory(IPersistApplication app) {
		m_app = app;
	}

	public static AppSubscrListFactory getInstance(IPersistApplication app) {
		return new AppSubscrListFactory(app);
	}

	public ISubscrList getSubscrList() throws SubscrException {
		if (m_list == null) {
			try {
				if (Log.isDebugEnabled()) {
					Log.debug("Trying to load subscriptions for app:[" + m_app.getId() + "]...");
				}
				if (!m_app.isValid()) {
					Log.debug("The app:[" + m_app.getId() + "] is not valid!");
				} else {
					m_list = createSubscrList(m_app.getProperties(), m_app.getClassLoader());
					if (m_list != null && m_list instanceof IAppSubscrList) {
						if (Log.isDebugEnabled()) {
							Log.debug("Subscriptions found for app:[" + m_app.getId() + "]! SubscrList:[" + m_list + "]");
						}
						IAppSubscrList appList = (IAppSubscrList) m_list;
						appList.setApplication(m_app);
						Document configDoc = getConfigDocument(m_app.getClassLoader());
						if (configDoc != null) {
							IXMLPersistManager persistManager = new XMLPersistManager();
							persistManager.begin();
							try {
								persistManager.load(configDoc.getDocumentElement(), appList);
								persistManager.commit();
							} catch (PersistException e) {
								persistManager.revert();
								throw e;
							}
						}
						if (Log.isDebugEnabled()) {
							Log.debug("Subscriptions loaded for app:[" + m_app.getId() + "]");
						}
					}
				}
			} catch (Exception e) {
				throw new SubscrException(IResourceKeys.CANT_GET_APP_SUBCR, e);
			}
		}
		return m_list;
	}

	private Document getConfigDocument(ClassLoader appLoader) throws SAXException, IOException, ParserConfigurationException {
		InputStream is = appLoader.getResourceAsStream(CONFIG_NAME);
		if (is == null) {
			if (Log.isDebugEnabled()) {
				Log.debug("Config file:[" + CONFIG_NAME + "] was not found for app:[" + m_app.getId() + "], Subscriptions were not loaded");
			}
			return null;
		}
		if (Log.isDebugEnabled()) {
			Log.debug("Config file:[" + CONFIG_NAME + "] found for app:[" + m_app.getId() + "]");
		}
		// Create a builder factory
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setValidating(false);
		factory.setIgnoringComments(true);

		// Create the builder and parse the file
		return factory.newDocumentBuilder().parse(is);
	}

	public IApplication getApplication() {
		return m_app;
	}

	/**
	 * @see com.sap.caf.rt.services.notify.subscr.SubscrListFactory#getListClassName(java.util.Properties)
	 */
	protected String getListClassName(Properties props) throws SubscrException {
		String sEnabled = props.getProperty(PROP_SUBSCR_ENABLED);
		if (sEnabled == null) {
			if (Log.isDebugEnabled()) {
				Log.debug("Property:[" + PROP_SUBSCR_ENABLED + "] was not found, subscriptions will be ignored.");
			}
			return null;
		}
		if (!Boolean.valueOf(sEnabled).booleanValue()) {
			if (Log.isDebugEnabled()) {
				Log.debug("Property:[" + PROP_SUBSCR_ENABLED + "] is set to 'false', subscriptions will be ignored.");
			}
			return null;
		}
		return AppSubscrList.class.getName();
	}

	protected ISubscrList m_list;
	protected IPersistApplication m_app;

	protected final static String CONFIG_NAME = "META-INF/subscriptions/subscriptions.xml";
	protected final static String PROP_SUBSCR_ENABLED = "Subscription.Enabled";

}
