package com.sap.caf.rt.services.notify.notif;

import java.util.Collection;
import java.util.Iterator;

import com.sap.caf.rt.services.notify.res.IResourceKeys;
import com.sap.caf.rt.services.notify.common.Log;
import com.sap.caf.rt.services.notify.common.NSBaseException;
import com.sap.caf.rt.services.notify.notif.send.NotifSender;
import com.sap.caf.rt.services.notify.subscr.ISubscrFilter;
import com.sap.caf.rt.services.notify.subscr.ISubscrList;
import com.sap.caf.rt.services.notify.subscr.ISubscrListFactory;
import com.sap.caf.rt.services.notify.subscr.ISubscrRecipients;
import com.sap.caf.rt.services.notify.subscr.ISubscription;
import com.sap.caf.rt.services.notify.subscr.SubscrException;
import com.sap.caf.rt.services.notify.subscr.app.IApplication;
import com.sap.caf.rt.services.notify.subscr.persist.subscr.IPersistSubscription;

public abstract class NotifDispatcher implements INotifDispatcher {

	/**
	 * Find a subscription by event Id, apply filters, evaluate recipients
	 * and send a message
	 */
	public void notify(INotifEvent event) throws NotifException {
		final String method = "notify(INotificationEvent)";
		Log.enter(method, new Object[] { event });
		try {
			boolean bDebug = Log.isDebugEnabled();
			if (!event.isEnabled()) {
				if (bDebug) {
					Log.debug("Event is disabled, do nothing");
				}
				return;
			}
			ISubscrListFactory subscrFactory = getSubscrListFactory();
			ISubscrList subscrList = subscrFactory.getSubscrList();
			if (subscrList == null) {
				if (bDebug) {
					Log.debug("Could not find subscription list");
				}
				return;
			}
			Iterator foundSubscrs = getSubscriptions(subscrList, event);
			if (foundSubscrs == null || !foundSubscrs.hasNext()) {
				if (Log.isDebugEnabled()) {
					Log.debug("Could not find subscriptions by id:[{0}]", new Object[] { event.getId()});
				}
				return;
			}
			while (foundSubscrs.hasNext()) {
				ISubscription subscr = (ISubscription) foundSubscrs.next();
				if (bDebug) {
					Log.debug("Subscription with id [{0}] has been found", new Object[] { subscr.getId()});
				}
				if (!isValid(subscr)) {
					continue;
				}
				ISubscrFilter filter = subscr.getFilter();
				if (!filter.accept(event)) {
					if (bDebug) {
						Log.debug("Subscription filter did not accept event");
					}
					continue;
				}
				ISubscrRecipients inclUsers = subscr.getRecipients(event);
				ISubscrRecipients exclUsers = subscr.getExcludedRecipients(event);
				Collection userIds = (inclUsers != null ? inclUsers.getUserIds(event, exclUsers) : null);
				if (userIds == null || userIds.isEmpty()) {
					if (bDebug) {
						Log.debug("Recipients list is empty");
					}
					continue;
				}
				INotifSender sender = getSender();
				fireOnBeforeSend(subscr, event);
				event.setAttribute(INotifEvent.EVENT_ID, subscr.getId());
				sender.send(userIds, subscr, event);
			}
		} catch (Exception e) {
			NotifException ownEx = new NotifException(IResourceKeys.CANT_PROCESS_EVENT, e);
			Log.fatal(ownEx, method);
			throw ownEx;
		} finally {
			Log.exit(method);
		}
	}

	protected boolean isValid(ISubscription subscr) {
		if (!subscr.isEnabled()) {
			Log.fatal(IResourceKeys.DISABLED_SUBSCR, new Object[] { subscr.getId()});
			return false;
		}
		if (subscr instanceof IPersistSubscription) {
			IPersistSubscription persistSubscr = (IPersistSubscription) subscr;
			if (!persistSubscr.isActive()) {
				IApplication app = persistSubscr.getApplication();
				Log.fatal(IResourceKeys.INACTIVE_SUBSCR, new Object[] { subscr.getId(), app != null ? app.getId() : null });
				return false;
			}
		}
		return true;
	}

	protected abstract Iterator getSubscriptions(ISubscrList subscrList, INotifEvent event) throws SubscrException;

	protected abstract void fireOnBeforeSend(ISubscription subscr, INotifEvent event) throws NSBaseException;

	protected abstract ISubscrListFactory getSubscrListFactory() throws SubscrException;

	public boolean isSubscrAvailable(String subscrId) throws SubscrException {
		ISubscrListFactory factory = getSubscrListFactory();
		if (factory == null) {
			return false;
		}
		ISubscrList subscrList = factory.getSubscrList();
		return subscrList.getSubscription(subscrId) != null;
	}

	protected INotifSender getSender() {
		if (m_sender == null) {
			m_sender = new NotifSender();
		}
		return m_sender;
	}

	protected INotifSender m_sender;
	protected ClassLoader m_classLoader;
}
