package com.sap.caf.rt.services.localization;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.ejb.CreateException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.Context;
import javax.naming.InitialContext;

import com.sap.caf.rt.exception.CAFFindException;
import com.sap.caf.rt.exception.CAFUpdateException;
import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.srv.IDataContainerBean;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * @ejbLocal <{com.sap.caf.rt.services.localization.LocalizationServiceLocal}>
 * @ejbLocalHome <{com.sap.caf.rt.services.localization.LocalizationServiceLocalHome}>
 * @stateless 
 * @transactionType Container
 */
public class LocalizationServiceBean implements SessionBean, ILocalizationService {

	protected SessionContext sessionContext;

	/* Logging properties for this class */
	private static final String APPLICATION = LocalizationServiceBean.class.getName();
	private static final String jARMRequest = LocalizationResourceAccessor.JARM_REQUEST + APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);


	public void ejbRemove() {
	}

	public void ejbActivate() {
	}

	public void ejbPassivate() {
	}

	public void setSessionContext(SessionContext context) {
		sessionContext = context;	
	}

	/**
	 * Create Method.
	 */
	public void ejbCreate() throws CreateException {
	}

	public String create(LocalizationDataBean localizationBean) throws ServiceException {
		final String method = "create(LocalizationDataBean)";
		CAFPublicLogger.entering(getUser(), jARMRequest, method, logger, 1);
		try {
//			localizationBean.validateData();
			return LocalizationDataAccessor.createObject(localizationBean);			
		} finally {
			CAFPublicLogger.exiting(getUser(), jARMRequest, method, logger, 1);
		}
	}
	
	public LocalizationDataBean findByPK(String pk) throws ServiceException {
		final String method = "findByPK(String)";
		CAFPublicLogger.entering(getUser(), jARMRequest, method, logger, 1);
		try {
			return LocalizationDataAccessor.selectObjectByPk(pk);
			
		} finally {
			CAFPublicLogger.exiting(getUser(), jARMRequest, method, logger, 1);
		}
	}

	public void delete(LocalizationDataBean localizationBean) throws ServiceException {
		final String method = "delete(LocalizationDataBean)";
		CAFPublicLogger.entering(getUser(), jARMRequest, method, logger, 1);
		try {
			String pk = (String) localizationBean.getProperty(LocalizationDataBean.PROP_OBJECTKEY);
			IDataContainerBean dbObj = null;
			if (! isEmpty(pk)) {
				try {
					dbObj = LocalizationDataAccessor.selectObjectByPk(pk);
				} catch (ServiceException e) {
					CAFPublicLogger.traceThrowable(Severity.INFO, logger, method, e);
				}
			}
			if (dbObj != null) {
				LocalizationDataAccessor.deleteObject(localizationBean);
			}		
		} finally {
			CAFPublicLogger.exiting(getUser(), jARMRequest, method, logger, 1);
		}
	}

	public void update(LocalizationDataBean localizationBean) throws ServiceException {
		final String method = "update(LocalizationDataBean)";
		CAFPublicLogger.entering(getUser(), jARMRequest, method, logger, 1);
		try {
			String pk = (String) localizationBean.getProperty(LocalizationDataBean.PROP_OBJECTKEY);
			IDataContainerBean dbObj = null;
			if (! isEmpty(pk)) {
				try {
					dbObj = LocalizationDataAccessor.selectObjectByPk(pk);
				} catch (ServiceException e) {
					CAFPublicLogger.traceThrowable(Severity.INFO, logger, method, e);
				}
			}
			
			String text = (String) localizationBean.getProperty(LocalizationDataBean.PROP_TEXT);
			if (dbObj == null) {
				if(text!=null && !"".equals(text.trim())) {
					create(localizationBean);
				}
			} else {
//				localizationBean.validateData();
				if(text==null || "".equals(text.trim())) {
					LocalizationDataAccessor.deleteObject(localizationBean);
				} else {
					LocalizationDataAccessor.updateObject(localizationBean);
				}
			}
		} catch (ServiceException e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
			throw e;
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
			throw new CAFUpdateException("BO_UPDATE", e);
		} finally {
			CAFPublicLogger.exiting(getUser(), jARMRequest, method, logger, 1);
		}
	}

	/* (non-Javadoc)
	 * @see com.sap.caf.rt.services.localization.ILocalizationService#findByResourceName(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.util.Locale)
	 */
	public LocalizationDataBean findByResourceName(
			String serviceModuleName, String aspectName, String queryName, 
			String actionName, String paramName, String locStr) 
			throws ServiceException {
		final String method = "findByResourceName(String, String, String, String, String, String)";
		CAFPublicLogger.entering(getUser(), jARMRequest, method, logger, 1);
		try {
			if (isEmpty(paramName) || "*".equals(paramName)) {	// need all attributes (parameters)
				throw new IllegalArgumentException("Name of Attribute (Parameter) mustn't be empty or '*'");
			}
			String resourceName = LocalizationResourceAccessor.getResourceName(
					serviceModuleName, aspectName, queryName, actionName, paramName);
			// find customized texts for the resource FOR THE LOCALE ONLY
			Collection dbBeans = LocalizationDataAccessor.selectObjectByResName(
					resourceName, locStr);
			if (dbBeans.size() == 1) {	// need one attribute (parameter) and we have one customized text for it
				// return customized text for the attribute
				return (LocalizationDataBean) dbBeans.iterator().next();
				
			} else if (dbBeans.isEmpty()) {	// need one attribute (parameter) and we have no customized texts
				// return standard text for the attribute
				return getSingleLocalizationDataForLocale(
						serviceModuleName, resourceName, locStr);
			} else {	// need one attribute (parameter) and we have no any texts for it at all
				throw new RuntimeException("There are more then one resource named " + 
						resourceName + " in locale " + locStr);
			}			
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
			throw new CAFFindException("BO_READ", e);
		} finally {
			CAFPublicLogger.exiting(getUser(), jARMRequest, method, logger, 1);
		}
	}
	
	private String getProp(IDataContainerBean bean, String propName) {
		Object o = bean.getProperty(propName);
		return o == null ? null : o.toString();
	}
	
	private boolean isEmpty(String l) {
		return l == null || l.equals("");
	}
	
	private String getUser() {
		return sessionContext.getCallerPrincipal().getName();
	}

	/* (non-Javadoc)
	 * @see com.sap.caf.rt.services.localization.ILocalizationService#findDefaultByResourceName(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public Collection findDefaultByResourceName(
		String serviceModuleName, String aspectName, String queryName, String actionName,
		Collection paramNames, String locStr)
		throws ServiceException
	{
		final String method = "findDefaultByResourceName(String, String, String, String, Collection, String)";
		CAFPublicLogger.entering(getUser(), jARMRequest, method, logger, 1);
		try {
			String resourceName = LocalizationResourceAccessor.getResourceName(
					serviceModuleName, aspectName, queryName, actionName, null);
			// get standard Cool texts for the resources FOR THE LOCALE ONLY
			List localizedTexts = getLocalizationDataForLocale(
					serviceModuleName, resourceName, locStr, paramNames);

			List result = new ArrayList(localizedTexts.size() / 2);
			// make substitution standard texts with customized texts
			for (Iterator i = localizedTexts.iterator(); i.hasNext(); ) {
				String fullResourceName = (String) i.next();
				String text = (String) i.next();
				result.add(new LocalizationDataBean(
						null, fullResourceName, text, locStr, 
						LocalizationDataAccessor.getTypeName(fullResourceName)));
			}
			// sorting by attribute name
			Collections.sort(result, new LocalizationDataBean.LocComparator());
			return result;
						
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
			throw new CAFFindException("BO_READ", e);
		} finally {
			CAFPublicLogger.exiting(getUser(), jARMRequest, method, logger, 1);
		}
	}

	/* (non-Javadoc)
	 * @see com.sap.caf.rt.services.localization.ILocalizationService#findDefaultLocalizationData(com.sap.caf.rt.services.localization.LocalizationDataBean)
	 */
	public LocalizationDataBean findDefaultLocalizationData(LocalizationDataBean bean)
			throws ServiceException
	{
		final String method = "findDefaultLocalizationData(LocalizationDataBean)";
		CAFPublicLogger.entering(getUser(), jARMRequest, method, logger, 1);
		try {
			if (bean == null) {
				throw new IllegalArgumentException("Passed Localization data bean is null");
			}
			// return standard text for the attribute
			String resourceName = bean.getResourceName();
			return getSingleLocalizationDataForLocale(
					LocalizationResourceAccessor.getServiceModuleName(resourceName), 
					resourceName, bean.getLanguage());
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
			throw new CAFFindException("BO_READ", e);
		} finally {
			CAFPublicLogger.exiting(getUser(), jARMRequest, method, logger, 1);
		}
	}
	
	/* (non-Javadoc)
	 * @see com.sap.caf.rt.services.localization.ILocalizationService#findByResourceName(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.util.Collection, java.lang.String)
	 */
	public Collection findByResourceName(
			String serviceModuleName,	String aspectName, String queryName, String actionName, 
			Collection paramNames, String locStr)
			throws ServiceException
	{
		final String method = "findByResourceName(String, String, String, String, Collection, String)";
		CAFPublicLogger.entering(getUser(), jARMRequest, method, logger, 1);
		try {
			String resourceName = LocalizationResourceAccessor.getResourceName(
					serviceModuleName, aspectName, queryName, actionName, null);
			// find customized texts for the resource FOR THE LOCALE ONLY
			Collection dbBeans = LocalizationDataAccessor.selectObjectByResName(
					resourceName, locStr);
			// make Resource Name->Bean map
			Map resourceName2dbBean = new HashMap(dbBeans.size());
			for (Iterator i = dbBeans.iterator(); i.hasNext(); ) {
				LocalizationDataBean bean = (LocalizationDataBean) i.next();
				resourceName2dbBean.put(bean.getResourceName(), bean);
			}
			// get standard Cool texts for the resources FOR THE LOCALE ONLY
			List localizedTexts = getLocalizationDataForLocale(
					serviceModuleName, resourceName, locStr, paramNames);

			List result = new ArrayList((localizedTexts.size() / 2) + resourceName2dbBean.size());
			// make substitution standard texts with customized texts
			for (Iterator i = localizedTexts.iterator(); i.hasNext(); ) {
				String fullResourceName = (String) i.next();
				String text = (String) i.next();
				LocalizationDataBean dbBean = 
						(LocalizationDataBean) resourceName2dbBean.remove(fullResourceName);
				IDataContainerBean resultBean = (dbBean != null) ? dbBean : 
						new LocalizationDataBean(
								null, fullResourceName, text, locStr, 
								LocalizationDataAccessor.getTypeName(fullResourceName));
				result.add(resultBean);
			}
			result.addAll(resourceName2dbBean.values());
				
			// sorting by attribute name
			Collections.sort(result, new LocalizationDataBean.LocComparator());
			return result;
			
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
			throw new CAFFindException("BO_READ", e);
		} finally {
			CAFPublicLogger.exiting(getUser(), jARMRequest, method, logger, 1);
		}
	}

	private LocalizationDataBean getSingleLocalizationDataForLocale(
			String serviceModuleName, String resourceName, String locStr)
			throws Exception
	{
		String localizedParam = LocalizationResourceAccessor.getLocalizedText(
				LocalizationDataAccessor.getLocaleByStringCode(locStr), resourceName, 
				serviceModuleName, getModuleClassLoader(serviceModuleName));
		return new LocalizationDataBean(null, resourceName, localizedParam, locStr, 
				LocalizationDataAccessor.getTypeName(resourceName));
	}
	
	private List getLocalizationDataForLocale(
			String serviceModuleName, String resourceName, String locStr,	
			Collection paramNames)
			throws Exception
	{
		// get standard Cool texts for the resources FOR THE LOCALE ONLY
		List localizedTexts = LocalizationResourceAccessor.getLocalizedTexts(
				LocalizationDataAccessor.getLocaleByStringCode(locStr), resourceName,
				serviceModuleName, getModuleClassLoader(serviceModuleName));
		if (localizedTexts.isEmpty() && paramNames != null) {
			localizedTexts = new ArrayList(paramNames.size() * 2);
			for (Iterator i = paramNames.iterator(); i.hasNext(); ) {
				localizedTexts.add(LocalizationResourceAccessor.completeResourceName(
						resourceName, (String) i.next()));
				localizedTexts.add(null);
			}
		}
		return localizedTexts;
	}
	
	static ClassLoader getModuleClassLoader(String serviceModuleName) throws Exception
	{
		final String method = "getModuleClassLoader(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
			Context ctx = new InitialContext();
			Object lh = ctx.lookup(getServiceJNDI_KEY(serviceModuleName));
			return lh.getClass().getClassLoader();
			
		} finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
	}

	/**
	 * Return JNDI key of lookup service for specified Service Module
	 * @param serviceModuleName
	 * @return
	 */
	static String getServiceJNDI_KEY(String serviceModuleName)
	{
		return "localejbs/" + serviceModuleName;
	}

}
