/*
 * Created on 24.05.2004
 */
package com.sap.caf.rt.services.localization;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import com.sap.caf.rt.srv.IDataContainerBean;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.localization.ResourceAccessor;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * @author pavel_henrykhsen
 */
public class LocalizationResourceAccessor extends ResourceAccessor {

	/* Logging properties for this class */
	static final String JARM_REQUEST = "CAF:RT:oal";
	private static final String APPLICATION = LocalizationResourceAccessor.class.getName();
	private static final String jARMRequest = JARM_REQUEST + APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);
	
	static final char LOCALIZED_KEY_SPLITTER = '@'; 
	static final String LOCALIZED_KEY_SPLITTER_STR = "@";
	
	private static final int METHOD_TYPE_QUERY = 0;
	private static final int METHOD_TYPE_ACTION = 1;

	private ClassLoader cl;
	private HashMap bundles = new HashMap(2);
	
	private static HashMap accessors = new HashMap(10);
	

	private LocalizationResourceAccessor(String resourceBundleName, ClassLoader cl)
	{
		super(resourceBundleName);
		this.cl = cl;
	}

	private static LocalizationResourceAccessor getResourceAccessor(
			String serviceModuleName, ClassLoader cl)
			throws Exception	
	{
		return getResourceAccessorInt(
				getBundleNameByServiceModule(serviceModuleName), cl);
	}
	 
	private static synchronized LocalizationResourceAccessor getResourceAccessorInt(
			String bundleName, ClassLoader classLoader)
			throws Exception
	{
		final String method = "getResourceAccessor(String, ClassLoader)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
			if (bundleName == null || classLoader == null) {
				throw new IllegalArgumentException("One of passed parameters is null: LocalizationResourceAccessor.getResourceAccessorInt(String bundleName, ClassLoader classLoader)");
			}
			String key = bundleName + ":" + System.identityHashCode(classLoader);
			LocalizationResourceAccessor ra = (LocalizationResourceAccessor) accessors.get(key);
			if (ra == null) {
				ra = new LocalizationResourceAccessor(bundleName, classLoader);
				accessors.put(key, ra);
			}
			return ra;
			
		} finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
	}
	
	private static String getBundleNameByServiceModule(String moduleName)
	{
		if (moduleName != null) {
			int lastIndex = moduleName.lastIndexOf('/');
			return moduleName.substring((lastIndex < 0) ? 0 : lastIndex + 1);
		}
		return null;
	}
	
	public static String getResourceName(
			String serviceModuleName, String aspectName, String queryName, String actionName, 
			String parameterName)
			throws Exception
	{			
		StringBuffer result = new StringBuffer(50);
		if (isEmpty(serviceModuleName)) {
			throw new IllegalArgumentException("Service Module name must be not null.");
		}
		result.append(serviceModuleName).append(LOCALIZED_KEY_SPLITTER);
		
		boolean subModuleSetted = true;
		if (! isEmpty(aspectName)) {
			result.append(aspectName).append(LOCALIZED_KEY_SPLITTER);
		} else if (! isEmpty(queryName)) { 
			result.append(queryName).append(LOCALIZED_KEY_SPLITTER).
					append(METHOD_TYPE_QUERY).append(LOCALIZED_KEY_SPLITTER);
		} else if (! isEmpty(actionName)) {
			result.append(actionName).append(LOCALIZED_KEY_SPLITTER).
					append(METHOD_TYPE_ACTION).append(LOCALIZED_KEY_SPLITTER);
		} else {
			subModuleSetted = false;
		}
		if (! isEmpty(parameterName)) {
			if (! subModuleSetted) {
				throw new IllegalArgumentException("Aspect, Query or Action name must be specified along with property/parameter name.");
			}
			result.append(parameterName);
		}
		return result.toString();
	}
	
	static String getServiceModuleName(String resourceName)
	{
		if (resourceName != null) {
			int firstSplitterIdx = resourceName.indexOf(LOCALIZED_KEY_SPLITTER);
			return (firstSplitterIdx >= 0) ? resourceName.substring(0, firstSplitterIdx) : 
					resourceName;
		}
		return null;
	}
	
	static String getAttributeName(String resourceName)
	{
		if (resourceName != null) {
			int ind = resourceName.lastIndexOf(LOCALIZED_KEY_SPLITTER);
			return (ind >= 0) ? resourceName.substring(ind + 1) : "";
		}
		return null;
	}
	
	static String completeResourceName(String resourceNamePrefix, String paramName)
			throws Exception
	{
		if (paramName == null || resourceNamePrefix == null) {
			throw new IllegalArgumentException("One of passed parameters are null");
		}
		StringBuffer sb = new StringBuffer(
				resourceNamePrefix.length() + paramName.length());
		return sb.append(resourceNamePrefix).append(paramName).toString();
	}
	
	public static String getCustomizedText(
			Locale locale, String patternKey, String moduleName, ClassLoader cl)
			throws Exception
	{
		LocalizationResourceAccessor accessor = getResourceAccessor(moduleName, cl);
		return accessor.getMessageText(locale, patternKey);
	}
	
	private String getValueFromLocalizationService(Locale locale, String patternKey)
	{
		final String method = "getValueFromLocalizationService(Locale, String)"; 
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		
		Collection resources;
		String result = null;
		try {
			resources = LocalizationDataAccessor.selectObjectByResName(
					patternKey, locale.toString(), true);
			int size = resources.size(); 
			if (size > 1) {
				throw new RuntimeException("There are more then one resource named " + patternKey + " in locale " + locale);
			}
			if (size == 1) {
				// Result is the localized text from the only LocalizationDataBean in the search result.
				result = (String) ((IDataContainerBean) resources.iterator().next()).getProperty(
						LocalizationDataBean.PROP_TEXT);
			}
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
		}
		CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		return result;
	}
	
	public String getMessageText(Locale locale, String patternKey)
	{
		final String method = "getMessageText(Locale, String)"; 
		//CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);
		
		String message = null;
		try {
			if (locale == null) { 
				locale = Locale.getDefault();
			}
			message = getValueFromLocalizationService(locale, patternKey);
			
			if (message == null && patternKey != null) {
				ResourceBundle rb = getResourceBundle(locale);
				if (rb != null) {
					message = rb.getString(patternKey);
				}
			}
		} catch(MissingResourceException mre) {
			//$JL-EXC$
			//skip this exception, it means that resource doesn't exist			
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
		}
		//CAFPublicLogger.exiting(null, jARMRequest, method, logger, 1);
		return message;
	}
	
	public static String getLocalizedText(
			Locale locale, String patternKey, String moduleName, ClassLoader cl)
			throws Exception
	{
		LocalizationResourceAccessor accessor = getResourceAccessor(moduleName, cl);
		return accessor.getMessageTextForLocale(locale, patternKey);
	}
	
	public static List getLocalizedTexts(
			Locale locale, String patternKey, String moduleName, ClassLoader cl)
			throws Exception
	{
		LocalizationResourceAccessor accessor = getResourceAccessor(moduleName, cl);
		return accessor.getMessageTextsForLocale(locale, patternKey);
	}
	
	public String getMessageTextForLocale(Locale locale, String patternKey) 
	{
		final String method = "getMessageTextForLocale(Locale, String)"; 
		//CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);

		String result = null;
		try {
			if (locale == null) { 
				locale = Locale.getDefault();
			}
			ResourceBundle rb = getResourceBundle(locale);
			if (rb != null && locale.equals(rb.getLocale())) {	// only for specified locale
				result = rb.getString(patternKey);
			}
		} catch(MissingResourceException mre) {
			//$JL-EXC$
			//skip this exception, it means that resource doesn't exist
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
		}
		//CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		return result;
	}
	
	public List getMessageTextsForLocale(Locale locale, String patternKey) 
	{
		final String method = "getMessageTextsForLocale(Locale, String)"; 
		//CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);
		
		List result = Collections.EMPTY_LIST;				
		try {
			if (locale == null) { 
				locale = Locale.getDefault();
			}
			if (patternKey != null) {
				ResourceBundle rb = getResourceBundle(locale);
				if (rb != null && locale.equals(rb.getLocale())) {	// only for specified locale
					Enumeration bundleKeys = rb.getKeys();
					result = new ArrayList();
					while (bundleKeys.hasMoreElements()) {
						String key = (String) bundleKeys.nextElement();
						if (key.startsWith(patternKey)) {
							result.add(key);
							result.add(rb.getString(key));
						}
					}
				}
			}
		} catch(MissingResourceException mre) {
			//$JL-EXC$
			//skip this exception, it means that resource doesn't exist
		} catch (Exception e) {
			CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
		}
		//CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		return result;
	}
	
	private ResourceBundle getResourceBundle(Locale locale) throws Exception
	{
		final String method = "getResourceBundle(Locale)";
		//CAFPublicLogger.entering(null, jARMRequest, method, logger);
		
		ResourceBundle rb = null;
		try {
			synchronized (bundles) {
				rb = (ResourceBundle) bundles.get(locale);
				if (rb == null && getResourceBundleName() != null && cl != null) {
					rb = ResourceBundle.getBundle(getResourceBundleName(), locale, cl);
					if (rb != null) {
						bundles.put(locale, rb);
					}
				}
			}
		} finally {
			//CAFPublicLogger.exiting(null, jARMRequest, method, logger);	
		}
		return rb;
	}
	
	private static boolean isEmpty(String str)
	{
		return (str == null || str.trim().length() == 0);
	}
		
}
