/*
 * Created on 04.06.2004
 */
package com.sap.caf.rt.services.localization;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.StringTokenizer;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

import com.sap.caf.rt.exception.CAFCreateException;
import com.sap.caf.rt.exception.CAFDeleteException;
import com.sap.caf.rt.exception.CAFFindException;
import com.sap.caf.rt.exception.CAFRetrieveException;
import com.sap.caf.rt.exception.CAFUpdateException;
import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.srv.IDataContainerBean;
import com.sap.s2x.tools.S2XGUID;

/**
 * @author Siarhei_Pisarenka
 */
public class LocalizationDataAccessor {

	private static DataSource dataSource;
	
	private static final String JNDI_NAME = "jdbc/notx/SAP/CAF_RT";

	private static final String TABLE_NAME = "CAF_RT_LOCALIZ";
	
	private static final String DELETE = "delete from " + TABLE_NAME + " where \"OBJECTKEY\" = ?"; 
	private static final String INSERT = "insert into " + TABLE_NAME + " (\"OBJECTKEY\", \"RESOURCE_NAME\", \"TEXT\", \"LANG\") values(?, ?, ?, ?)"; 
	private static final String SELECT_PK = "select \"OBJECTKEY\", \"RESOURCE_NAME\", \"TEXT\", \"LANG\" from " + TABLE_NAME + " where \"OBJECTKEY\" = ?"; 
	private static final String SELECT_RES = "select \"OBJECTKEY\", \"RESOURCE_NAME\", \"TEXT\", \"LANG\" from " + TABLE_NAME + " where \"RESOURCE_NAME\" like ? and \"LANG\" = ?"; 
	private static final String SELECT_LANG = "select \"OBJECTKEY\", \"RESOURCE_NAME\", \"TEXT\", \"LANG\" from " + TABLE_NAME + " where \"LANG\" = ?"; 
	private static final String SELECT_DEF_RES = "select \"OBJECTKEY\", \"RESOURCE_NAME\", \"TEXT\", \"LANG\" from " + TABLE_NAME + " where \"RESOURCE_NAME\" like ? and \"LANG\" is null"; 
	private static final String SELECT_DEF_LANG = "select \"OBJECTKEY\", \"RESOURCE_NAME\", \"TEXT\", \"LANG\" from " + TABLE_NAME + " where \"LANG\" is null"; 
	private static final String UPDATE = "update " + TABLE_NAME + " set \"RESOURCE_NAME\" = ?, \"TEXT\" = ?, \"LANG\" = ? where \"OBJECTKEY\" = ?";
	
	/* Locale to resource set collection */
	private static Map cacheObject = new LinkedHashMap(129);

	private static Map localeByStringCashe = new HashMap();

	private static Connection getConnection() throws SQLException {
		return getDataSource().getConnection();
	}

	private static DataSource getDataSource() {
		final String method = "getDataSource()";

		if (dataSource == null) {
			try {
				InitialContext context = new InitialContext();
				dataSource = (DataSource)context.lookup(JNDI_NAME);
			} catch (NamingException e) {
				dataSource = null;
			}
		}
		return dataSource;
	}	

	private static PreparedStatement getDeleteStatement(Connection connection)
		throws SQLException {
		return  getStatement(DELETE, connection);
	}

	private static PreparedStatement getInsertStatement(Connection connection)
		throws SQLException {
		return  getStatement(INSERT, connection);
	}

	private static PreparedStatement getSelectPkStatement(Connection connection)
		throws SQLException {
		return  getStatement(SELECT_PK, connection);
	}

	private static PreparedStatement getSelectResStatement(Connection connection)
		throws SQLException {
		return  getStatement(SELECT_RES, connection);
	}

	private static PreparedStatement getSelectDefaultLang(Connection connection)
		throws SQLException {
		return  getStatement(SELECT_DEF_LANG, connection);
	}

	private static PreparedStatement getSelectLanguage(Connection connection)
		throws SQLException {
		return  getStatement(SELECT_LANG, connection);
	}

	private static PreparedStatement getSelectDefaultResources(Connection connection)
		throws SQLException {
		return  getStatement(SELECT_DEF_RES, connection);
	}

	private static PreparedStatement getStatement(String sql, Connection connection )
		throws SQLException {
		return connection.prepareStatement(sql);
	}
	
	private static PreparedStatement getUpdateStatement(Connection connection)
		throws SQLException {
		return  getStatement(UPDATE, connection);
	}

	public static String createObject(IDataContainerBean bean) throws ServiceException {
		String guid;
		try {
			guid = S2XGUID.getGUID();
			Connection conn = getConnection();
			try {
				PreparedStatement pst = getInsertStatement(conn);
				try {
					String resName = getProp(bean, LocalizationDataBean.PROP_RES_NAME);
					pst.setString(1, guid);
					pst.setString(2, resName);
					
					String text = getProp(bean, LocalizationDataBean.PROP_TEXT);
					if(text!=null) {
						text = text.trim();
					}
					if("".equals(text)) text = null;
					pst.setString(3, text);
					
					String lang = getProp(bean, LocalizationDataBean.PROP_LANGUAGE);
					pst.setString(4, isEmpty(lang) ? null : lang);
					pst.execute();
					String typeName = getProp(bean, LocalizationDataBean.PROP_CUSTOM_TYPE);
					// TODO uncomment to support CustomEnumType service from here
/*					if (! isEmpty(typeName)) { 
						ITypeLinkage typeService = PersistedValueServices.getTypeLinkageInstance();
						typeService.createTypeLink(resName, typeName);
					}*/
				} finally {
					pst.close();
				}
			} finally {
				conn.close();
			}
			clearCache();
		} catch (Exception e) {
			throw new CAFCreateException("BO_CREATE", e);
		}
		// backward compatibility
		bean.setProperty(LocalizationDataBean.PROP_OBJECTKEY, guid);
		return guid;
	} 
	
	public static void deleteObject(IDataContainerBean bean) throws ServiceException {
		try {
			Connection conn = getConnection();
			try {
				PreparedStatement pst = getDeleteStatement(conn);
				try {
					pst.setString(1, getProp(bean, LocalizationDataBean.PROP_OBJECTKEY));
					pst.execute();
				} finally {
					pst.close();
				}
			} finally {
				conn.close();
			}
			clearCache();
			String typeName = getProp(bean, LocalizationDataBean.PROP_CUSTOM_TYPE);
			// TODO uncomment to support CustomEnumType service from here
/*			if (!isEmpty(typeName)) { 
				ITypeLinkage typeService = PersistedValueServices.getTypeLinkageInstance();
				typeService.deleteTypeLink((String) bean.getProperty(LocalizationDataBean.PROP_RES_NAME));
			}*/
		} catch (Exception e) {
			throw new CAFDeleteException("BO_DELETE", e);
		}
	}

	public static String getTypeName(String resName) throws ServiceException {
		// TODO uncomment to support CustomEnumType service from here
/*		ITypeLinkage typeService = PersistedValueServices.getTypeLinkageInstance();
		Map typeNames = typeService.getTypeNames(resName);
		return typeNames.size() > 0 ? (String) typeNames.values().iterator().next() : null;*/
		return null;
	}
	
	public static LocalizationDataBean selectObjectByPk(String pk) throws ServiceException {
		try {
			Connection conn = getConnection();
			try {
				PreparedStatement pst = getSelectPkStatement(conn);
				try {
					pst.setString(1, pk);
					ResultSet rs = pst.executeQuery();
					try {
						if (rs.next()) {
							String resName = rs.getString(1);
							// SAP SQL JDBC API doesn't allow to write empty strings into tables, 
							// so <null> in the situation is empty string
							String resText = rs.getString(3);
							if (resText == null) {
								resText = "";
							}
							LocalizationDataBean dataBean = new LocalizationDataBean(
									resName, rs.getString(2), resText, rs.getString(4), 
									getTypeName(resName));
							if (rs.next()) {
								throw new RuntimeException("Primary key (OBJECTKEY) " + pk + " is not unique in " + TABLE_NAME);
							}
							return dataBean;
							
						} else {
							throw new CAFRetrieveException("BO_READ");
						}
					} finally {
						rs.close();
					}
				} finally {
					pst.close();
				}
			} finally {
				conn.close();
			}
		} catch (ServiceException se) {
			throw se;
		} catch (Exception e) {
			throw new CAFRetrieveException("BO_READ", e);
		}
	}

	public static Collection selectObjectByResName(String resName, String locale)
			throws ServiceException {
		return selectObjectByResName(resName, locale, false);
	}

	static Locale getLocaleByStringCode(String locale) {
		Locale localeObj = (Locale) localeByStringCashe.get(locale);
		if(localeObj==null) {
			String values[] = new String[] { "", "", "" };
			int i = 0;
			for (StringTokenizer st = new StringTokenizer(locale, "_", false); st.hasMoreTokens(); i++) {
				values[i] = st.nextToken();
			}
			localeObj = new Locale(values[0], values[1], values[2]);
			localeByStringCashe.put(locale, localeObj);
		}
		return localeObj;
	}
	
	private static Map readEntireLocale(String locale) throws SQLException, ServiceException {
		Connection conn = getConnection();
		try {
			String lang = isEmpty(locale) ? null : locale;
			PreparedStatement pst;
			if (lang == null) {
				pst = getSelectDefaultLang(conn);
			} else {
				pst = getSelectLanguage(conn);
				pst.setString(1, lang);
			}
			try {
				ResultSet rs = pst.executeQuery();
				try {
					Map result = new HashMap(33);
					while (rs.next()) {
						String resName = rs.getString(2);
						// SAP SQL JDBC API doesn't allow to write empty strings into tables, 
						// so <null> in the situation is empty string
						String resText = rs.getString(3);
						if (resText == null) {
							resText = "";
						}
						result.put(resName, new LocalizationDataBean(rs.getString(1), 
								resName, resText, rs.getString(4), getTypeName(resName)));
					} 
					return result;
				} finally {
					rs.close();
				}
			} finally {
				pst.close();
			}
		} finally {
			conn.close();
		}
	}

	/**
	 * 
	 * @param resName
	 * @param locale
	 * @param generalizeLocale
	 * @return
	 * @throws NullPointerException if resName is null.
	 * @throws ServiceException in other erroneous situations
	 */
	public static synchronized Collection selectObjectByResName(String resName, String locale, boolean generalizeLocale) 
				throws ServiceException {
		Map result = new HashMap(10);
		try {
			List localeNames = null; 
			if (generalizeLocale) {
				localeNames = calculateLocaleNames(getLocaleByStringCode(locale));
			} else {
				localeNames = new ArrayList(1);
				localeNames.add(locale);
			}
			int size = localeNames.size();
			
			boolean likeCriteria = (resName != null && resName.endsWith(
					LocalizationResourceAccessor.LOCALIZED_KEY_SPLITTER_STR));
			// Such a way of processing languages simpifies and speeds up the creation of the result collection, 
			// but it consumes some external memory for the default one that may never be utilized.
			for (int i = 0; i < size; i++) {
				String loc = (String) localeNames.get(i);
				Map resourceSet = (Map) cacheObject.get(loc);
				if (resourceSet == null) {
					resourceSet = readEntireLocale(loc);
					cacheObject.put(loc, resourceSet);
				}
				for (Iterator j = resourceSet.entrySet().iterator(); j.hasNext(); ) {
					Map.Entry entry = (Map.Entry) j.next();
					String entryKey = (String) entry.getKey();
					boolean found = likeCriteria ?
							entryKey.startsWith(resName) : 
							entryKey.equals(resName);
					if (found) {
						// Making defensive copy of cache object
						LocalizationDataBean ldc = new LocalizationDataBean(
								(LocalizationDataBean) entry.getValue());
						result.put(ldc.getResourceName(), ldc);
					}
				}
			} 
		} catch (Exception e) {
			throw new CAFFindException("BO_READ", e);
		}
		return result.values();
	}
	
	protected static List calculateLocaleNames(Locale locale) {
		final List result = new ArrayList(4);
		final String language = locale.getLanguage();
		final int languageLength = language.length();
		final String country = locale.getCountry();
		final int countryLength = country.length();
		result.add(null); // Add root locale
		result.add(language);
		if (languageLength > 0 && countryLength > 0) {
			result.add(language + "_" + country);
		}
		return result;
	}

	public static void updateObject(IDataContainerBean bean) throws ServiceException {
		try {
			Connection conn = getConnection();
			try {
				PreparedStatement pst = getUpdateStatement(conn);
				try {
					String resName = getProp(bean, LocalizationDataBean.PROP_RES_NAME);
					pst.setString(1, resName);
					
					String text = getProp(bean, LocalizationDataBean.PROP_TEXT);
					if(text!=null) {
						text = text.trim();
					}
					if("".equals(text)) text = null;
					pst.setString(2, text);
					
					String lang = getProp(bean, LocalizationDataBean.PROP_LANGUAGE);
					pst.setString(3, isEmpty(lang) ? null : lang);
					pst.setString(4, getProp(bean, LocalizationDataBean.PROP_OBJECTKEY));
					pst.execute();
					String typeName = getProp(bean, LocalizationDataBean.PROP_CUSTOM_TYPE);
					// TODO uncomment to support CustomEnumType service from here
/*					ITypeLinkage typeService = PersistedValueServices.getTypeLinkageInstance();
					if (!isEmpty(typeName)) { 
						typeService.updateTypeLink(resName, typeName);
					} else {
						typeService.deleteTypeLink(resName);
					}*/
				} finally {
					pst.close();
				}
			} finally {
				conn.close();
			}
			clearCache();
		} catch (Exception e) {
			throw new CAFUpdateException("BO_UPDATE", e);
		}
	}
	
	private static String getProp(IDataContainerBean bean, String propName) {
		Object o = bean.getProperty(propName);
		return o == null ? null : o.toString();
	}
	
	private static boolean isEmpty(String s) {
		return s == null || s.trim().length() == 0;
	}
	
	public static void clearCache() {
		cacheObject.clear();
	}
}
