package com.sap.caf.rt.metamodel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Vector;

import com.sap.caf.metamodel.Attribute;
import com.sap.caf.metamodel.AttributeImpl;
import com.sap.caf.metamodel.BusinessEntityInterface;
import com.sap.caf.metamodel.DataObject;
import com.sap.caf.rt.exception.DataAccessException;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.dictionary.runtime.IDataType;
import com.sap.ip.mmr.IConnection;
import com.sap.ip.mmr.ResourceException;
import com.sap.ip.mmr.foundation.Utilities;
import com.sap.ip.mmr.search.QueryFilter;
import com.sap.ip.mmr.search.QueryFilters;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;
import com.sap.tc.webdynpro.repository.DataTypeBroker;

/**
 * @author d040882
 *
 * Implementation of the meta data browser. This class contains methods
 * to handle the connection to the model repository as well as methods to browse
 * and query the meta data of business objects.
 */
public class MetaModel {
	final static public int ASSOCIATION = 0;
	final static public int AGGREGATION = 1;
	private static final String APPLICATION = MetaModel.class.getName();
	private static final String JARM_REQUEST = "CAF:RT:oal" + APPLICATION;
	private static final Location location =
		Location.getLocation(MetaModel.class);

	// currently MMR does not cache data object by Class
	private static final HashMap dataObjects = new HashMap(23, 0.5f);

	public MetaModel() throws DataAccessException {
		//the connection is initially opened here.
		//the reason is that if there is a problem with
		//the connection, the exception can be thrown
		//before an instance of this class is created
		IConnection con = RepositoryConnection.getConnection();

		if (con == null)
			throw new DataAccessException("MMR_CON_OPEN_ERROR");
	}

	/**
	 * Returns the native shareableobject for the passed id. Only "real" business objects can be
	 * queried.
	 * @param qualifiedName - qualifiedName of the required shareableobject
	 * consistent of provider/application/id
	 * @return The required shareableobject
	 */
	public BusinessEntityInterface getBusinessEntityInterface(String qualifiedName)
		throws DataAccessException {
		return getBusinessEntityInterface(
			qualifiedName.substring(0, qualifiedName.indexOf("/")),
			qualifiedName.substring(
				qualifiedName.indexOf("/") + 1,
				qualifiedName.lastIndexOf("/")),
			qualifiedName.substring(qualifiedName.lastIndexOf("/") + 1));
	}

	/**
	 * Returns the native shareableobject for the passed id. Only "real" business objects can be
	 * queried.
	 * @param qualifiedName - qualifiedName of the required shareableobject
	 * consistent of provider/application/id
	 * @return The required shareableobject
	 */
	public BusinessEntityInterface getBusinessEntityInterface(Class objType)
		throws DataAccessException {
		try {
			return getBusinessEntityInterface(
				objType.getDeclaredField("provider").get(null).toString(),
				objType.getDeclaredField("application").get(null).toString(),
				objType.getDeclaredField("dataObjectId").get(null).toString());
		} catch (Exception e) {
			throw new DataAccessException(objType.getName());
		}
	}

	/**
	 * Returns the native shareableobject for the passed id. Only "real" business objects can be
	 * queried.
	 * @param provider - name of the application provider
	 * @param application - name of the application
	 * @param id - id of the required shareableobject
	 * @return The required shareableobject
	 */
	public BusinessEntityInterface getBusinessEntityInterface(
		String provider,
		String application,
		String id)
		throws DataAccessException {
		String method = "getBusinessEntityInterface(String, String, String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { provider, application, id },
			CAFPublicLogger.LEVEL_MEDIUM);
		BusinessEntityInterface be = null;
		ArrayList typeList = new ArrayList();
		typeList.add("com");
		typeList.add("sap");
		typeList.add("caf");
		typeList.add("metamodel");
		typeList.add("BusinessEntityInterface");

		QueryFilters filters = new QueryFilters();
		QueryFilter filter;
		//UNCOMMMENT / REWRITE for search by provider and application name
		/*		filter = new QueryFilter( "providerName", QueryFilter.COMPARATOR_EQ, provider);
				filters.add(filter);
				filter = new QueryFilter( "Application", QueryFilter.COMPARATOR_EQ, application);
				filters.add(filter);*/
		//		filter = new QueryFilter( "objectType", QueryFilter.COMPARATOR_EQ, "0");
		//		filters.add(filter);
		filter = new QueryFilter("objectName", QueryFilter.COMPARATOR_EQ, id);
		filters.add(filter);

		IConnection con = RepositoryConnection.getConnection();
		Iterator objs = null;
		try {
			objs = con.queryM1(typeList, filters, true).iterator();
		} catch (ResourceException e) {
			Object[] args = { provider + ":" + application + ":" + id };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error getting BusinessEntityInterface of type {0}",
				args,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException("MMR_BO_NOTFOUND", args, e);
		}
		//take the first one in the list
		//it shouldn´t be possible that there are more than one objects in the list
		while (objs.hasNext()) {
			be = (BusinessEntityInterface) objs.next();
			if ((be.getApplication() != null)
				&& (application != null)
				&& (provider != null)
				&& (application.equals(be.getApplication().getObjectName()))
				&& (provider.equals(be.getApplication().getProviderName()))) {
				break;
			}
			be = null;
		}
		if (be == null) {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException(
				"MMR_BO_NOTFOUND",
				new Object[] { id });
		}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return be;
	}

	public BusinessEntityInterface getBusinessEntityInterfaceByGUID(String guid)
		throws DataAccessException {
		String method =	"getBusinessEntityInterfaceByGUID(String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { guid },
			CAFPublicLogger.LEVEL_MEDIUM);

		BusinessEntityInterface be = null;
		ArrayList typeList = new ArrayList();
		typeList.add("com");
		typeList.add("sap");
		typeList.add("caf");
		typeList.add("metamodel");
		typeList.add("BusinessEntityInterface");

		Collection col = new ArrayList();
		col.add(guid);

		IConnection con = RepositoryConnection.getConnection();
		Iterator objs = con.selectM1(col, true).iterator();

		//take the first one in the list
		//it shouldn´t be possible that there are more than one objects in the list
		while (objs.hasNext()) {
			be = (BusinessEntityInterface) objs.next();
		}

		if (be == null) {
			CAFPublicLogger.exiting(null, JARM_REQUEST, method, location);
			throw new DataAccessException(
				"MMR_BO_NOTFOUND",
				new Object[] { guid });
		}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return be;
	}
	public Collection getAllBusinessEntityInterfacesByApplication(
		String provider,
		String application)
		throws DataAccessException {
		String method =	"getBusinessEntityInterfaceByApplication(String, String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { provider, application },
			CAFPublicLogger.LEVEL_MEDIUM);
		BusinessEntityInterface bo = null;
		List typeList =
			Utilities.getUtilities().decodeQualifiedName(
				"com.sap.caf.metamodel.BusinessEntityInterface");

		Collection col = null;
		IConnection con = RepositoryConnection.getConnection();
		try {
			Iterator i = con.queryM1(typeList, null, true).iterator();
			col = new LinkedList();
		
			while (i.hasNext())
			{
				bo = (BusinessEntityInterface)i.next();
				if (bo.getApplication().getObjectName().equals(application)  &&
					bo.getApplication().getProviderName().equals(provider))
				{
					col.add(bo);
				}
			}
		} catch (ResourceException e) {
			Object[] args = { provider + ":" + application };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error getting BusinessEntityInterface of type {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException("MMR_BO_NOTFOUND", args, e);
		} finally {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
		return col;
	}

	public BusinessEntityInterface getBusinessEntityInterfaceByObjectName(String objectName)
		throws DataAccessException {
		String method = "getBusinessEntityInterfaceByObjectName(String)";
		List typeList =
			Utilities.getUtilities().decodeQualifiedName(
				"com.sap.caf.metamodel.BusinessEntityInterface");
		QueryFilters filters = new QueryFilters();
		com.sap.ip.mmr.search.QueryFilter filter =
			new com.sap.ip.mmr.search.QueryFilter(
				"objectName",
				com.sap.ip.mmr.search.QueryFilter.COMPARATOR_EQ,
				objectName);
		filters.add(filter);
		BusinessEntityInterface result = null;
		try {
			result =
				(BusinessEntityInterface) (RepositoryConnection
					.getConnection()
					.queryM1(typeList, filters, true)
					.iterator()
					.next());
		} catch (ResourceException e) {
			Object[] args = { objectName };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error getting BusinessEntityInterface with name {0}",
				args,
				e);
			location.throwing(method, e);
			//CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException("MMR_BO_NOTFOUND", args, e);
		}
		return result;
	}
	
	/**
	 * Returns all native shareableobject. Only "real" business objects can be
	 * queried. The collection consists of shareableobject types.
	 * @return Collection with elements of type BusinessEntityInterface
	 */
	public Collection getAllBusinessEntityInterfaces()
		throws DataAccessException {
		String method = "getAllBusinessEntityInterfaces()";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		ArrayList typeList = new ArrayList();
		typeList.add("com");
		typeList.add("sap");
		typeList.add("caf");
		typeList.add("metamodel");
		typeList.add("BusinessEntityInterface");

		Collection col = null;

		IConnection con = RepositoryConnection.getConnection();
		try {
			col = con.queryM1(typeList, null, true);
		} catch (ResourceException e) {
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error getting BusinessEntityInterface of type {0}",
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException("MMR_BO_NOTFOUND", e);
		}
		if ((col == null) || (col.size() == 0)) {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException("MMR_BO_NOTFOUND");
		}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return col;
	}

	/**
	 * Returns all native shareableobject. Only "real" business objects can be
	 * queried. The collection consists of shareableobject types.
	 * @return Collection with elements of type BusinessEntityInterface
	 */
	public Collection getAllBusinessEntityInterfaces(int objectType)
		throws DataAccessException {
		String method = "getAllBusinessEntityInterfaces(int)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { Integer.toString(objectType)},
			CAFPublicLogger.LEVEL_MEDIUM);
		ArrayList typeList = new ArrayList();
		typeList.add("com");
		typeList.add("sap");
		typeList.add("caf");
		typeList.add("metamodel");
		typeList.add("BusinessEntityInterface");
		//there are several possible object types
		//object type=0 means business object
		QueryFilters filters = new QueryFilters();
		String strType = new Integer(objectType).toString();
		QueryFilter filter =
			new QueryFilter("backend", QueryFilter.COMPARATOR_EQ, strType);
		filters.add(filter);
		Collection col = null;

		IConnection con = RepositoryConnection.getConnection();
		try {
			col = con.queryM1(typeList, filters, true);
		} catch (ResourceException e) {
			Object[] args = { Integer.toString(objectType)};
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error getting BusinessEntityInterface of type {0}",
				args,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException("MMR_BO_NOTFOUND", args, e);
		}
		if ((col == null) || (col.size() == 0)) {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException(
				"MMR_BO_NOTFOUND",
				new Object[] { Integer.toString(objectType)});

		}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return col;
	}

	/**
	 * Returns all related shareableobjects of one shareableobject
	 * @param guid Guid of the source shareableobject
	 * @return HashMap which contains all related shareable objects. The key field contains the mofid
	 * 		   of the source role of the relation. The actual value of the map entry contains the
	 * 		   shareable object.
	 * @throws ConnectionException
	 * @throws QueryException
	 * @throws BONotFoundException
	 */
	public Map getRelatedBEsByGUID(String guid) throws DataAccessException {
		String method = "getRelatedBEsByGUID(String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { guid },
			CAFPublicLogger.LEVEL_MEDIUM);

		BusinessEntityInterface be = getBusinessEntityInterfaceByGUID(guid);
		Map map = getRelatedBEs(be);

		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return map;
	}

	private Map getRelatedBEs(BusinessEntityInterface be)
		throws DataAccessException {
		String method = "getRelatedBEs(BusinessEntityInterface)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { be },
			CAFPublicLogger.LEVEL_MEDIUM);

		Map map = new HashMap();
		//get all objectattrbuteproperties and filter the relations
		Collection attr = be.getBusinessEntity().getAttributes();
		if (attr == null)
			return new HashMap();

		Iterator itRelations = attr.iterator();
		while (itRelations.hasNext()) {
			Attribute oap = (Attribute) itRelations.next();
			Object obj = oap.getReferencedObject();
			if (obj!=null && obj instanceof DataObject) {

				BusinessEntityInterface relatedBE =
					(BusinessEntityInterface) ((DataObject) obj).getBusinessEntityInterface();
				if(relatedBE == null)
					continue;
									 	
				map.put(((Attribute) oap).refMofId(), relatedBE);
			}
		}

		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return map;
	}

	/**
	 * Returns all shareable objects which have a relation to the given
	 * shareable object. 
	 * @param application - name of the application in which the business object is registered
	 * @param bo - id of the business object
	 * @return HashMap which contains all related shareable objects. The key field contains the mofid
	 * 		   of the source role of the relation. The actual value of the map entry contains the
	 * 		   shareable object.
	 */
	public Map getRelatedBEs(String provider, String application, String bo)
		throws DataAccessException {
		String method = "getRelatedBEs(String, String, String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { provider, application, bo },
			CAFPublicLogger.LEVEL_MEDIUM);

		//get the business object which is the source of the relation
		BusinessEntityInterface be =
			getBusinessEntityInterface(provider, application, bo);
		Map map = getRelatedBEs(be);

		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return map;
	}

	/**
	 * Returns the target shareable object of a relation.
	 * @param relationGiod - guid of the objectattributeproperty which represents the relation
	 * @return The target sharableobject of the relation
	 */
	public BusinessEntityInterface getRelatedBOByRelation(String relationGuid)
		throws DataAccessException {
		String method = "getRelatedBOByRelation(String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		BusinessEntityInterface be = null;
		Collection guids = new Vector();
		guids.add(relationGuid);
		Collection col = null;

		IConnection con = RepositoryConnection.getConnection();
		col = con.selectM1(guids, true);

		//loop through all objectattributeproperties which were found and extract the appropriate shareableobject
		Iterator itRelations = col.iterator();
		while (itRelations.hasNext()) {
			Attribute oap = (Attribute) itRelations.next();
			if (oap.getObjectId().equals(relationGuid)) {
				Object obj = oap.getReferencedObject();
				if (obj instanceof DataObject) {
					//add navigation form DataObject to BEI
					be = ((DataObject) obj).getBusinessEntityInterface();
				}
			}
		}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return be;
	}

	/**
	 * Returns the native dataobject for the passed guid. 
	 * @param guid - guid of the dataobject on the m1 layer
	 * @return The required dataobject
	 */
	public DataObject getDataObjectByGUID(String guid)
		throws DataAccessException {
		String method = "getDataObjectByGUID(String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { guid },
			CAFPublicLogger.LEVEL_MEDIUM);
		DataObject dataObj = null;
		ArrayList typeList = new ArrayList();
		typeList.add("com");
		typeList.add("sap");
		typeList.add("caf");
		typeList.add("metamodel");
		typeList.add("DataObject");

		Collection col = new ArrayList();
		col.add(guid);

		IConnection con = RepositoryConnection.getConnection();
		Iterator objs = con.selectM1(col, true).iterator();

		//take the first one in the list
		//it shouldn´t be possible that there are more than one objects in the list
		while (objs.hasNext()) {
			dataObj = (DataObject) objs.next();
		}

		if (dataObj == null) {
			CAFPublicLogger.exiting(null, JARM_REQUEST, method, location);
			throw new DataAccessException(
				"MMR_BO_NOTFOUND",
				new Object[] { guid });
		}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return dataObj;
	}

	/**
	 * Returns the native dataobject for the passed id. 
	 * @param qualifiedName - qualifiedName of the required dataobject
	 * consistent of provider/application/id
	 * @return The required dataobject
	 */
	public DataObject getDataObject(String qualifiedName)
		throws DataAccessException {
		return getDataObject(
			qualifiedName.substring(0, qualifiedName.indexOf("/")),
			qualifiedName.substring(
				qualifiedName.indexOf("/") + 1,
				qualifiedName.lastIndexOf("/")),
			qualifiedName.substring(qualifiedName.lastIndexOf("/") + 1));
	}

	/**
	 * Returns the native dataobject for the passed id. 
	 * @param id - id of the required dataobject
	 * @return The required dataobject
	 */
	public DataObject getDataObject(
		String provider,
		String application,
		String objectName)
		throws DataAccessException {
		String method = "getDataObject(String, String, String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { provider, application, objectName },
			CAFPublicLogger.LEVEL_MEDIUM);
		DataObject bo = null;
		ArrayList typeList = new ArrayList();
		typeList.add("com");
		typeList.add("sap");
		typeList.add("caf");
		typeList.add("metamodel");
		typeList.add("DataObject");

		QueryFilters filters = new QueryFilters();
		QueryFilter filter =
			new QueryFilter(
				"objectName",
				QueryFilter.COMPARATOR_EQ,
				objectName);
		filters.add(filter);
		//UNCOMMMENT / REWRITE for search by provider and application name		
		/*filter = new QueryFilter( "Application", QueryFilter.COMPARATOR_EQ, application);
		filters.add(filter);
		filter = new QueryFilter( "providerName", QueryFilter.COMPARATOR_EQ, provider);
		filters.add(filter);*/

		IConnection con = RepositoryConnection.getConnection();
		Iterator objs = null;
		try {
			objs = con.queryM1(typeList, filters, true).iterator();
		} catch (ResourceException e) {
			Object[] args = { provider + ":" + application + ":" + objectName };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error getting DataObject of type {0}",
				args,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException("MMR_BO_NOTFOUND", args, e);
		}
		//take the first one in the list
		//it shouldn´t be possible that there are more than one objects in the list
		while (objs.hasNext()) {
			bo = (DataObject) objs.next();
			if ((bo.getApplication() != null)
				&& (application != null)
				&& (provider != null)
				&& (application.equals(bo.getApplication().getObjectName()))
				&& (provider.equals(bo.getApplication().getProviderName()))) {
				break;
			}

		}

		if (bo == null) {
			CAFPublicLogger.exiting(null, JARM_REQUEST, method, location);
			throw new DataAccessException(
				"MMR_BO_NOTFOUND",
				new Object[] { objectName });
		}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return bo;
	}

	public DataObject getDataObject(Class objType) throws DataAccessException {
		String method = "getDataObject(Class)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { objType },
			CAFPublicLogger.LEVEL_MEDIUM);

		try {
			synchronized (dataObjects) {
				DataObject dobj = (DataObject) dataObjects.get(objType);
				if (dobj == null) {
					dobj =
						getDataObject(
							objType
								.getDeclaredField("provider")
								.get(null)
								.toString(),
							objType
								.getDeclaredField("application")
								.get(null)
								.toString(),
							objType
								.getDeclaredField("dataObjectId")
								.get(null)
								.toString());
					dataObjects.put(objType, dobj);
				}
				return dobj;
			}
		} catch (Exception e) {
			Object[] args = { objType };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error getting DataObject of type {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException("MMR_BO_NOTFOUND", args, e);

		} finally {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}

	public Attribute getAttribute(String guid) throws DataAccessException {
		String method = "getAttribute(String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { guid },
			CAFPublicLogger.LEVEL_MEDIUM);
		Attribute attr = null;
		ArrayList typeList = new ArrayList();
		typeList.add("com");
		typeList.add("sap");
		typeList.add("caf");
		typeList.add("metamodel");
		typeList.add("Attribute");

		Collection guids = new Vector();
		guids.add(guid);

		IConnection con = RepositoryConnection.getConnection();
		Iterator it = con.selectM1(guids, true).iterator();

		//take the first one in the list
		//it shouldn´t be possible that there are more than one objects in the list
		while (it.hasNext()) {
			attr = (Attribute) it.next();
		}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return attr;
	}

	/**
	 * Read the names of all business objects which are stored in mmr 
	 * @return Collection of names of the business objects
	 * @throws QueryException
	 * @throws ConnectionException
	 */
	public Collection getBusinessObjectNames() throws DataAccessException {
		String method = "getBusinessObjectNames()";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);

		Collection colNames = new ArrayList();

		Collection colObjects = getAllBusinessEntityInterfaces(0);
		Iterator it = colObjects.iterator();
		while (it.hasNext()) {
			BusinessEntityInterface be = (BusinessEntityInterface) it.next();
			// TODO: add assigned application
			//				Application app = be.getApplication();
			//				colNames.add(app.getProviderName() + "/" + app.getId() + "/" + be.getId());
			colNames.add(be.getObjectName());
		}

		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return colNames;
	}

	/**
	 * Returns a list of all attributes for a business object
	 * @param qualifiedName Full qualified name of the business object "provider/application/id"
	 * @return Map of all attributes. The index of the map is the name of the attribute. The corresponding value
	 * 			indicates if the attribute is simple or if it is a structured attributes that contains additional
	 * 			sub-attributes. The value is of type java.lang.Boolean. True means it is a structures attribute, for
	 * 			simple attributes, the value is false
	 * @throws QueryException
	 * @throws ConnectionException
	 * @throws BONotFoundException
	 */
	public Map getAttributes(String qualifiedName) throws DataAccessException {
		String method = "getAttributes(String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { qualifiedName },
			CAFPublicLogger.LEVEL_MEDIUM);

		Map mapAttributes = new HashMap();

		BusinessEntityInterface be = getBusinessEntityInterface(qualifiedName);
		Iterator it = be.getBusinessEntity().getAttributes().iterator();

		while (it.hasNext()) {
			Attribute attr = (Attribute) it.next();
			//if the attribute is a simple one, there is no reference to another object
			if (attr.getReferencedObject() == null) {
				mapAttributes.put(attr.getObjectName(), new Boolean(false));
			} else {
				//check if the referenced object is another business object or a structured attribute
				Object obj = attr.getReferencedObject();
				if (obj instanceof DataObject) {
					// TODO: add navigation from DataObject to BEI
					mapAttributes.put(attr.getObjectName(), new Boolean(true));
				}
			}
		}

		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return mapAttributes;
	}

	/**
	 * This method returns the names of all attributes of a business object which are of a specific data type.
	 * The collection that is returned also contains all sub-attributes of complex attributes. Comples attributes
	 * are named like "name_of_complex_attribute.name_of_attribute"
	 * @param qualifiedName Full qualified name of the business object (e.g. sap.com/xapps.xpd/Idea)
	 * @param cl Class object which determines the data type which is used to filter the attributes
	 * @return Collection of all attributes which are of the given types (including complex attributes) 
	 * @throws QueryException
	 * @throws ConnectionException
	 * @throws BONotFoundException
	 */
	public Collection getAttributes(String qualifiedName, Class cl)
		throws DataAccessException {
		String method = "getAttributes(String, Class)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { qualifiedName, cl },
			CAFPublicLogger.LEVEL_MEDIUM);

		Collection col = new ArrayList();

		BusinessEntityInterface be = getBusinessEntityInterface(qualifiedName);
		if (be == null) {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new DataAccessException(
				"MMR_BO_NOTFOUND",
				new Object[] { qualifiedName + cl });
		}

		Iterator it = be.getBusinessEntity().getAttributes().iterator();
		while (it.hasNext()) {
			Attribute attr = (Attribute) it.next();
			//if the attribute is a simple one, there is no reference to another object
			if (attr.getReferencedObject() == null) {
				try {
					IDataType type =
						DataTypeBroker.getDataType(
							"ddic:" + attr.getTypeJavaDdic(),
							Locale.getDefault(),
							Thread.currentThread().getContextClassLoader());
					if (type.getAssociatedClass().equals(cl))
						col.add(attr.getObjectName());
				} catch (Exception e) {
					Object[] args = { qualifiedName + cl };
					CAFPublicLogger.traceThrowableT(
						Severity.DEBUG,
						location,
						method,
						"Error getting Attributes of type {0}",
						args,
						e);
					location.throwing(method, e);
					throw new DataAccessException("MMR_BO_NOTFOUND", args, e);

				}
			} else {
				//check if the referenced object is another business object or a structured attribute
				Object obj = attr.getReferencedObject();
				if (obj instanceof DataObject) {
					Iterator it2 =
						((DataObject) obj).getAttributes().iterator();
					while (it.hasNext()) {
						Attribute attr2 = (Attribute) it.next();

						try {
							IDataType type2 =
								DataTypeBroker.getDataType(
									"ddic:" + attr.getTypeJavaDdic(),
									Locale.getDefault(),
									Thread
										.currentThread()
										.getContextClassLoader());
							if (type2.getAssociatedClass().equals(cl))
								col.add(
									attr.getObjectName()
										+ "."
										+ attr2.getObjectName());
						} catch (Exception e) {
							Object[] args = { qualifiedName + cl };
							CAFPublicLogger.traceThrowableT(
								Severity.DEBUG,
								location,
								method,
								"Error getting Attributes of type {0}",
								args,
								e);
							location.throwing(method, e);
							throw new DataAccessException(
								"MMR_BO_NOTFOUND",
								args,
								e);
						}
					}
				}
			}
		}

		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return col;
	}

	public Collection readMMRObjects(
		String modelPackage,
		String objectType,
		String guid)
		throws DataAccessException {
		IConnection connection = RepositoryConnection.getConnection();
		if (connection == null) {
			throw new DataAccessException("MMR_CON_OPEN_ERROR");
		}

		Utilities utils = Utilities.getUtilities();
		List typeList =
			utils.decodeQualifiedName(modelPackage + "." + objectType);
		QueryFilters filters = null;

		if (guid != null) {
			filters = new QueryFilters();
			QueryFilter filter =
				new QueryFilter("objectID", QueryFilter.COMPARATOR_EQ, guid);
			filters.add(filter);
		}

		Collection c = null;

		try {
			c = connection.queryM1(typeList, filters, true);
		} catch (ResourceException ex) {
			throw new DataAccessException(ex.getMessage());
		}

		return c;
	}

	public void closeConnection() throws DataAccessException {
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			"closeConnection()",
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		RepositoryConnection.closeConnection();
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			"closeConnection()",
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
	}
}
