package com.sap.caf.rt.bol.util;

import java.math.BigDecimal;
import java.util.Date;

/**
 *  The <code>QueryFilter</code> class  describes interfaces to define 
 *  query filter, to be used by query methods in application. 
 *
 */
public class QueryFilter {
	public String attribute;
	public String valueLow;
	public String valueHigh;
	public BigDecimal bigIntValueLow;
	public BigDecimal bigIntValueHigh;
	public int intValueLow;
	public int intValueHigh;
	public Date dateValueLow;
	public Date dateValueHigh;
	public String condition;
	public boolean isString;
	public boolean isBigInt;
	public boolean isInt;
	public boolean isDate;
	
	public String operation;
	public String datatype;
	public String action;
	
	//condition values that are supported in the attribute "condition"
	public static final String CONDITION_EQ = "==";
	public static final String CONDITION_LT = "<";
	public static final String CONDITION_GT = ">";
	public static final String CONDITION_LE = "<=";
	public static final String CONDITION_GE = ">=";
	public static final String CONDITION_NOT = "!=";
	public static final String CONDITION_BETWEEN = "<>";
	
	//operation types that are supported in the attribute "operation"
	public static final String OPERATION_CATEGORY = "category";
	public static final String OPERATION_ATTRIBUTE = "attribute";
	public static final String OPERATION_TERM = "term";
	public static final String OPERATION_AND = "and";	
	public static final String OPERATION_OR = "or";
	public static final String OPERATION_NOT = "not";
	public static final String OPERATION_BRACKET_OPEN = "(";
	public static final String OPERATION_BRACKET_CLOSE = ")";
	
	//data types that are supported for the attribute "datatype"
	public static final String DATATYPE_STRING = "String";
	public static final String DATATYPE_INT = "int";
	public static final String DATATYPE_DATE = "Date";
	public static final String DATATYPE_TEXT = "text";
	
	//types of text searches that are supported in the attribute "action"
	public static final String ACTION_LINGUISTIC = "linguistic";
	public static final String ACTION_FUZZY = "fuzzy";
	public static final String ACTION_EXACT = "exact";

	public QueryFilter() {
	}

	public static QueryFilter getQueryFilter(Object value) {
			QueryFilter qf = null;

			if (value instanceof QueryFilter) {
				qf = (QueryFilter) value;
			} else if (value instanceof IntQueryFilter) {
				qf = (IntQueryFilter) value;
			} else if (value instanceof String) {
				if (((String) value).trim().equals("")) {
					qf = new QueryFilter((String) "*");
				} else
					qf = new QueryFilter((String) value);
			} else if (value instanceof Date) {
				qf = new QueryFilter((Date) value);
			} else if (value instanceof Integer) {
				qf = new QueryFilter(((Integer) value).intValue());
			} else if (value instanceof Number) {
				qf = new QueryFilter(new BigDecimal(((Number) value).toString()));
			} else if (value == null) {
				qf = new QueryFilter((String) "*");
			}
			return qf;
		}

	
	protected QueryFilter(QueryFilter qFilter) {
			this.condition = qFilter.condition;
			if(qFilter.isString == true) {
				this.isString = true;
				this.valueLow = qFilter.valueLow;
				this.valueHigh = qFilter.valueHigh;
			}
			else if(qFilter.isBigInt == true) {
				this.isBigInt = true;
				this.bigIntValueLow = qFilter.bigIntValueLow;
				this.bigIntValueHigh = qFilter.bigIntValueHigh;
			}
			else if(qFilter.isInt == true) {
				this.isInt = true;
				this.intValueLow = qFilter.intValueLow;
				this.intValueHigh = qFilter.intValueHigh;
			}
			else if(qFilter.isDate == true) {
				this.isDate = true;
				this.dateValueLow = qFilter.dateValueLow;
				this.dateValueHigh = qFilter.dateValueHigh;
			}
			this.datatype = qFilter.datatype;
			this.action = qFilter.action;
	}

	/**
	 * Creates instance for string type attribute
	 * @param value refers to the value of query attribute
	 */
	public QueryFilter(String value) {
		this.isString = true;
		this.valueLow = value;
		this.valueHigh = value;
		this.condition = "==";		
		this.datatype = QueryFilter.DATATYPE_STRING;
		this.action = QueryFilter.ACTION_LINGUISTIC;
	}
	
	public QueryFilter(String value, String condition) {
		this.isString = true;
		this.valueLow = value;
		this.valueHigh = value;
		this.condition = condition;		
		this.datatype = QueryFilter.DATATYPE_STRING;
		this.action = QueryFilter.ACTION_LINGUISTIC;
	}

	/**
	 * @param value refers to the value of query attribute
	 * @param condition refers to the  condition used in query
	 */
	public QueryFilter(BigDecimal value, String condition) {
		this.isBigInt = true;
		this.bigIntValueLow = value;
		this.bigIntValueHigh = value;
		this.condition = condition;
		this.datatype = QueryFilter.DATATYPE_INT;
	}
	
	/**
	 * Creates instance for BigDecimal type attribute to query values
	 * that lie between the  valueLow and valueHigh
	 * @param valueLow refers to the lowerbound of query attribute
	 * @param valueHigh refers to the upperbound of query attribute
	 */
	public QueryFilter(BigDecimal valueLow, BigDecimal valueHigh) {
		this.isBigInt = true;
		this.bigIntValueLow = valueLow;
		this.bigIntValueHigh = valueHigh;
		this.condition = "<>";
		this.datatype = QueryFilter.DATATYPE_INT;
	}
	
	/**
	 *  Creates instance for int type attribute
	 * @param value refers to the value of query attribute
	 */
	public QueryFilter(int value) {
			this.isInt = true;
			this.intValueLow = value;
			this.intValueHigh = value;
			this.condition = "==";
			this.datatype = QueryFilter.DATATYPE_INT;
	}
	
	/**
	 * Creates instance for BigDecimal type attribute
	 * @param value refers to the value of query attribute
	 */
	public QueryFilter(BigDecimal value) {
		this.isBigInt = true;
		this.bigIntValueLow = value;
		this.bigIntValueHigh = value;
		this.condition = "==";
		this.datatype = QueryFilter.DATATYPE_INT;
	}
	
	/**
	 * Creates instance for int type attribute with given condition
	 * @param value refers to the value of query attribute 
	 * @param condition refers to the  condition used in query
	 */
	public QueryFilter(int value, String condition) {
		this.isInt = true;
		this.intValueLow = value;
		this.intValueHigh = value;
		this.condition = condition;
		this.datatype = QueryFilter.DATATYPE_INT;
	}
	
	/**
	 * Creates instance for int type attribute to query values
	 * that lie between the  valueLow and valueHigh
	 * @param valueLow refers to the lowerbound of query attribute
	 * @param valueHigh refers to the upperbound of query attribute
	 */
	public QueryFilter(int valueLow, int valueHigh) {
		this.isInt = true;
		this.intValueLow = valueLow;
		this.intValueHigh = valueHigh;
		this.condition = "<>";
		this.datatype = QueryFilter.DATATYPE_INT;
	}
	
	/**
	 * Creates instance for Date type attribute
	 * @param value refers to the value of query attribute
	 */
	public QueryFilter(Date value) {
			this.isDate = true;
			this.dateValueLow = value;
			this.dateValueHigh = value;
			this.condition = "==";
			this.datatype = QueryFilter.DATATYPE_DATE;
	}
	
	/**
	 * Creates instance for Date type attribute with given condition
	 * @param value refers to the value of query attribute
	 * @param condition refers to the  condition used in query
	 */
	public QueryFilter(Date value, String condition) {
		this.isDate = true;
		this.dateValueLow = value;
		this.dateValueHigh = value;
		this.condition = condition;
		this.datatype = QueryFilter.DATATYPE_DATE;
	}
	
	/**
	 * Creates instance for Date type attribute to query values
	 * that lie between the  valueLow and valueHigh
	 * @param valueLow refers to the lowerbound of query attribute
	 * @param valueHigh refers to the upperbound of query attribute
	 */
	public QueryFilter(Date valueLow, Date valueHigh) {
		this.isDate = true;
		this.dateValueLow = valueLow;
		this.dateValueHigh = valueHigh;
		this.condition = "<>";
		this.datatype = QueryFilter.DATATYPE_DATE;
	}	
	
	
	/**
	 * Factory method for creating a query filter on a numeric attribute or category of a business object
	 * @param condition Condition for this filter. Use static members QueryFilter.CONDITION_* for valid possibilities
	 * @param value Value which should be searched for
	 * @param name Name of the attribute or category to search in. Complex attributes have to be named like "name_of_comlex_attribute.name_of_attribute"
	 * @param bAttribute True if parameter name is an attribute, false if it is a category
	 * @return QueryFilter for the given parameters
	 */
	public static QueryFilter createNumericFilter(String condition, int value, String name, boolean bAttribute) {
		QueryFilter filter = new QueryFilter();
		
		filter.condition=condition;
		filter.attribute=name;
		filter.intValueLow=value;
		filter.datatype=QueryFilter.DATATYPE_INT;
		filter.isInt=true;
		if(bAttribute)
			filter.operation=QueryFilter.OPERATION_ATTRIBUTE;
		else
			filter.operation=QueryFilter.OPERATION_CATEGORY;
		
		return filter;
	}

	/**
	 * Factory method for creatng a filter on string attributes. There are several opportunities for the type of
	 * search. An exact search, a wilcard search or a fuzzy search can be done
	 * @param action Type of search that should be performed. Use static members QueryFilter.ACTION_* for possible values
	 * @param value String that should be searched for. This string can contain wildcards (for all possible values, please check trex documentation)
	 * @param name Name of the attribute or category to search in. Complex attributes have to be named like "name_of_comlex_attribute.name_of_attribute"
	 * @param bAttribute True if parameter name is an attribute, false if it is a category
	 * @return QueryFilter for the given parameters
	 */
	public static QueryFilter createStringFilter(String action, String value, String name, boolean bAttribute) {
		QueryFilter filter = new QueryFilter();
		
		filter.action=action;
		filter.valueLow=value;
		filter.attribute=name;
		filter.datatype=QueryFilter.DATATYPE_STRING;
		filter.isString=true;
		if(bAttribute)
			filter.operation=QueryFilter.OPERATION_ATTRIBUTE;
		else
			filter.operation=QueryFilter.OPERATION_CATEGORY;
		
		return filter;
	}
	
	/**
	 * Factory method for creating a filter to perform a freetext search over all attributes of a business object	 * 
	 * @param action Type of search that should be performed. Use static members QueryFilter.ACTION_* for possible values
	 * @param value String that should be searched for. This string can contain wildcards (for all possible values, please check trex documentation)
	 * @return QueryFilter for the fiven parameters
	 */
	public static QueryFilter createFreetextFilter(String action, String value) {
		QueryFilter filter = new QueryFilter();
		
		filter.action=action;
		filter.valueLow=value;
		filter.datatype=QueryFilter.DATATYPE_STRING;		
		filter.isString=true;
		filter.operation=QueryFilter.OPERATION_ATTRIBUTE;
		filter.attribute="*";
		
		return filter;
	}			

	/**
	 * Factory method for creating a filter on a numeric attribute or category of a business object. For expressions on date
	 * attributes you have to keep in mind that a java.util.Date represents a timestamp which points to a time measured in milliseconds.
	 * So, for example, if you want to search for objects that have been created at a specific day, you have to define a query which
	 * contains two date filters that are combined by an AND-operator. The first timestamp points to the day you want to search for at 0:00 AM.
	 * The second timestamp represents 0:00 AM at the next day. 
	 * @param condition Condition for this filter. Use static members QueryFilter.CONDITION_* for valid possibilities
	 * @param value
	 * @param name
	 * @return QueryFilter for the fiven parameters
	 */
	public static QueryFilter createDateFilter(String condition, Date value, String name) {
		QueryFilter filter = new QueryFilter();
		
		filter.condition=condition;
		filter.dateValueLow=value;
		filter.attribute=name;
		filter.datatype=QueryFilter.DATATYPE_DATE;
		filter.isDate=true;
		filter.operation=QueryFilter.OPERATION_ATTRIBUTE;
		
		return filter;
	}
	
	/**
	 * Factory method for creating a bracket which can be part of a query string.
	 * @param type Type of bracket. Possible values are defined as static members QueryFilter.OPERATION_BRACKET_*
	 * @return QueryFilter for the fiven parameters
	 */
	public static QueryFilter createBracket(String type) {
		QueryFilter filter = new QueryFilter();
		
		filter.operation=type;
		
		return filter;
	}
	
	/**
	 * Factroy method for creating a bool operator which can be used to combine two expressions
	 * @param type Type of operator. Possible values are defined as static members QueryFilter.OPERATION_AND, QueryFilter.OPERATION_OR and
	 * 					QueryFilter.OPERATION_NOT.
	 * @return QueryFilter for the fiven parameters
	 */
	public static QueryFilter createBoolOperator(String type) {
		QueryFilter filter = new QueryFilter();
		
		filter.operation=type;
		
		return filter;
	}	
}
