package com.sap.caf.rt.bol.util;

import java.io.ObjectInputStream;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

import com.sap.caf.rt.exception.DataAccessException;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * @author d025479
 *
 * To change this generated comment edit the template variable "typecomment":
 * Window>Preferences>Java>Templates.
 * To enable and disable the creation of type comments go to
 * Window>Preferences>Java>Code Generation.
 */
public class JDBCConnection {
	public static final String dataSource = "jdbc/SAP/CAF_RT";
	private Connection con;
	private Context ctx;
	private static final String APPLICATION = JDBCConnection.class.getName();
	private static String JARM_REQUEST = "CAF:RT:oal" + APPLICATION;
	private static final Location location =
		Location.getLocation(JDBCConnection.class);

	public JDBCConnection() {
		String method = "JDBCConnection()";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		try {
			ctx = new InitialContext();
			createConnection();
		} catch (Exception e) {
			CAFPublicLogger.categoryCAF.logThrowableT(
				Severity.ERROR,
				location,
				method,
				"Error creating initial context",
				e);
			
		} finally {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}

	}

	public Collection selectColumn(String queryString, String columnName)
		throws DataAccessException {
		String method = "selectColumn(String , String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { queryString, columnName },
			CAFPublicLogger.LEVEL_MEDIUM);
		ArrayList result = new ArrayList();
		ResultSet rs = null;
		try {

			if (con == null)
				createConnection();

			Statement stmt = con.createStatement();
			rs = stmt.executeQuery(queryString);
			while (rs.next()) {
				result.add(rs.getString(columnName));
			}
		} catch (SQLException e) {
			Object[] args = { queryString, columnName };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"SQLException in selectColumn of queryString : {0} columnName : {1} ",
				args,
				e);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			location.throwing(method, e);
			throw new DataAccessException(e);

		} finally {
			try {
				rs.close();
				CAFPublicLogger.exiting(
					null,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
			} catch (SQLException e) {
				Object[] args = { queryString, columnName };
				CAFPublicLogger.traceThrowableT(
					Severity.DEBUG,
					location,
					method,
					"SQLException in selectColumn of queryString : {0} columnName : {1} ",
					args,
					e);
				CAFPublicLogger.exiting(
					null,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				location.throwing(method, e);
				throw new DataAccessException(e);

			}
		}
		return result;
	}

	/**
	 * Executes a jdbc query on the database and extracts one table column into a java collection
	 * @param queryString
	 * @param columnName
	 * @param length All values which are read from the database are shorten to this length  
	 * @return
	 * @throws SQLException
	 */
	public Collection selectColumn(
		String queryString,
		String columnName,
		String prefix,
		int length)
		throws DataAccessException {
		String method = "selectColumn(String , String , String , int )";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { queryString, columnName, prefix, length + "" },
			CAFPublicLogger.LEVEL_MEDIUM);
		Set result = new HashSet();
		ResultSet rs = null;
		try {

			if (con == null)
				createConnection();

			Statement stmt = con.createStatement();
			rs = stmt.executeQuery(queryString);
			while (rs.next()) {
				String tmp = rs.getString(columnName);
				if (tmp.length() > length) {
					if (prefix.length() > 0) {
						if (tmp.startsWith(prefix)) {
							result.add(tmp.substring(0, length));
						}
					} else {
						result.add(tmp.substring(0, length));
					}
				} else {
					if (prefix.length() > 0)
						if (tmp.startsWith(prefix))
							result.add(tmp);
						else
							result.add(tmp);
				}
			}
		} catch (SQLException e) {
			Object[] args = { queryString, columnName, prefix, length + "" };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"SQLException in selectColumn of queryString : {0} columnName : {1} prefix : {2} length : {3}",
				args,
				e);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			try {
				rs.close();
				CAFPublicLogger.exiting(
					null,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
			} catch (SQLException e) {
				Object[] args = { queryString, columnName };
				CAFPublicLogger.traceThrowableT(
					Severity.DEBUG,
					location,
					method,
					"SQLException in selectColumn of queryString : {0} columnName : {1} prefix : {2} length : {3} ",
					args,
					e);
				CAFPublicLogger.exiting(
					null,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				location.throwing(method, e);
				throw new DataAccessException(e);

			}
		}
		return result;
	}

	/**
	* Executes a jdbc query on the database ands extracts one table column. The column must be a BLOB 
	* which is automatically deserialized to java.util.Collection. 
	* @param queryString SQL-query
	* @param columnName	  
	* @return
	* @throws SQLException
	*/
	public Collection deserializeCollection(
		String queryString,
		String columnName)
		throws DataAccessException {
		String method =	"deserializeCollection(String , String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { queryString, columnName },
			CAFPublicLogger.LEVEL_MEDIUM);
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return deserializeCollection(queryString, columnName, "", -1);
	}

	public Collection deserializeCollection(
		String queryString,
		String columnName,
		String prefix,
		int length)
		throws DataAccessException {
		String method = "deserializeCollection(String, String, String)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { queryString, columnName, prefix, length + "" },
			CAFPublicLogger.LEVEL_MEDIUM);

		if (con == null)
			createConnection();

		Collection result = new HashSet();
		ResultSet rs = null;

		try {
			Statement stmt = con.createStatement();
			rs = stmt.executeQuery(queryString);

			while (rs.next()) {
				Blob blob = rs.getBlob(columnName);

				ObjectInputStream input =
					new ObjectInputStream(blob.getBinaryStream());
				Object obj = input.readObject();
				if (obj instanceof Collection) {
					Iterator it = ((Collection) obj).iterator();
					while (it.hasNext()) {
						String tmp = (String) it.next();
						if (length == -1) {
							if (prefix.length() > 0) {
								if (tmp.startsWith(prefix)) {
									result.add(tmp);
								}
							} else {
								result.add(tmp);
							}
						} else {
							if (prefix.length() > 0) {
								if (tmp.startsWith(prefix)) {
									result.add(tmp.substring(0, length));
								}
							} else {
								result.add(tmp.substring(0, length));
							}
						}
					}
				}
			}
		} catch (Exception e) {
			Object[] args = { dataSource };
			CAFPublicLogger.categoryCAF.logThrowableT(
				Severity.ERROR,
				location,
				method,
				"Exception in deserializeCollection of queryString : {0} columnName : {1} prefix : {2} length : {3}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			try {
				rs.close();
				CAFPublicLogger.exiting(
					null,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
			} catch (SQLException e) {
				Object[] args = { queryString, columnName };
				CAFPublicLogger.traceThrowableT(
					Severity.DEBUG,
					location,
					method,
					"SQLException in deserializeCollection of queryString : {0} columnName : {1} prefix : {2} length : {3}",
					args,
					e);
				CAFPublicLogger.exiting(
					null,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				location.throwing(method, e);
				throw new DataAccessException(e);

			}
		}
		return result;
	}

	private void createConnection() throws DataAccessException {
		String method = "createConnection()";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);

		try {
			DataSource ds = (DataSource) ctx.lookup(dataSource);
			con = ds.getConnection();
		} catch (NamingException e) {
			Object[] args = { dataSource };
			CAFPublicLogger.categoryCAF.logThrowableT(
				Severity.ERROR,
				location,
				method,
				"Error in JNDI lookup of {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException("JDBC_CONNECT_ERROR");
		} catch (SQLException e) {
			Object[] args = { dataSource };
			CAFPublicLogger.categoryCAF.logThrowableT(
				Severity.ERROR,
				location,
				method,
				"SQLException using datasource : {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException("JDBC_CONNECT_ERROR");
		} finally {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}

	public void close() throws DataAccessException {
		String method = "close()";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);

		if (con != null)
			try {
				con.close();
			} catch (SQLException e) {
				CAFPublicLogger.traceThrowableT(
					Severity.DEBUG,
					location,
					method,
					"SQLException in connection close",
					e);
				CAFPublicLogger.exiting(
					null,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				location.throwing(method, e);
				throw new DataAccessException(e);

			}
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);

	}
}
