package com.sap.caf.rt.bol.util;

import com.sap.caf.rt.exception.CAFBaseRuntimeException;
import com.sap.caf.rt.util.CAFResourceAccessor;
import com.sap.exception.BaseRuntimeException;
import com.sap.tc.logging.Location;

/**
 * The class is used internally by CAF to manipulate 
 * the JDO related Query filter to support queries using
 * different data types and conditions
 */
public class IntQueryFilter extends QueryFilter {

	public static final int BOOL_OPERATION_AND = 0;
	// AND is default in-statement operation
	public static final int BOOL_OPERATION_OR_START = 1;
	// OR condition demands parentheses correcly
	public static final int BOOL_OPERATION_OR_END = 2;
	// included into final statement to insure correct
	public static final int BOOL_OPERATION_INNER_OR = 3;
	// execution order of the statement conditions
	public static final int BOOL_OPERATION_NOT = 32;
	// NOT is unary operation and can be applied to the filter along with AND/OR binary operations

	protected int inStatementCondition; // assigned to AND by default

	private String attrName1;
	private String attrName2;
	private static int attrNumber;

	public boolean isCollection;
	private String collectionAttribute;
	private String collectionType;
	
	public boolean isEntityRefCollection;
	private String entityRefCollectionAttribute;
	private String entityRefCollectionType;
	
	private static Object[] allowedConditions =
		{ "==", "!=", ">", ">=", "<", "<=", "<>" };

	protected boolean m_isEmpty;

	private static final Location location =
		Location.getLocation(IntQueryFilter.class);

	public IntQueryFilter() {
		m_isEmpty = true;
	}

	public IntQueryFilter(QueryFilter qFilter) {
		super(qFilter);
		m_isEmpty = "*".equals(this.valueLow);
	}

	public Object getAttributeValue() {
		Object attribVal = null;
		if (this.isString == true) {
			if (this.valueLow != null && this.valueLow.startsWith("*")) {
				attribVal = valueLow.substring(1);
			} else if (this.valueLow != null && this.valueLow.endsWith("*")) {
				attribVal = valueLow.substring(0, (valueLow.length() - 1));
			} else
				//attribute value trimmed for whitespace
				if(valueLow !=null) {
					attribVal = this.valueLow.trim();
				}
		} else if (this.isInt == true)
			attribVal = new Integer(this.intValueLow);
		else if (this.isBigInt == true)
			attribVal = this.bigIntValueLow;
		else
			attribVal = this.dateValueLow;
		return attribVal;
	}

	public Object getAttributeValueHigh() {
		Object attribVal = null;
		if (this.isString == true) {
			//attribute value trimmed for whitespace
			attribVal = this.valueHigh.trim();
		} else if (this.isInt == true)
			attribVal = new Integer(this.intValueHigh);
		else if (this.isBigInt == true)
			attribVal = this.bigIntValueHigh;
		else
			attribVal = this.dateValueHigh;
		return attribVal;
	}

	public String getAttribute() {
		return this.attribute;
	}
	public void setAttribute(String parameter) {
		this.attribute = parameter;
		if (this.attribute != null  &&  this.collectionAttribute != null) {
			String extracted = extractAttribute(this.attribute);
			if (extracted.equals(extractAttribute(this.collectionAttribute))) {
				this.attribute = 
					this.attribute.substring(0, extracted.length()) + "_" +
					this.attribute.substring(extracted.length()); 
			}
		}
	}
	public String getCollectionAttribute() {
		return this.collectionAttribute;
	}
	public void setCollectionAttribute(String parameter) {
		this.isCollection = true;
		this.collectionAttribute = parameter;
		if (this.attribute != null  &&  this.collectionAttribute != null) {
			String extracted = extractAttribute(this.collectionAttribute);
			if (extracted.equals(extractAttribute(this.attribute))) {
				this.collectionAttribute = 
					this.collectionAttribute.substring(0, extracted.length()) + "_" +
					this.collectionAttribute.substring(extracted.length()); 
			}
		}
	}
	public String getCollectionType() {
		return this.collectionType;
	}
	public void setCollectionType(String type) {
		this.collectionType = type;
	}
	public String getEntityReCollectionAttribute() {
		return this.entityRefCollectionAttribute;
	}
	public void setEntityRefCollectionAttribute(String parameter) {
		this.isEntityRefCollection = true;
		this.entityRefCollectionAttribute = parameter;
	}
	public String getEntityRefCollectionType() {
		return this.entityRefCollectionType;
	}
	public void setEntityRefCollectionType(String type) {
		this.entityRefCollectionType = type;
	}
	

	public static String extractAttribute(String attr) {
		int nLength = attr.indexOf('.');
		return (nLength == -1) ? attr : attr.substring(0, nLength);
	}
	
	
	
	public static String createParameter(String attr) {
		return attr.replace('.','_');
	}


	public String getJdoFilter() {
		String filter = "";
		
		if (this.isEntityRefCollection) {
			String colInstance = extractAttribute(
				this.isCollection ? this.collectionAttribute : this.attribute);
			filter +=
				"("
					+ this.getEntityReCollectionAttribute()
					+ ".contains("
					+ colInstance
					+ "))"
					+ " && ";
		}
		if (this.isCollection) {
			String colInstance = extractAttribute(this.attribute);
			filter +=
				"("
					+ this.getCollectionAttribute()
					+ ".contains("
					+ colInstance
					+ "))"
					+ " && ";
		}
		
		filter += this.getJdoFilterInternal();
		return filter;
	}
	public String getJdoMapCollFilter() {
//		String colInstance = extractAttribute(this.attribute);
//		String filter = 
//			"("
//				+ this.collectionAttribute
//				+ ".contains("
//				+ colInstance
//				+ "))"
//				+ " && "
//				+ this.getJdoMapFilterInternal();

		String filter = this.getJdoMapFilterInternal();
		
		if(isEmpty()){
			filter = "";
		}
		
		return filter;
	}
	
	public String getJdoCollVariables() {
		if (this.isEmpty()) {
			return null;
		}
		
		String vars = "";
		
		if (this.isEntityRefCollection) {
			if (!"".equals(vars)) {
				vars += "; ";
			}
			String colInstance = extractAttribute(
				this.isCollection ? this.collectionAttribute : this.attribute);
			vars += this.getEntityRefCollectionType() + " " + colInstance;
		}
		if (this.isCollection) {
			if (!"".equals(vars)) {
				vars += "; ";
			}
			String colInstance = extractAttribute(this.attribute);
			vars += this.getCollectionType() + " " + colInstance;
		}
		return "".equals(vars) ?  null : vars;
	}

	private String getJdoFilterInternal() {
		String filter = null;
		boolean conditionOk = false;
		for (int i = 0; i < allowedConditions.length; i++) {
			if (allowedConditions[i] == condition) {
				conditionOk = true;
				break;
			}
		}
		if (conditionOk == false)
			throw new BaseRuntimeException(
				CAFResourceAccessor.getResourceAccessor(),
				"JDO_QUERY_INVALID_DATA",
				new Object[] { this.condition });
		if (this.isString == true) {
			if (this.valueLow != null && this.valueLow.endsWith("*")) {
				filter =
					attribute
						+ ".startsWith(\""
						+ valueLow.substring(0, (valueLow.length() - 1))
						+ "\")";
			} else if (
				this.valueLow != null && this.valueLow.startsWith("*")) {
				filter =
					attribute + ".endsWith(\"" + valueLow.substring(1) + "\")";
			} else {
				//attribute value trimmed for whitespace
				filter = attribute + ' ' + condition + ' ' + "\"" + valueLow.trim() + "\"";
			}
		} else if (this.isBigInt == true) {
			if (bigIntValueLow == bigIntValueHigh && condition != "<>")
				filter = attribute + ' ' + condition + ' ' + bigIntValueLow;
			else
				filter =
					attribute
						+ " >= "
						+ bigIntValueLow
						+ " && "
						+ attribute
						+ " <= "
						+ bigIntValueHigh;
		} else if (this.isInt == true) {
			if (intValueLow == intValueHigh && condition != "<>")
				filter = attribute + ' ' + condition + ' ' + intValueLow;
			else
				filter =
					attribute
						+ " >= "
						+ intValueLow
						+ " && "
						+ attribute
						+ " <= "
						+ intValueHigh;
		} else if (this.isDate == true) {
			if (dateValueLow.equals(dateValueHigh) && condition != "<>")
				filter = attribute + ' ' + condition + ' ' + "\"" + dateValueLow + "\"";
			//filter = parameter + condition + dateValueLow;
			else
				filter =
					attribute
						+ " >= "
						+ "\""
						+ dateValueLow
						+ "\""
						+ " && "
						+ attribute
						+ " <= "
						+ dateValueHigh;
		}
		return filter;
	}

	public String getJdoMapFilterInternal() {
		String filter = "";
		boolean conditionOk = false;
		for (int i = 0; i < allowedConditions.length; i++) {
			if (allowedConditions[i] == this.condition) {
				conditionOk = true;
				break;
			}
		}
		if (conditionOk == false)
			//TODO: change exception class - use standard?  - resource bundle use in standard?
			//throw new BOInvalidDataException(); //throw exception??
			throw new BaseRuntimeException(
				CAFResourceAccessor.getResourceAccessor(),
				"JDO_QUERY_INVALID_DATA",
				new Object[] { this.condition });

		String paramString = createParameter(this.attribute);
		attrName1 = paramString + getAttributeNumber();
		attrName2 = paramString + getAttributeNumber();
		if (this.isEmpty())
			return "";
			
		if (isEntityRefCollection) {
			filter += this.entityRefCollectionAttribute + ".contains(" +
				extractAttribute(this.isCollection ? this.collectionAttribute : this.attribute) + ") && ";
		}
		if (isCollection) {
			filter += this.collectionAttribute + ".contains(" + 
				extractAttribute(this.attribute) + ") && ";
		}
			
		if (this.isString == true) {
			if (this.valueLow != null && this.valueLow.startsWith("*")) {
				filter += attribute + ".endsWith(" + attrName1 + ")";
			} else if (this.valueLow != null && this.valueLow.endsWith("*")) {
				filter += attribute + ".startsWith(" + attrName1 + ")";
			} else {
				filter += this.attribute + ' ' + this.condition + ' ' + attrName1;
			}
		} else if (
			this.isDate == true
				|| this.isInt == true
				|| this.isBigInt == true) {
			if (this.condition != "<>")
				filter += this.attribute + ' ' + this.condition + ' ' + attrName1;
			else { //between case
				filter += this.attribute
					+ " >= "
					+ (attrName1)
					+ " && "
					+ this.attribute
					+ " <= "
					+ (attrName2);
			}
		}

		int inStatementCondition = getInStatementCondition();
		if ((inStatementCondition & BOOL_OPERATION_NOT) != 0) {
			filter = "!" + filter;
		}
		inStatementCondition = inStatementCondition & ~BOOL_OPERATION_NOT;
		switch (inStatementCondition) {
			case BOOL_OPERATION_AND :
				break;
			case BOOL_OPERATION_OR_START :
				filter = " (" + filter;
				break;
			case BOOL_OPERATION_INNER_OR :
				filter = " || " + filter;
				break;
			case BOOL_OPERATION_OR_END :
				filter = " || " + filter + ") ";
				break;
			default :
				;
		}
		return filter;
	}

	public String getJdoMapFilter() {
		String filter = null;
		if (this.isCollection == true)
			filter = getJdoMapCollFilter();
		else
			filter = getJdoMapFilterInternal();

		return filter;
	}

	public boolean isEmpty() {
		return m_isEmpty;
	}

	public static String joinFilter(String qf1, IntQueryFilter qf2) {
		return joinFilter(qf1, qf2, null);
	}

	public static String joinFilter(
		String qf1,
		IntQueryFilter qf2,
		String operation) {
		if (operation == null || operation.length() == 0) {
			operation = " && ";
		}
		return (qf1 == null || qf1.length() == 0)
			? (qf2 == null || qf2.isEmpty() ? "" : qf2.getJdoFilter())
			: (qf2.isEmpty() ? qf1 : qf1 + operation + qf2.getJdoFilter());
	}

	public static String joinFilter(IntQueryFilter qf1, IntQueryFilter qf2) {
		return joinFilter(qf1, qf2, null);
	}

	public static String joinFilterMap(
		IntQueryFilter qf1,
		IntQueryFilter qf2,
		String operation) {
		return (qf1 == null)
			? joinFilterMap("", qf2)
			: joinFilterMap(qf1.getJdoMapFilter(), qf2, operation);

	}

	public static String joinFilterMap(
		String qf1,
		IntQueryFilter qf2,
		String operation) {
		StringBuffer result = new StringBuffer(50);
		if (qf1 != null && qf1.length() > 0) {
			result.append(qf1);
			if ((operation == null) || (operation.length() == 0))
				operation = " && ";
			if (qf2 != null) {
				if (qf2.getInStatementCondition() != BOOL_OPERATION_INNER_OR
					&& qf2.getInStatementCondition() != BOOL_OPERATION_OR_END
					&& !qf2.isEmpty()) {
					result.append(operation);
				} else {
					// result.append(" || "); is not needed as soon as the logical OR is added inside getJdoMapFilter(). 
				}
				result.append(qf2.getJdoMapFilter());
			}
		} else if (qf2 != null) {
			// first filter is empty just add second and exit
			result.append(qf2.getJdoMapFilter());
		}
		return result.toString();
	}

	public static String joinFilter(
		IntQueryFilter qf1,
		IntQueryFilter qf2,
		String operation) {
		if (operation == null || operation.length() == 0) {
			operation = " && ";
		}
		String filter =
			(qf1.isEmpty()
				? (qf2 == null || qf2.isEmpty() ? "" : qf2.getJdoFilter())
				: (qf2.isEmpty()
					? qf1.getJdoFilter()
					: qf1.getJdoFilter() + operation + qf2.getJdoFilter()));
		return filter;
	}

	public static String joinFilterMap(String qf1, IntQueryFilter qf2) {
		return joinFilter(qf1, qf2, null);
	}

	public static String joinFilterMap(
		IntQueryFilter qf1,
		IntQueryFilter qf2) {
		return joinFilter(qf1, qf2, null);
	}

	public int getInStatementCondition() {
		return inStatementCondition;
	}

	public void setInStatementCondition(int condition) {
		inStatementCondition = condition;
	}

	public String getAttributeName1() {
		if (attrName1 == null) {
			throw new CAFBaseRuntimeException("You must call getJdoMapFilter() or joinFilterMap() to be able to get an attribute name.");
		}
		return attrName1;
	}

	public String getAttributeName2() {
		if (attrName1 == null) {
			throw new CAFBaseRuntimeException("You must call getJdoMapFilter() or joinFilterMap() to be able to get an attribute name.");
		}
		return attrName2;
	}

	private static synchronized int getAttributeNumber() {
		return attrNumber++;
	}
}
