package com.sap.caf.rt.bol.util;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Map.Entry;

/**
 * This class provides an MRU cache with a controled size of the cache
 * If the cache is not full any new object is added to the top of the
 * cache, <br>
 * Objects that already exist are moved to the top of the cache when they 
 * are accessed. <br>
 * If the cache is full, the last object is removed from the cache.
 */
public class Cache {
	HashMap cache;
	LinkedList mru;
	int size;
	
	public Cache(int cacheSize) {
		cache = new HashMap();
		mru = new LinkedList();
		size = cacheSize;
	}
	
	/**
	 * Adds a new Object to the cache or updates an existing reference
	 * @param key identifier of the object
	 * @param object identified by its identifier
	 */
	public void put(String key, Object value) {
		if (!cache.containsKey(key)) {
			cache.put(key, value);
			mru.addFirst(key);
			if (mru.size() > size) {
				cache.remove(mru.getLast());
				mru.removeLast();
			}
		}
		else {
			mru.remove(key);
			mru.addFirst(key);
		}
	}
	
	/**
	 * Adds the objects from the list that are yet not containes in the 
	 * cache to it and updates the references of the objects already contained
	 * in the cache.
	 * @param objects Map of objects and their identifiers
	 */
	public void putAll(Map objects) {
		Iterator it = objects.entrySet().iterator();
		while (it.hasNext()) {
			Map.Entry row = (Entry) it.next();
			put((String)row.getKey(), row.getValue());
		}
	}
	
	/**
	 * This method returns the object specified by its key from the
	 * cache. If the object is not contained in the cache it returns null.
	 * @param key key of the object
	 * @return Object specified by the key
	 */
	public Object get(String key) {
		if (mru.contains(key)) {
			mru.remove(key);
			mru.addFirst(key);
		}
		return cache.get(key);
	}
	
	/**
	 * This method removes the specified object from the cache
	 */
	public void remove(String key) {
		if (mru.contains(key))
			mru.remove(key);
			cache.remove(key);
	}
	/** 
	 * removes all entries from the cache. The cache will be empty after
	 * this method returns.
	 */
	public void clear() {
		cache.clear();
		mru.clear();
	}
	
	public boolean isEmpty(){
		boolean res = false;
		if  (mru.size() == 0 ){
			res = true;
		}
		return res; 
	}
}
