package com.sap.caf.rt.bol.da.jdo.registration;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;


import com.sap.caf.rt.bol.IBusinessObject;
import com.sap.caf.rt.bol.IDependentObject;
import com.sap.caf.rt.bol.context.CAFContext;
import com.sap.caf.rt.bol.pk.PrimaryKeyFactory;
import com.sap.caf.rt.exception.DataAccessException;
import com.sap.tc.logging.Location;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.sql.Timestamp;
import java.sql.Types;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.sql.DataSource;

/**
 * Class for registration deleted entries of business objects.
 * 
 * @author C5050970
 */
public class DeletedObjectsRegistrationService {
	private static final String APPLICATION = DeletedObjectsRegistrationService.class.getName();
	private static final String JARM_REQUEST = "CAF:RT:oal" + APPLICATION;
	private static final Location location = 
		Location.getLocation(DeletedObjectsRegistrationService.class);
		
	public static final String dataSource = "jdbc/SAP/CAF_RT";
	private static Connection con;

	private static final String TABLE_NAME = "CAF_RT_DEL_OBJECTS";
	private static final String ID = "ID";
	private static final String MOFID = "MOFID";
	private static final String OBJECT_KEY = "OBJECT_KEY";
	private static final String DELETED_AT = "DELETED_AT";

	private static final String insertQuery = "INSERT INTO " + TABLE_NAME 
		+ " ( " + ID +  "," + MOFID + "," + OBJECT_KEY + "," + DELETED_AT + " ) " 
		+ " VALUES (?,?,?,?)";
	private static final String clearQuery = "DELETE FROM " + TABLE_NAME;
	private static final String selectAllQuery = "SELECT * FROM " + TABLE_NAME;
	private static final String selectByMofidQuery = "SELECT * FROM " + TABLE_NAME + " WHERE " + MOFID + " =? ";
	private static final String selectByDatesQuery = "SELECT * FROM " + TABLE_NAME 
		+ " WHERE " + DELETED_AT + " >=? AND " + DELETED_AT + " <=? ";
	private static final String selectByDatesQueryNullHighDate = "SELECT * FROM " + TABLE_NAME 
		+ " WHERE " + DELETED_AT + " >=? ";
	private static final String selectByDatesQueryNullLowDate = "SELECT * FROM " + TABLE_NAME 
		+ " WHERE " + DELETED_AT + " <=? ";

	private static final String selectByMofidAndDatesQuery = "SELECT * FROM " + TABLE_NAME 
		+ " WHERE " + MOFID + " =? AND " + DELETED_AT + " >=? AND " + DELETED_AT + " <=?";
	private static final String selectByMofidAndDatesQueryNullHighDate = "SELECT * FROM " + TABLE_NAME 
		+ " WHERE " + MOFID + " =? AND " + DELETED_AT + " >=? ";
	private static final String selectByMofidAndDatesQueryNullLowDate = "SELECT * FROM " + TABLE_NAME 
		+ " WHERE " + MOFID + " =? AND " + DELETED_AT + " <=? ";

	/**
	 * Register delete of the object
	 * @param obj
	 * @param jdoDataAccessService
	 * @throws DataAccessException
	 */
	public static void registerDelete(IDependentObject obj) throws DataAccessException {
		if(isRegistrationServiceEnabled()) {
			if(obj instanceof IBusinessObject) {
				if(con == null) {
					initJDBCConnection();
				}
				
				try {
					PreparedStatement ps = con.prepareStatement(insertQuery);
					String key = PrimaryKeyFactory.getInstance().getPrimaryKey();
					String mofid = getObjectMofid((IBusinessObject) obj);
					String objectKey = obj.getKey();
					Date deletedAt = new Date();
						
					ps.setString(1, key);
					ps.setString(2, mofid);
					ps.setString(3, objectKey);
					ps.setTimestamp(4, new Timestamp(deletedAt.getTime()));
		
					ps.executeUpdate();
					ps.close();
				} catch(Exception e) {
					throw new DataAccessException(e);
				}
			}
		}
	}
	
	/**
	 * Clear all log of deleted objects
	 * @return
	 * @throws DataAccessException
	 */
	public static int clear() throws DataAccessException {
		if(con == null) {
			initJDBCConnection();
		}
		
		try {
			Statement stmt = con.createStatement();
			int count = stmt.executeUpdate(clearQuery);
			stmt.close();
			return count;
		} catch(Exception e) {
			throw new DataAccessException(e);
		}
	}
	
	/**
	 * Find all log entries for deleted objects
	 * @return
	 * @throws DataAccessException
	 */
	public static Collection findAllDeleted() throws DataAccessException {
		ArrayList list = new ArrayList();
		if(con == null) {
			initJDBCConnection();
		}
		
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(selectAllQuery);
			while(rs.next()) {
				DeletedObject delObj = new DeletedObject();
				delObj.setKey(rs.getString(ID));
				delObj.setMofid(rs.getString(MOFID));
				delObj.setObjectKey(rs.getString(OBJECT_KEY));
				delObj.setDeletedAt(rs.getTimestamp(DELETED_AT));
				
				list.add(delObj);
			}
			rs.close();
			stmt.close();
		} catch(Exception e) {
			throw new DataAccessException(e);
		}
		
		return list;
	}
	
	/**
	 * Find all log entries by mofid for deleted objects.
	 * @param mofid
	 * @return
	 * @throws DataAccessException
	 */
	public static Collection findDeletedByMofid(String mofid) throws DataAccessException {
		ArrayList list = new ArrayList();
		if(con == null) {
			initJDBCConnection();
		}
	
		try {
			PreparedStatement ps = con.prepareStatement(selectByMofidQuery);
			ps.setString(1, mofid);
	
			ResultSet rs = ps.executeQuery();
			while(rs.next()) {
				DeletedObject delObj = new DeletedObject();
				delObj.setKey(rs.getString(ID));
				delObj.setMofid(rs.getString(MOFID));
				delObj.setObjectKey(rs.getString(OBJECT_KEY));
				delObj.setDeletedAt(rs.getTimestamp(DELETED_AT));
			
				list.add(delObj);
			}
			rs.close();
			ps.close();
		} catch(Exception e) {
			throw new DataAccessException(e);
		}
	
		return list;
	}
	
	/**
	 * Find all log entries by dates for deleted objects (entries with startDate<= DELETED_AT <=endDate).
	 * @param startDate
	 * @param endDate
	 * @return
	 * @throws DataAccessException
	 */
	public static Collection findDeletedByDates(Date startDate, Date endDate) throws DataAccessException {
		ArrayList list = new ArrayList();
		if(con == null) {
			initJDBCConnection();
		}
	
		try {
			PreparedStatement ps = null;
			if(startDate == null) {
				if(endDate == null) {
					ps = con.prepareStatement(selectAllQuery);
				} else {
					ps = con.prepareStatement(selectByDatesQueryNullLowDate);
					ps.setTimestamp(1, new Timestamp(endDate.getTime()));										
				}
			} else {
				if(endDate == null) {
					ps = con.prepareStatement(selectByDatesQueryNullHighDate);
					ps.setTimestamp(1, new Timestamp(startDate.getTime()));					
				} else {
					ps = con.prepareStatement(selectByDatesQuery);
					ps.setTimestamp(1, new Timestamp(startDate.getTime()));
					ps.setTimestamp(2, new Timestamp(endDate.getTime())); 					
				}
			}
						
			ResultSet rs = ps.executeQuery();
			while(rs.next()) {
				DeletedObject delObj = new DeletedObject();
				delObj.setKey(rs.getString(ID));
				delObj.setMofid(rs.getString(MOFID));
				delObj.setObjectKey(rs.getString(OBJECT_KEY));
				delObj.setDeletedAt(rs.getTimestamp(DELETED_AT));
			
				list.add(delObj);
			}
			rs.close();
			ps.close();
		} catch(Exception e) {
			throw new DataAccessException(e);
		}
	
		return list;
	}
	
	/**
	 * Init JDBC connection
	 * @throws DataAccessException
	 */
	private static void initJDBCConnection() throws DataAccessException {
		try {
			Context ctx = new InitialContext();
			DataSource ds = (DataSource) ctx.lookup(dataSource);
			con = ds.getConnection();
		} catch (Exception e) {
			throw new DataAccessException(e);
		}
	}
	
	/**
	 * Get object mofid
	 * @param obj
	 * @return
	 */
	private static String getObjectMofid(IBusinessObject obj) throws DataAccessException {
		try {
			Field field = obj.getClass().getDeclaredField("objectGuid");
			field.setAccessible(true);
			String objectGuid = (String) field.get(null);
			if(objectGuid!=null) {
				return objectGuid;
			}
		} catch(Exception e) {
			throw new DataAccessException(e);
		}
		
		try {
			String objRid = obj.getRid();
			
			int index1 = objRid.indexOf('/');
			objRid = objRid.substring(index1+1);
			
			index1 = objRid.indexOf('/');
			objRid = objRid.substring(0, index1);
			if(objRid!=null) {
				return objRid;
			}
		} catch(Exception e) {
			throw new DataAccessException(e);
		}
		
		return "";
	}
	
	/**
	 * Check if registration service is enabled.
	 * @return
	 */
	private static boolean isRegistrationServiceEnabled() {
		return CAFContext.DELETION_REGISTRATION_ENABLED;
	}
	
	/**
	 * Find all log entries by mofid and dates for deleted objects (entries with startDate<= DELETED_AT <=endDate).
	 * @param mofid
	 * @param startDate
	 * @param endDate
	 * @return
	 * @throws DataAccessException
	 */
	public static Collection findDeletedByMofidAndDates(String mofid, Date startDate, Date endDate) throws DataAccessException {
		ArrayList list = new ArrayList();
		if(con == null) {
			initJDBCConnection();
		}
	
		try {
			
			PreparedStatement ps = null;
			if(startDate == null) {
				if(endDate == null) {
					ps = con.prepareStatement(selectByMofidQuery);
				} else {
					ps = con.prepareStatement(selectByMofidAndDatesQueryNullLowDate);
					ps.setTimestamp(2, new Timestamp(endDate.getTime()));										
				}
			} else {
				if(endDate == null) {
					ps = con.prepareStatement(selectByMofidAndDatesQueryNullHighDate);
					ps.setTimestamp(2, new Timestamp(startDate.getTime()));					
				} else {
					ps = con.prepareStatement(selectByMofidAndDatesQuery);
					ps.setTimestamp(2, new Timestamp(startDate.getTime()));
					ps.setTimestamp(3, new Timestamp(endDate.getTime())); 					
				}
			}
			ps.setString(1, mofid);
						
			ResultSet rs = ps.executeQuery();
			while(rs.next()) {
				DeletedObject delObj = new DeletedObject();
				delObj.setKey(rs.getString(ID));
				delObj.setMofid(rs.getString(MOFID));
				delObj.setObjectKey(rs.getString(OBJECT_KEY));
				delObj.setDeletedAt(rs.getTimestamp(DELETED_AT));
			
				list.add(delObj);
			}
			rs.close();
			ps.close();
		} catch(Exception e) {
			throw new DataAccessException(e);
		}
		return list;
	}

}
