package com.sap.caf.rt.bol.da.jdo;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import javax.jdo.Extent;
import javax.jdo.PersistenceManager;
import javax.jdo.PersistenceManagerFactory;
import javax.jdo.Query;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import com.sap.caf.bw.api.BWProcessorLocalFactory;
import com.sap.caf.rt.bol.IDependentObject;
import com.sap.caf.rt.bol.context.CAFContext;
import com.sap.caf.rt.bol.da.IDataAccessService;
import com.sap.caf.rt.bol.da.jdo.registration.DeletedObjectsRegistrationService;
import com.sap.caf.rt.bol.util.IntQueryFilter;
import com.sap.caf.rt.bol.util.UserContext;
import com.sap.caf.rt.exception.DataAccessException;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.exception.BaseRuntimeException;
import com.sap.jdo.SAPJDOHelper;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * This class implements the Dataaccess service for local persistence.
 * It provides life cycle and query methods for local data store entities.
 * The JDO PersistenceManagerFactory is initalized first time the class is instantiated.
 * The instance of PersistenceManager itself is created and removed in the respective
 * life cycle or query method.
 */
public class JDODataAccessService implements IDataAccessService {
	private static final long serialVersionUID = 1;

	private static PersistenceManagerFactory pmf = null;
	/*
	 * Refers to the JNDI lookup name of the JDO Connector
	 */
	private static final String JNDI_NAME = CAFContext.JNDI_NAME_JDO;

	private static final String APPLICATION =
		JDODataAccessService.class.getName();
	/*
	 * Is request name used in JARM Monitoring
	 */
	private static final String JARM_REQUEST = "CAF:RT:oal" + APPLICATION;
	/*
	 * Refers to the location of the trace files based on its configuration
	 * in the log-configuration.xml file
	 */
	private static final Location location =
		Location.getLocation(JDODataAccessService.class);

	/*
	 * Used to get and set values specific to a user session
	 */
	private UserContext userContext = null;
	
	public JDODataAccessService() throws DataAccessException {
		String method =  "JDODataAccessService()";
		try {
			Object[] _obj = { JNDI_NAME };
			if (pmf == null) {
				InitialContext jndi = new InitialContext();
				pmf = (PersistenceManagerFactory) jndi.lookup(JNDI_NAME);
			}
			location.debugT("JNDI look up of {0} Successful", _obj);
		} catch (NamingException e) {
			Object[] args = { JNDI_NAME };
			CAFPublicLogger.categoryCAF.logThrowableT(
				Severity.ERROR,
				location,
				method,
				"Error in JNDI lookup of {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		}
	}

	/** Returns instance of  JDODataAccessService  after initalizing the 
	 *  JDO PersistenceManagerFactory 
	 *   
	 * @throws  DataAccessException when the JNDI lookup of JDO PersistenceManagerFactory fails
	 */
	public static IDataAccessService getInstance() throws DataAccessException {
		String method = "getInstance()";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		IDataAccessService jdoDataAccessService = new JDODataAccessService();
		CAFPublicLogger.exiting(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);
		return jdoDataAccessService;
	}

	/**
	 * Resets the JDO PersistenceManagerFactory instance
	 */
	public void destroy() {
		//pmf = null;
	}

	/**
	 * Deletes a JDO Persistent instance.
	 * 
	 * @param obj Instance of IDependentObject to be removed
	 * @throws DataAccessException Unable to remove the persistent instance
	 */
	public void remove(IDependentObject obj) throws DataAccessException {
		String method = "remove(IDependentObject)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { obj },
			CAFPublicLogger.LEVEL_MEDIUM);
		PersistenceManager pm = null;
		try {
			// For CAF&BW integration lock
			if(CAFContext.CAF_BW_INTEGRATION_ENABLED) {
				if(BWProcessorLocalFactory.getProcessorLocalInstance().checkIsCAFServiceLogicallyLocked(obj.getObjectType())) {
					throw new DataAccessException("BO_LOCKED_BY_CAF_BW_INTEGRATION", new Object[]{obj.getObjectType()});
				}				
			}
			pm = pmf.getPersistenceManager();
			SAPJDOHelper.registerPCClass(obj.getClass());
			Object removeObj = pm.getObjectById(obj.getPK(), true);
			pm.deletePersistent(removeObj);
			
			DeletedObjectsRegistrationService.registerDelete(obj);
		} catch (Exception e) {
			Object[] args = { obj };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error removing Business-Object-Instance of {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			this.close(pm);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}

	/**
	 * Creates a JDO Persistent instance.
	 * 
	 * @param obj Instance of IDependentObject to be created
	 * @throws DataAccessException Unable to create the persistent instance
	 */
	public String create(IDependentObject obj) throws DataAccessException {
		String method = "create(IDependentObject)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { obj.getKey()},
			CAFPublicLogger.LEVEL_MEDIUM);
		PersistenceManager pm = null;
		try {
			// For CAF&BW integration lock
			if(CAFContext.CAF_BW_INTEGRATION_ENABLED) {
				if(BWProcessorLocalFactory.getProcessorLocalInstance().checkIsCAFServiceLogicallyLocked(obj.getObjectType())) {
					throw new DataAccessException("BO_LOCKED_BY_CAF_BW_INTEGRATION", new Object[]{obj.getObjectType()});
				}				
			}
			pm = pmf.getPersistenceManager();
			if (userContext != null)
				pm.setUserObject(userContext);
			pm.makePersistent(obj);

		} catch (javax.jdo.JDOUserException e) {
			Object[] args = { obj };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"JDOUserException in JDO createPersistent of {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} catch (Exception e) {
			Object[] args = { obj.getKey()};
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in JDO createPersistent using key {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			this.close(pm);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
		return obj.getKey();
	}

	public void destroy(Object obj) {
		// not implemented for JDO
	}

	/** 
	 * Retrieves the JDO Instance by PK.
	 * 
	 * @param objId  Id of the object to be retrieved
	 * @param objClass Class of the object to be retrieved 
	 * @return Object Instance of retrieved object
	 * @throws DataAccessException if unable to retrieve the object
	 */
	public Object findByPrimaryKey(Object objId, Class objectClass)
		throws DataAccessException {
		String method = "findByPrimaryKey(Object,Class)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { objId, objectClass },
			CAFPublicLogger.LEVEL_MEDIUM);

		Object keyObj = new Object();
		PersistenceManager pm = null;
		try {
			pm = pmf.getPersistenceManager();
			SAPJDOHelper.registerPCClass(objectClass);
			keyObj = pm.getObjectById(objId, true);
		} catch (javax.jdo.JDODataStoreException e) {
			Object[] args = { objId, objectClass };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in JDO getPersistent of ObjectId {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			this.close(pm);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
		return keyObj;
	}

	/**
	 * Retrieves all instances related to a class
	 * 
	 * @param objectClass class whose instances are to be retrieved
	 * @param operationName the name of the findBy operation (as given in the modeller) 
	 * @return Collection Collection of retrieved objects
	 * @throws DataAccessException if unable to retrieve the related instances
	 */
	public Collection query(Class objectClass, String operationName) throws DataAccessException {
		String method = "query(Class)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { objectClass },
			CAFPublicLogger.LEVEL_MEDIUM);
		Collection resultSet = null;
		PersistenceManager pm = null;
		try {
			pm = pmf.getPersistenceManager();
			if (userContext != null)
				pm.setUserObject(userContext);
			Extent ext = pm.getExtent(objectClass, true);
			Iterator iter = ext.iterator();
			Query query = null;
			query = pm.newQuery(ext);
			resultSet = (Collection) query.execute();
			location.debugT(
				"Size of result set retrieved is: {0}",
				new Object[] { Integer.toString(resultSet.toArray().length)});
		} catch (Exception e) {
			Object[] args = { objectClass };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in JDO getPersistent of ObjectClass {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			this.close(pm);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
		return resultSet;
	}

	/**
	 * Retrieves all instances related to a class, filter, parameters
	 *  and Map values
	 * @param objectClass class whose instances are to be retrieved
	 * @param filter query filter
	 * @param param query parameter
	 * @param hm  HashMap of query parameters
	 * @param operationName the name of the findBy operation (as given in the modeller)
 	 * @return Collection Collection of retrieved objects
	 * @throws DataAccessException if unable to retrieve the related instances
	 */
	public Collection query(
		Class objectClass,
		String filter,
		String param,
		Map hm,
		String operationName)
		throws DataAccessException {
		String method = "query(Class,String,String,Map)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { objectClass, filter, param, hm },
			CAFPublicLogger.LEVEL_MEDIUM);

		Collection resultSet = null;
		PersistenceManager pm = null;
		try {
			pm = pmf.getPersistenceManager();
			if (userContext != null)
				pm.setUserObject(userContext);
			Extent ext = pm.getExtent(objectClass, true);
			Query query = null;
			query = pm.newQuery(ext, filter);
			query.declareParameters(param);
			resultSet = (Collection) query.executeWithMap(hm);
			location.debugT(
				"Size of result set retrieved is: {0}",
				new Object[] { Integer.toString(resultSet.toArray().length)});
		} catch (Exception e) {
			Object[] args = { objectClass, filter, param, hm };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in JDO getPersistent of ObjectClass {0} using filter {1}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			this.close(pm);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
		return resultSet;
	}

	/**
	 * Retrieves all instances related to a class and query filter
	 * 
	 * @param objectClass class whose instances are to be retrieved
	 * @param intFilter query filter
	 * @param operationName the name of the findBy operation (as given in the modeller)
 	 * @return Collection Collection of retrieved objects
	 * @throws DataAccessException if unable to retrieve the related instances
	 */
	public Collection query(Class objectClass, IntQueryFilter intFilter, String operationName)
		throws DataAccessException {
		String method = "query(Class,IntQueryFilter)";
		String filter = null;
		Collection resultSet = null;

		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] {
				objectClass,
				intFilter.attribute,
				intFilter.condition,
				intFilter.getAttributeValue().toString(),
				intFilter.getAttributeValueHigh().toString()},
			CAFPublicLogger.LEVEL_MEDIUM);

		if (intFilter == null) {
			//TODO: change exception class - use standard?
			throw new DataAccessException(new NullPointerException("query filter is null"));
		} else if (intFilter.isDate == true) {
			IntQueryFilter[] intDateFilter = { intFilter };
			resultSet = query(objectClass, intDateFilter, operationName);
		} else {
			PersistenceManager pm = null;
			try {
				filter = intFilter.getJdoFilter();
				pm = pmf.getPersistenceManager();
				if (userContext != null)
					pm.setUserObject(userContext);
				Extent ext = pm.getExtent(objectClass, true);
				Query query = null;
				query = pm.newQuery(ext, filter);
				String jdoVariables = intFilter.getJdoCollVariables();
				if (jdoVariables != null) { 
					query.declareVariables(jdoVariables);
				}
				resultSet = (Collection) query.execute();
				location.debugT(
					"Size of result set retrieved is: {0} using attribute : {1} condition : {2} value low : {3} value high {4}",
					new Object[] {
						Integer.toString(resultSet.toArray().length),
						intFilter.attribute,
						intFilter.condition,
						intFilter.getAttributeValue().toString(),
						intFilter.getAttributeValueHigh().toString()});
			} catch (Exception e) {
				Object[] args =
					{
						objectClass,
						intFilter.attribute,
						intFilter.condition,
						intFilter.getAttributeValue().toString(),
						intFilter.getAttributeValueHigh().toString()};
				CAFPublicLogger.traceThrowableT(
					Severity.DEBUG,
					location,
					method,
					"Error in JDO getPersistent of ObjectClass {0} using attribute : {1} condition : {2} value low : {3} value high {4} ",
					args,
					e);
				location.throwing(method, e);
				throw new DataAccessException(e);
			} finally {
				this.close(pm);
				CAFPublicLogger.exiting(
					null,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
			}
		}
		return resultSet;

	}

	/**
	 * Retrieves all instances related to a class and query filter array
	 * 
	 * @param objectClass class whose instances are to be retrieved
	 * @param intFilter query filter array
	 * @param operationName the name of the findBy operation (as given in the modeller)
	 * @return Collection Collection of retrieved objects
	 * @throws DataAccessException if unable to retrieve the related instances
	 */
	public Collection query(Class objectClass, IntQueryFilter[] intFilter, String operationName)
		throws DataAccessException {
		String method = "query(Class,IntQueryFilter[])";
		String filter = null;
		String typeString = null;
		String paramString = null;
		Map queryParameters = new HashMap();
		HashSet declaredVariables = null;
		Collection resultSet = null;
		PersistenceManager pm = null;

		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			CAFPublicLogger.LEVEL_MEDIUM);

		try {
			for (int i = 0; i < intFilter.length; i++) {
				if (intFilter[i] != null && !intFilter[i].isEmpty()) {
					filter =
						IntQueryFilter.joinFilterMap(
							filter,
							intFilter[i],
							null);

					String jdoVariables = intFilter[i].getJdoCollVariables();
					if (jdoVariables != null) {
						if (declaredVariables==null) {
							declaredVariables = new HashSet(); 
						}
						declaredVariables.add(jdoVariables);
					}
					
					// joinFilterMap assigns proper attribute names by calling getJdoFilterMap. 
					// These names must be used as names of attributes in the map.
					String attrName1 = intFilter[i].getAttributeName1();
					String attrName2 = intFilter[i].getAttributeName2();
					if (!queryParameters.containsKey(attrName1))
						//note keys must be unique
						queryParameters.put(
							(attrName1),
							intFilter[i].getAttributeValue());
					if ("<>".equals(intFilter[i].condition))
						//find value between case
						queryParameters.put(
							(attrName2),
							intFilter[i].getAttributeValueHigh());
				}
			}

			Iterator iter = queryParameters.entrySet().iterator();
			while (iter.hasNext()) {
				Map.Entry param = (Map.Entry) iter.next();
				Object value = param.getValue();
				String type =
					(value != null
						? value.getClass().getName()
						: "java.lang.String");
				String parameter = param.getKey().toString();
				if (paramString != null)
					paramString = paramString + ", " + type + " " + parameter;
				else
					paramString = type + " " + parameter;
				if (typeString != null)
					typeString = typeString + "; import " + type;
				else
					typeString = "import " + type;
			}

			//SAPJDOHelper.registerPCClass(objectClass);

			pm = pmf.getPersistenceManager();
			if (userContext != null)
				pm.setUserObject(userContext);
			Extent ext = pm.getExtent(objectClass, true);
			Query query = null;
			query = pm.newQuery(ext, (filter == null ? "" : filter));
			if (typeString != null) {
				query.declareImports(typeString);
			}
			if (paramString != null) {
				query.declareParameters(paramString);
			}
			if (declaredVariables != null) {
				StringBuffer sb  = new StringBuffer();
				for (Iterator i=declaredVariables.iterator();i.hasNext();) {
					if (sb.length()>0) {
						sb.append(';');
					}
					sb.append(i.next());
				}
				query.declareVariables(sb.toString());
			}

			if (queryParameters == null || queryParameters.isEmpty()) {
				resultSet = (Collection) query.execute();
			} else {
				resultSet = (Collection) query.executeWithMap(queryParameters);
			}
			location.debugT(
				"Size of result set retrieved is: {0} using queryParameters : {1} declaredVariables : {2} paramString : {3} typeString : {4} ",
				new Object[] {
					Integer.toString(resultSet.toArray().length),
					queryParameters,
					declaredVariables,
					paramString,
					typeString });
			//			TODO: change exception class - use standard?
		} catch (BaseRuntimeException e) {
			Object[] args = { objectClass, e.getLocalizedMessage()};
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in JDO getPersistent of ObjectClass {0} as : {1} ",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} catch (Exception e) {
			Object[] args =
				{
					objectClass,
					queryParameters,
					declaredVariables,
					paramString,
					typeString };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in JDO getPersistent of ObjectClass {0} using queryParameters : {1} declaredVariables : {2} paramString : {3} typeString : {4} ",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			this.close(pm);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
		return resultSet;

	}

	/**
	 * Retrieves all instances related to a usage, class and query filter array
	 * 
 	 * @param usage
	 * @param objectClass class whose instances are to be retrieved
	 * @param intFilter query filter
	 * @param operationName the name of the findBy operation (as given in the modeller)
	 * @return Collection Collection of retrieved objects
	 * @throws DataAccessException if unable to retrieve the related instances
	 */
	public Collection query(
		String usage,
		Class objectClass,
		IntQueryFilter[] filter,
		String operationName)
		throws DataAccessException {
		//TODO: implement method, temporary workaround applied
		return query(objectClass, filter, operationName);
	}

	/** 
	 * Retrieves the JDO Instance by PK.
	 * 
	 * @param objId  Id of the object to be retrieved
	 * @param objClass Class of the object to be retrieved 
	 * @return Object Instance of retrieved object
	 * @throws DataAccessException if unable to retrieve the object
	 */
	public Object load(Object objId, Class objClass)
		throws DataAccessException {
		String method = "load(Object,Class)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { objId, objClass },
			CAFPublicLogger.LEVEL_MEDIUM);

		Object keyObj = new Object();
		PersistenceManager pm = null;
		try {
			pm = pmf.getPersistenceManager();
			if (userContext != null)
				pm.setUserObject(userContext);
			SAPJDOHelper.registerPCClass(objClass);
			keyObj = pm.getObjectById(objId, true);
		} catch (Exception e) {
			Object[] args = { objId, objClass };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in JDO load of ObjectClass {0} using filter {1}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			this.close(pm);
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
		return keyObj;
	}

	/**
	 * Not relavent for local persistence
	 */
	public void store(IDependentObject obj) throws DataAccessException {
		// not implemented for JDO
		// For CAF&BW integration lock
		if(CAFContext.CAF_BW_INTEGRATION_ENABLED) {
			try {
				if(BWProcessorLocalFactory.getProcessorLocalInstance().checkIsCAFServiceLogicallyLocked(obj.getObjectType())) {
					throw new DataAccessException("BO_LOCKED_BY_CAF_BW_INTEGRATION", new Object[]{obj.getObjectType()});
				}
			} catch(Exception exc) {
				throw new DataAccessException(exc);
			}			
		}
	}

	/**
	 * Not relavent for local persistence
	 */
	public void activate() {
		// not implemented for JDO
	}

	/**
	 * Not relavent for local persistence
	 */
	public void passivate() throws DataAccessException {
		// not implemented for JDO
	}

	/**
	 * Closes JDO PersistenceManager instance
	 * 
	 * @param pm JDO PersistenceManager instance
	 * @throws DataAccessException Unable to close JDO PersistenceManager instance
	 */
	public void close(PersistenceManager pm) throws DataAccessException {
		String method = "close(PersistenceManager)";
		CAFPublicLogger.entering(
			null,
			JARM_REQUEST,
			method,
			location,
			new Object[] { pm },
			CAFPublicLogger.LEVEL_MEDIUM);
		try {
			if (pm != null) {
				pm.close();
				pm = null;
			}
		} catch (Exception e) {
			Object[] args = { pm };
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in close of JDO PersistenceManager {0}",
				args,
				e);
			location.throwing(method, e);
			throw new DataAccessException(e);
		} finally {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}

	/**
	 *  Not relavent for local persistence
	 * 
	 */
	public Object execute(
		String action,
		Class objClass,
		Object input,
		Object result)
		throws DataAccessException {
		throw new DataAccessException("Operation not Supported");
	}

	/**
	 * Sets the userContext Properties 
	 * 
	 * @param userCtx
	 */
	public void setUserContext(UserContext userCtx) {
		this.userContext = userCtx;
	}
}
