/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 */
package com.sap.tc.cmi.metadata;

import java.util.Collection;
import java.util.Iterator;

/**
 * Represents the root of runtime meta data of a CMI model
 * This type can be called, extended or implemented by applications or frameworks using CMI.
 *
 * @SAPCMIPart 1
 * @author Markus Cherdron, Stephan Ritter
 * @version $Id: //tc/CommonModelInterface/630_VAL_REL/src/_cmi_api/java/com/sap/tc/cmi/metadata/ICMIModelInfo.java#1 $
 */
public interface ICMIModelInfo extends ICMIAbstractInfo {

  /**
   * This model type means that the CMI model was created out of the import of
   * an RFC model
   */
  static public final CMIModelType TYPE_RFCADAPTER = new CMIModelType("RFCAdapter");

  /**
   * This model type means that the CMI model was created (at design time) by a
   * XMI import into the SAP Netweaver Developer Studio. Usually the imported XMI 
   * file was exported by an UML modelling tool.
   */
  static public final CMIModelType TYPE_REVERSEENGINEERED = new CMIModelType("ReverseEngineered");

  /**
   * This model type means that the CMI model consists only of usually few generic
   * model classes to which no typed access is possible or desirable
   */
  static public final CMIModelType TYPE_GENERIC = new CMIModelType("GenericModel");

  /**
   * @clientCardinality 0..1
   * @supplierCardinality 0..*
   * @label / & optional 
   */
  /*# ICMIModelClassInfo lnkICMIModelClassInfo; */

  /**
   * @supplierCardinality 0..*
   * @clientCardinality 1
   * @label / & optional
   * @link aggregation*/
  /*# ICMIRelationInfo lnkICMIRelationInfo; */

  /**
   * @supplierCardinality 1
   * @directed
   * @clientCardinality 0..*
   * @label / 
   */
  /*# CMIModelType lnkCMIModelType; */

  /**
   * returns a collection of all model class infos of this model.
   * The collection should be immutable.
   */
  Collection getModelClassInfos();

  /**
   * returns an iterator over all model class infos of this model.
   * The iterator should be immutable.
   */
  Iterator iterateModelClassInfos();

  /**
   * returns the model class infos for the model class with the given (full qualified) name.
   */
  ICMIModelClassInfo getModelClassInfo(String name);

  /**
   * returns <code>true</code>, if modelRelationInfos are accessible at runtime. A model implementation
   * that doesn't support this must return <code>false</code>, but must implement the class ICMIModelRelationRoleInfos
   * anyway.
   * @see ICMIRelationRoleInfo
   */
  boolean supportsModelRelationInfos();

  /**
   * returns a collection of all model class relation infos or <code>null</code>, if the model
   * implementation doesn't support relation infos.
   * The collection should be immutable.
   */
  Collection getModelRelationInfos();

  /**
   * returns an iterator over all model class relation infos or <code>null</code>, if the model
   * implementation doesn't support relation infos.
   * The iterator should be immutable.
   */
  Iterator iterateModelRelationInfos();

  /**
   * returns a collection of all model class relation infos with the given name
   * or <code>null</code>, if the model implementation doesn't support relation infos.
   * The collection should be immutable.
   */
  Collection getModelRelationInfo(String name);

  /**
   * returns the model type of the model according to the defined TYPE_... constants.
   */
  CMIModelType getModelType();
}
