/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 */
package com.sap.tc.cmi.metadata;

import java.util.Collection;
import java.util.Iterator;

import com.sap.dictionary.runtime.IStructure;

/**
 * Represents the metadata of a model class at runtime.
 * This type can be called, extendeded or implemented by applications or frameworks using CMI.
 *
 * @SAPCMIPart 1
 * @author Markus Cherdron, Stephan Ritter
 * @version $Id: //tc/CommonModelInterface/630_VAL_REL/src/_cmi_api/java/com/sap/tc/cmi/metadata/ICMIModelClassInfo.java#1 $
 */
public interface ICMIModelClassInfo extends ICMIAbstractInfo {

  /**
   * returns the name of the model class. There possible, this should be the full 
   * qualified class name of the model class. However, at least for generic
   * model implementations this makes not always sense. 
   */
  String getName();

  /**
   * Returns the metadata of the model to which the class belongs.
   */
  ICMIModelInfo getModelInfo();

  /**
   * @clientCardinality 1
   * @supplierCardinality 0..*
   * @label /
   * @link aggregationByValue 
   */
  /*# ICMIModelClassPropertyInfo lnkICMIModelClassPropertyInfo; */

  /**
   * Returns a collection of all properties of the class.
   * The collection should be immutable.
   */
  Collection getPropertyInfos();

  /**
   * returns an iterator over all properties of the classes.
   * The iterator should be immutable.
   */
  Iterator iteratePropertyInfos();

  /**
   * returns the property metadata of the model class for the property with the given
   * <code>name</code>.
   */
  ICMIModelClassPropertyInfo getPropertyInfo(String name);

  /**
   * @clientCardinality 1
   * @supplierCardinality 0..*
   * @directed
   * @supplierRole sourceRole
   * @label / 
   */
  /*# ICMIRelationRoleInfo lnkICMIRelationRoleInfo; */

  /**
   * returns the Java DDIC {@link com.sap.dictionary.runtime.IStructure} 
   * of the model class info.
   * <p>
   * A <code>ICMIModelClassInfo</code> can have an Java DDIC <code>IStructure</code>
   * associated. In this case, for all {@link com.sap.dictionary.runtime.IField}
   * instances in the associated <code>IStructure</code> instance, there can be a
   * corresponding {@link ICMIModelClassPropertyInfo} instance, 
   * that belongs to the <code>ICMIModelClassInfo</code>, with  
   * <pre>
   *   field.getDatatype().equals(propertyInfo.getDataType())
   * </pre>
   * <p>
   * For the time being all <code>IField</code> instances of an 
   * <code>IStructure</code> must have a corresponding 
   * <code>ICMIModelClassPropertyInfo</code> instance in the 
   * <code>ICMIModelClassInfo</code>. 
   * However in the future an <code>ICMIModelClassInfo</code> may contain only some
   * of the fields of the structure. Therefore today's clients must be
   * programmed already in a way that they work properly, even if some of the 
   * <code>IField</code> instances have no corresponding 
   * <code>ICMIModelClassPropertyInfo</code> instance.
   * <p> 
   * An <code>ICMIModelClassInfo</code> instance that has an association to an 
   * <code>IStructure</code> may contain also <code>ICMIModelClassPropertyInfo</code>
   * instances that have no corresponding <code>IField</code> instances in the associated
   * <code>IStructure</code>.
   * <p>
   * If an <code>ICMIModelClassInfo</code> is not associated to an <code>IStructure</code>, the method returns <code>null</code>.
   * <p>
   * An Model Implemention does not have to support this association. Then <code>null</code> will be
   * returned in all cases.  
   * 
   */
  IStructure getStructureType();
  
  /**
   * Returns a collection of all source roles of the class.
   * The collection should be immutable.
   */
  Collection getSourceRoleInfos();

  /**
   * Returns an iterator over all source roles of the class.
   * The iterator should be immutable.
   */
  Iterator iterateSourceRoleInfos();

  /**
   * Returns the metadata for the target role with the given name.
   * If there is no such role, <code>null</code> will be returned.
   */
  ICMIRelationRoleInfo getTargetRoleInfo(String targetRoleName);

  /**
   * Convenience method which returns the metadata for the model class to 
   * which this model class is related via a relation, where the relation is 
   * determined through the named target role of this model class.
   * If there is no such relation, <code>null</code> will be returned.</p>
   * Note, that the semantic of the <code>String</code> parameter changed from 
   * <code>sourceRoleName</code> to <code>targetRoleName</code> on 2003-04-16. Implementations must 
   * adapt to this change.
   * 
   * The behavior of this method is equivalent to 
   * <pre>
   * if (getTargetRoleInfo(targetRoleName) != null)
   *      return getTargetRoleInfo(targetRoleName).getModelClassInfo();
   * return null;  
   * </pre> 
   */
  ICMIModelClassInfo getRelatedModelClassInfo(String targetRoleName);

  /**
   * Returns <code>true</code>, if the class by this meta info implements
   * ICMIGenericModelClass and thereby provides generic access to 
   * properties and relation roles.
   * @see com.sap.tc.cmi.model.ICMIGenericModelClass
   */
  boolean isGeneric();
}
