/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.repository.service/60NW_SP_COR/src/_appproperties/java/api/com/sapportals/wcm/repository/service/appproperties/IApplicationProperties.java#7 $
 */

package com.sapportals.wcm.repository.service.appproperties;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.IPropertyName;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.enum.Operator;
import com.sapportals.wcm.util.uri.IRidList;
import com.sapportals.wcm.util.uri.RID;

/**
 * The application properties service is used to persist arbitrary data for a
 * resource. These so called "application properties" ({@link IAppProperty}) are
 * not saved as custom properties of a resource in the repositories. Instead, an
 * implementation of this service will have its own persistence e.g. a database.
 * This means that "application properties" are never returned with a {@link IResource#getProperty(IPropertyName)}
 * or {@link IResource#getProperties()} call (unless a
 * property filter is used to do so). <p>
 *
 * All properties managed by this service are bound to an application via an
 * application-defined namespace. Additionally, the properties can be
 * personalized, that means that a user ID is used as an additional key to
 * persist such properties. <p>
 *
 * An application properties service implementation must handle rename, move and
 * delete events of all repository instances it is attached to. In it's event
 * handlers it must update the persistence accordingly. <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @version $Id: //kmgmt/bc.rf.repository.service/dev/src/_appproperties/java/api/com/sapportals/wcm/repository/service/appproperties/IApplicationProperties.java#3
 *      $
 */
public interface IApplicationPropertiesService {

  public final static String SERVICE_ID = "AppPropertiesGlobalService";

  /**
   * Returns the personalized property for the specified resource and user ID.
   * Will return null if the property does not exist.
   *
   * @param propertyName The property name.
   * @param resourceID The resource identified.
   * @param userID The ID of the user the property belongs to.
   * @param context The resource context.
   * @return A property object or null if the property does not exist
   * @exception WcmException
   */
  public IAppProperty getProperty(IPropertyName propertyName, RID resourceID, String userID, IResourceContext context)
    throws WcmException;


  /**
   * Returns the specified personalized property for all resources IDs in the
   * list and the specified user ID.
   *
   * @param propertyName The property name
   * @param ridList A list of resource identifiers
   * @param userID The ID of the user the property belongs to
   * @param context The resource context.
   * @return A map: key = RID, value = IAppProperty
   * @exception WcmException
   */
  public IResourceAppPropertyMap getProperty(IPropertyName propertyName, IRidList ridList, String userID, IResourceContext context)
    throws WcmException;

  /**
   * Returns all personalized properties of a resource for a namespace and the
   * user ID.
   *
   * @param namespace The namespace
   * @param resourceID The resource identifier.
   * @param userID The ID of the user the property belongs to
   * @param context The resource context.
   * @return A map of properties: key = IAppPropertyName, value = IAppProperty
   * @exception WcmException
   */
  public IAppPropertyMap getProperties(String namespace, RID resourceID, String userID, IResourceContext context)
    throws WcmException;

  /**
   * Returns all personalized properties of a resource.
   *
   * @param resourceID The resource indentifier.
   * @param context The resource context.
   * @param userID TBD: Description of the incoming method parameter
   * @return A map of properties: key = IAppPropertyName, value = IAppProperty
   * @exception WcmException
   */
  public IAppPropertyMap getProperties(RID resourceID, String userID, IResourceContext context)
    throws WcmException;

  /**
   * Deletes a personalized property. If the user ID is <code>null</code> the
   * property will be deleted for all users.
   *
   * @param propertyName The name of the property to delete.
   * @param resourceID The resource identifier.
   * @param userID The user ID. If the user ID is <code>null</code> the property
   *      will be deleted for all users.
   * @param context The resource context.
   * @exception WcmException
   */
  public void deleteProperty(IPropertyName propertyName, RID resourceID, String userID, IResourceContext context)
    throws WcmException;


  //////////////////////////////////////////////////////////////////////////////

  /**
   * Returns the property for the specified resource identifier and name. Will
   * return null if the property does not exist.
   *
   * @param propertyName The property name.
   * @param resourceID The resource identifier.
   * @param context The resource context.
   * @return the property for the specified resource identifier and namespace.
   * @exception WcmException
   */
  public IAppProperty getProperty(IPropertyName propertyName, RID resourceID, IResourceContext context)
    throws WcmException;

  /**
   * Returns the properties with the specified name and for all RIDs in the
   * list.
   *
   * @param propertyName The property name.
   * @param ridList A list of resource identifiers.
   * @param context The resource context.
   * @return A map: key = RID, value = IAppProperty
   * @exception WcmException
   */
  public IResourceAppPropertyMap getProperty(IPropertyName propertyName, IRidList ridList, IResourceContext context)
    throws WcmException;

  /**
   * Returns all application properties for the specified namespace and RID (but
   * not the personalized properties).
   *
   * @param namespace The namespace.
   * @param resourceID The resource identifier.
   * @param context The resource context.
   * @return A map of properties: key = IAppPropertyName, value = IAppProperty
   * @exception WcmException
   */
  public IAppPropertyMap getProperties(String namespace, RID resourceID, IResourceContext context)
    throws WcmException;

  /**
   * Returns all application properties for the specified RID (not the
   * personalized properties)
   *
   * @param resourceID The resource identifier.
   * @param context The resource context.
   * @return A map of properties: key = IAppPropertyName, value = IAppProperty
   * @exception WcmException
   */
  public IAppPropertyMap getProperties(RID resourceID, IResourceContext context)
    throws WcmException;

  /**
   * Returns all personalized properties for a given property name and RID.
   *
   * @param propertyName the name of the property
   * @param resourceID the ressource identifier
   * @param context The resource context.
   * @return A collection of application properties
   * @exception WcmException
   */
  public IAppPropertyCollection getProperties(IPropertyName propertyName, RID resourceID, IResourceContext context)
    throws WcmException;

  /**
   * Deletes the property with the specified name from the persistent storage.
   *
   * @param propertyName The name of the property to delete.
   * @param resourceID The resource identifier
   * @param context The resource context.
   * @exception WcmException Exception raised in failure situation
   */
  public void deleteProperty(IPropertyName propertyName, RID resourceID, IResourceContext context)
    throws WcmException;


  //////////////////////////////////////////////////////////////////////////////

  /**
   * Stores the property for the specified RID in some persistent storage. If
   * the property object contains a user ID the property is personalized.
   *
   * @param property The property to store
   * @param resourceID The resource identifier.
   * @param context The resource context.
   * @exception WcmException
   */
  public void setProperty(IAppProperty property, RID resourceID, IResourceContext context)
    throws WcmException;

  /**
   * Queries all properties and returns a list of RIDs that match the search
   * criteria. The property's namespace, name and userID (optional) are compared
   * for equality . A query for the property's values is possible with different
   * operators and for all property types. The operators LIKE
   * and REGEXP can only be used for properties of type STRING and XML. <p>
   * The search for type Boolean is restricted to the operator EQUAL.
   *
   * The search result is a list of resource identifiers. All properties for all
   * repositories the service is registered at are searched, thus the result set
   * can include RIDs from different repositories. The RIDs in the result list
   * are unique. <p>
   *
   * Search for multi-valued properties is supported. At least one of the values
   * must match the specified value.
   *
   * @param name The property name to search for.
   * @param userID The userID to search for or <code>null</code> .
   * @param value The value to search for. Must be a String, Date, Integer,
   *      Long or Boolean object.
   * @param op The comparison operator to use.
   * @param multiValued If multi-valued properties will be included in the
   *      search.
   * @param context The resource context.
   * @return A list of resource identifiers. The RIDs in the list are unique.
   * @exception WcmException If an unsopprted operator was used If a syntax
   *      error occured in the regular expression
   */
  public IRidList search(IPropertyName name, String userID, Object value, Operator op, boolean multiValued, IResourceContext context)
    throws WcmException;

  public IAppProperty getProperty(IPropertyName propertyName, IResource resource, String userID)
    throws WcmException;

  public IAppPropertyMap getProperties(String namespace, IResource resource, String userID)
    throws WcmException;

  public IAppPropertyMap getProperties(IResource resource, String userID)
    throws WcmException;

  public void deleteProperty(IPropertyName propertyName, IResource resource, String userID)
    throws WcmException;

  public IAppProperty getProperty(IPropertyName propertyName, IResource resource)
    throws WcmException;

  public IAppPropertyMap getProperties(String namespace, IResource resource)
    throws WcmException;

  public void setProperty(IAppProperty property, IResource resource)
    throws WcmException;

  public IAppPropertyMap getProperties(IResource resource)
    throws WcmException;

  public IAppPropertyCollection getProperties(IPropertyName propertyName, IResource resource)
    throws WcmException;

  public void deleteProperty(IPropertyName propertyName, IResource resource)
    throws WcmException;

}