package com.sap.caf.rt.security.acl.impl.permission;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.SessionBean;

import com.sap.caf.rt.bol.IBusinessObject;
import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.security.acl.impl.CAFPermission;
import com.sap.caf.rt.security.acl.impl.SecurityServiceBase;
import com.sap.caf.rt.security.util.CAFPermissionCache;
import com.sap.caf.rt.security.util.CAFPermissionName;
import com.sap.caf.rt.security.util.CAFPermissionUtil;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.security.api.IPrincipal;
import com.sap.security.api.IPrincipalFactory;
import com.sap.security.api.UMException;
import com.sap.security.api.UMFactory;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;
/**
 * @ejbLocal <{com.sap.caf.rt.security.acl.impl.permission.PermissionLocal}>
 * @ejbLocalHome <{com.sap.caf.rt.security.acl.impl.permission.PermissionLocalHome}>
 * @stateless 
 */
public class PermissionBean extends SecurityServiceBase implements SessionBean {

	private static final String APPLICATION = PermissionBean.class.getName();
	private static final Location LOGGER = Location.getLocation(APPLICATION);
	private static final String JARMREQPREFIX = "CAF:RT:security:";
	private static final String JARMREQUEST = JARMREQPREFIX+APPLICATION;
	private static final int PERMISSIONNAME_COUNT = 8;
	

	/**
	 * Creates permission related to the specified business rule.  
	 * @param businessRuleID business rule id.
	 * @param principalID principal id.
	 * @param permissionName permission name.
	 * @return <code>Permission</code> data object.
	 * @throws ServiceException if some error is encountered.
	 */
	public void create(String businessRuleID, String principalID, String permissionName) throws ServiceException {
		final String method = "create(String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		String loginUserID = getCurrentUserID();
		try {
			IPrincipal iPrincipal = UMFactory.getPrincipalFactory().getPrincipal(principalID);
			CAFPermission.createPermission(loginUserID, businessRuleID, permissionName, iPrincipal);
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			try {
				CAFPermissionCache.invalidRuleCache();
				CAFPermissionCache.invalidPermissionCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
				throw new ServiceException("AUTH_RESET_CACHE");
			}

			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	/**
	* Search permission in UME by business rule id.
	* @param permissionID permission id.
	* @return <code>Permission</code> data object.
	* @throws ServiceException if some error is encountered.
	*/
	public boolean isPermissionExist(String businessRuleID, String principalID, String permissionName) throws ServiceException {
		final String method = "isPermissionExist(String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
	
		boolean result = false;
		try {
			String loginUserID = getCurrentUserID();

			IPrincipalFactory principalFactory = UMFactory.getPrincipalFactory();
			IPrincipal iPrincipal = principalFactory.getPrincipal(principalID);

			Collection permissions = CAFPermission.retrievePermission(
						loginUserID, businessRuleID, permissionName, iPrincipal);
						
			if (!permissions.isEmpty()) {
				result = true;
			} else { 
				result = false;
			}
			
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return result;
	}

	/**
	 * Updates permission.
	 * @throws ServiceException if some error is encountered.
	 */
	public void update(String businessRuleID, String permissionName, String nameBackUp, String principalID) throws ServiceException {
		final String method = "update(String, String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		final String loginUserID = getCurrentUserID();

		try {
			IPrincipal iPrincipal = UMFactory.getPrincipalFactory().getPrincipal(principalID);
			CAFPermission.modifyPermission(loginUserID, businessRuleID, nameBackUp, permissionName, iPrincipal);
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			try {
				CAFPermissionCache.invalidRuleCache();
				CAFPermissionCache.invalidPermissionCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
				throw new ServiceException("AUTH_RESET_CACHE");
			}
			
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	/**
	 * Removes a permission from UME.
	 * @throws ServiceException if some error is encountered.
	 */
	public void delete(String businessRuleID, String principalID, String permissionName) throws ServiceException {
		final String method = "delete(String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
		try {
			String loginUserID = getCurrentUserID();

			IPrincipal principal = UMFactory.getPrincipalFactory().getPrincipal(principalID);
			CAFPermission.removePermission(loginUserID, businessRuleID, permissionName, principal);
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			try {
				CAFPermissionCache.invalidRuleCache();
				CAFPermissionCache.invalidPermissionCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
				throw new ServiceException("AUTH_RESET_CACHE");
			}
			
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	/**
	* Returns collection of permissions related to this objectID. 
	* @return Collection of permissions.
	* @throws ServiceException if some error is encountered.
	*/
	public Collection findByObjectID(String objectID) throws ServiceException {
		final String method = "findByObjectID(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
		
		Collection result = new ArrayList();
		try {
			if (CAFPermission.getACL(objectID) == null) {
				return result;
			}
			
//			if (!isInstanceCheckedForPermission(objectTypeID)) {
//				return result;
//			}

			String currentUserID = getCurrentUserID();
			Collection permissions = CAFPermission.retrievePermission(currentUserID, objectID, null,(IPrincipal)null);
			final IPrincipalFactory principalFactory = UMFactory.getPrincipalFactory();
			for (Iterator iter = permissions.iterator(); iter.hasNext();) 
			{
				HashMap permissionMap = (HashMap) iter.next();
				
				for (Iterator permissionMapIter = permissionMap.keySet().iterator(); permissionMapIter.hasNext();) 
				{
					IPrincipal principal = (IPrincipal) permissionMapIter.next();
					
					String permissionName = (String) permissionMap.get(principal);
					String principalType = principalFactory.getPrincipalType(principal.getUniqueID());
					
					PermissionVo permissionVo = new PermissionVo();
					permissionVo.setBusinessRuleID(objectID);
					permissionVo.setPermissionName(permissionName);
					permissionVo.setPrincipalType(principalType);
					permissionVo.setPrincipalName(principal.getDisplayName());
					permissionVo.setPrincipalID(principal.getUniqueID());
					permissionVo.setNameBackUp(permissionName);
					 
					result.add(permissionVo);
				}
			}
			return result;
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	/**
	 * Returns collection of permissions related to this object id, principal id, objectType. 
	 * @return Collection of permissions.
	 * @throws ServiceException if some error is encountered.
	 */
	public Collection checkPermission(String principalID,String objectID,String objectType) throws ServiceException {			
			final String method = "checkPermission(String, String, String)";
			CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
			
			Collection result = new ArrayList();
			try {
				IPrincipalFactory principalFactory = UMFactory.getPrincipalFactory();
				IPrincipal principal = principalFactory.getPrincipal(principalID);
				String principalType = principalFactory.getPrincipalType(principalID);
				
				Object depObj = CAFPermissionUtil.getDependentObject(objectID,objectType);
				Collection permissionList =  CAFPermission.getAllowedPermission(principalID,depObj,objectType);
				
				for (Iterator iter = permissionList.iterator(); iter.hasNext();) {
					String permissionName = (String) iter.next();
					
					PermissionVo permissionVo = new PermissionVo();
					permissionVo.setBusinessRuleID(objectID);
					permissionVo.setPermissionName(permissionName);
					permissionVo.setPrincipalType(principalType);
					permissionVo.setPrincipalName(principal.getDisplayName());
					permissionVo.setPrincipalID(principalID);
					permissionVo.setNameBackUp(permissionName);
					 
					result.add(permissionVo);
				}
				return result;
			} catch (Exception e) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
				throw new ServiceException(e);
			} finally {
				CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
			}			
	}

	/**
	 * Returns collection of permissions related to this object id, principal id, objectType. 
	 * @return Collection of permissions.
	 * @throws ServiceException if some error is encountered.
	 */
	public Collection findByObjectType(String principalID, String objectType) throws ServiceException {			
			final String method = "findByObjectType(String, String)";
			CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
			
			HashMap inputSet = new HashMap();
			Collection result = new ArrayList();
			try {
				Collection objects = CAFPermissionUtil.getDependentObjects(objectType, inputSet) ;		
				IPrincipalFactory factory = UMFactory.getPrincipalFactory();
				IPrincipal principal = factory.getPrincipal(principalID) ;
				String principalType = factory.getPrincipalType(principalID) ;
				String permissionName ;
			
				//remove [create] permission from list, as it isn't valid permission for object instance
				Collection permissionNames = new ArrayList(PERMISSIONNAME_COUNT) ; 
				for (Iterator namesIterator = CAFPermissionName.getAllNames(); namesIterator.hasNext();){
					String name = ((CAFPermissionName)namesIterator.next()).getName() ;
					if (!name.equals(CAFPermissionName.create)){
						permissionNames.add(name) ;	
					}
				}

				for (Iterator objIterator = objects.iterator(); objIterator.hasNext();){
					Object object = objIterator.next() ;
					String objectID = ((IBusinessObject)object).getKey() ;		
					for (Iterator permNamesIterator = permissionNames.iterator(); permNamesIterator.hasNext();){
						permissionName = (String)permNamesIterator.next() ;
						if(CAFPermission.checkAclPermission(object, principal, permissionName, objectType, null, true))
						{
							PermissionVo permissionVo = new PermissionVo();
							permissionVo.setBusinessRuleID(objectID);
							permissionVo.setPermissionName(permissionName);
							permissionVo.setPrincipalType(principalType);
							permissionVo.setPrincipalName(principal.getDisplayName());
							permissionVo.setPrincipalID(principalID);
							permissionVo.setNameBackUp(permissionName);
					 
							result.add(permissionVo);
						}
					}
				}
				return result;
			} catch (Exception e) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
				throw new ServiceException(e);
			} finally {
				CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
			}			
	}

	/**
	 * Returns collection of permissions related to this business rule id, principal id, objectType. 
	 * @return Collection of permissions.
	 * @throws ServiceException if some error is encountered.
	 */
	public Collection findByPrincipalBusinessRules(String businessRuleID, String principalID, String objectType) throws ServiceException {			
			final String method = "findByPrincipalBusinessRules(String, String, String)";
			CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
			
			HashMap inputSet = new HashMap();
			Collection result = new ArrayList();
			try {
				Collection objects = CAFPermissionUtil.getDependentObjects(objectType, inputSet) ;		
				IPrincipalFactory factory = UMFactory.getPrincipalFactory();
				IPrincipal principal = factory.getPrincipal(principalID) ;
				String principalType = factory.getPrincipalType(principalID) ;
				String permissionName;
			
				//remove [create] permission from list, as it isn't valid permission for object instance
				Collection permissionNames = new ArrayList(PERMISSIONNAME_COUNT) ; 
				for (Iterator namesIterator = CAFPermissionName.getAllNames(); namesIterator.hasNext();){
					String name = ((CAFPermissionName)namesIterator.next()).getName() ;
					if (!name.equals(CAFPermissionName.create)){
						permissionNames.add(name) ;	
					}
				}
				
				for (Iterator objIterator = objects.iterator(); objIterator.hasNext();){
					Object object = objIterator.next() ;
					String objectID = ((IBusinessObject)object).getKey() ;		
					for (Iterator permNamesIterator = permissionNames.iterator(); permNamesIterator.hasNext();){
						permissionName = (String)permNamesIterator.next() ;
						if(CAFPermission.checkAclPermission(object, principal, permissionName, objectType, businessRuleID, false))
						{
							PermissionVo permissionVo = new PermissionVo();
							permissionVo.setBusinessRuleID(objectID);
							permissionVo.setPermissionName(permissionName);
							permissionVo.setPrincipalType(principalType);
							permissionVo.setPrincipalName(principal.getDisplayName());
							permissionVo.setPrincipalID(principalID);
							permissionVo.setNameBackUp(permissionName);
					 
							result.add(permissionVo);
						}
					}
				}
				return result;
			} catch (Exception e) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
				throw new ServiceException(e);
			} finally {
				CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
			}			
	}

	/**
	* Returns collection of permissions related to this buisiness rule. 
	* @param businessRuleID business rule id.
	* @return Collection of permissions.
	* @throws ServiceException if some error is encountered.
	*/
	public Collection findByBusinessRuleID(String currentUserID, String businessRuleID) throws ServiceException {
		final String method = "findByBusinessRuleID(String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
		
		Collection result = new ArrayList();
		PermissionVo permissionVo;
		try {
			Collection permissions = CAFPermission.retrievePermission(currentUserID, businessRuleID, null,(IPrincipal)null);
			final IPrincipalFactory principalFactory = UMFactory.getPrincipalFactory();
			for (Iterator iter = permissions.iterator(); iter.hasNext();) {
				HashMap permissionMap = (HashMap) iter.next();
				for (Iterator permissionMapIter = permissionMap.keySet().iterator(); permissionMapIter.hasNext();) {
					permissionVo = new PermissionVo();

					permissionVo.setBusinessRuleID(businessRuleID);
					
					IPrincipal principal = (IPrincipal) permissionMapIter.next();
					String principalID = principal.getUniqueID();

					String permissionName = (String) permissionMap.get(principal);
					permissionVo.setKey(businessRuleID + "|" + principalID + "|" + permissionName);
					permissionVo.setPrincipalID(principalID);
					permissionVo.setPrincipalName(principal.getDisplayName());
					permissionVo.setPrincipalType(principalFactory.getPrincipalType(principalID));
					permissionVo.setPermissionName(permissionName);
					permissionVo.setNameBackUp(permissionName);

					result.add(permissionVo);
				}
			}
			return result;
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1); 
		}
	}

}
