package com.sap.caf.rt.security.acl.impl.objecttype;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.ejb.SessionBean;

import com.sap.caf.metamodel.Application;
import com.sap.caf.metamodel.ApplicationInterface;
import com.sap.caf.metamodel.BusinessEntityInterface;
import com.sap.caf.metamodel.DataObject;
import com.sap.caf.metamodel.MOFInterface;
import com.sap.caf.metamodel.Operation;
import com.sap.caf.metamodel.Permission;
import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.metamodel.RepositoryConnection;
import com.sap.caf.rt.security.acl.impl.SecurityServiceBase;
import com.sap.caf.rt.security.util.CAFPermissionUtil;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.ip.mmr.IConnection;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * @ejbLocal <{com.sap.caf.rt.security.acl.impl.objecttype.ObjectTypeLocal}>
 * @ejbLocalHome <{com.sap.caf.rt.security.acl.impl.objecttype.ObjectTypeLocalHome}>
 * @stateless 
 * @transactionAttribute Required
 */

/**
 * @author c5051699
 */

public class ObjectTypeBean extends SecurityServiceBase implements SessionBean {

	private static final String APPLICATION = ObjectTypeBean.class.getName();
	private static final Location LOGGER = Location.getLocation(APPLICATION);
	private static final String JARMREQPREFIX = "CAF:RT:security:";
	private static final String JARMREQUEST = JARMREQPREFIX+APPLICATION;

	public Collection findByAllObjectTypes() throws ServiceException {
		final String method = "findByAllObjectType()";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		Collection resultSet = null;
		try {
			resultSet = getObjectTypeVos();
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return resultSet;
	}

	public ObjectTypeVo findByObjectType(String objectType) throws ServiceException {
		final String method = "findByObjectType(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		ObjectTypeVo objectTypeVo = null;
		try {
			DataObject dataObject = CAFPermissionUtil.getDataObject( objectType );
			String objectTypeID = dataObject.getObjectId();

			objectTypeVo = new ObjectTypeVo(objectTypeID, objectType);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return objectTypeVo;
	}

	public ObjectTypeVo findByPrimaryKey(String objectTypeID) throws ServiceException {
		final String method = "findByPrimaryKey(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		ObjectTypeVo objectTypeVo = null;
		try {
			DataObject dataObject = CAFPermissionUtil.getDataObjectByGUID( objectTypeID );
			BusinessEntityInterface businessEntityInterface = dataObject.getBusinessEntityInterface();
			Application app = businessEntityInterface.getApplication();

			String providerName;
			String applicationName;

			if (app!=null){
				providerName = app.getProviderName();
				applicationName = app.getObjectName();
			} else {
				providerName = "provider";
				applicationName = "application";
			}

			String objectType = providerName + "/" + applicationName	+ "/"	+ dataObject.getObjectName();
			objectTypeVo = new ObjectTypeVo(objectTypeID, objectType);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return objectTypeVo;
	}

	private Collection getObjectTypeVos() throws ServiceException {
		final String method = "getObjectTypeVos()";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		Collection result = new ArrayList();

		try {
			List allInterfaces = getAllInterfaces();
			for (Iterator it = allInterfaces.iterator(); it.hasNext();)	{
				MOFInterface mofInterface = (MOFInterface)it.next();
				if (isBusinessEntityInterfaceSecured(mofInterface)) {
					ObjectTypeVo objectTypeVo = new ObjectTypeVo();
												
					BusinessEntityInterface beInterface = (BusinessEntityInterface)mofInterface;
					DataObject dataObject = beInterface.getBusinessEntity();
					String objectId = dataObject.getObjectId();

					BusinessEntityInterface businessEntityInterface = dataObject.getBusinessEntityInterface();
					Application app = businessEntityInterface.getApplication();

					String providerName;
					String applicationName;

					if (app!=null) {
						providerName = app.getProviderName();
						applicationName = app.getObjectName();
					} else {
						providerName = "provider";
						applicationName = "application";
					}

					String objectTypeFullName = providerName + "/" + applicationName + "/" + dataObject.getObjectName();

					objectTypeVo.setKey( objectId );
					objectTypeVo.setObjectType( objectTypeFullName );
					result.add(objectTypeVo);
				}
			}
			return result;									
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	private List getAllInterfaces() throws ServiceException {
		List allInterfaces = new ArrayList();
		try {
			IConnection connection = RepositoryConnection.getConnection();
			Collection applications = null;
								
			List typeList = new ArrayList();
			typeList.add("com");
			typeList.add("sap");
			typeList.add("caf");
			typeList.add("metamodel");
			typeList.add("Application");
					
			applications = connection.queryM1(typeList, null, true);
			for (Iterator it = applications.iterator(); it.hasNext();) {
				Application application = (Application)it.next();
				for(Iterator iter = application.getBaseObjects().iterator();iter.hasNext();) {
					Object obj = iter.next();
					if(obj instanceof BusinessEntityInterface || obj instanceof ApplicationInterface) {
						allInterfaces.add(obj);
					} 
				}
			}
			return allInterfaces;
		} catch (Exception ex) {
			throw new ServiceException(ex);
		}
	}

	private boolean isBusinessEntityInterfaceSecured(MOFInterface mofInterface) throws ServiceException {
		try {
			boolean result = false;
			if (mofInterface instanceof BusinessEntityInterface) {
				BusinessEntityInterface businessEntityInterface = (BusinessEntityInterface)mofInterface;
				if (businessEntityInterface.getInstancePermission() != null &&
							businessEntityInterface.getInstancePermission().intValue() > 0) {
					result = true;
    		} else { 
					List operationMappings = businessEntityInterface.getOperations();
					for (Iterator iter = operationMappings.iterator(); iter.hasNext();)	{
						Operation operation = (Operation)iter.next();
						
						Permission permission = operation.getPermission();
						if (permission != null &&	permission.getImplicitCheck()!= null 
								&& permission.getImplicitCheck().intValue() > 0) {
							result = true;
							break;
						}			 
					}
				}
			}
			return result;
		} catch (Exception ex) {
			throw new ServiceException(ex);
		}
	}
}