package com.sap.caf.rt.security.acl.impl.condition;

import java.util.Collection;
import java.util.HashMap;

import javax.ejb.SessionBean;

import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.security.acl.impl.SecurityServiceBase;
import com.sap.caf.rt.security.util.CAFPermissionCache;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.caf.rt.exception.DataAccessException;
import com.sap.caf.rt.bol.pk.PrimaryKeyFactory;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * @ejbLocal <{com.sap.caf.rt.security.acl.impl.condition.ConditionLocal}>
 * @ejbLocalHome <{com.sap.caf.rt.security.acl.condition.ConditionLocalHome}>
 * @stateless 
 * @transactionAttribute Required
 */

/**
 * @author c5051699
 */
public class ConditionBean extends SecurityServiceBase implements SessionBean {

	private static final String APPLICATION = ConditionBean.class.getName();
	private static final Location LOGGER = Location.getLocation(APPLICATION);
	private static final String JARMREQPREFIX = "CAF:RT:security:";
	private static final String JARMREQUEST = JARMREQPREFIX+APPLICATION;

	public void update(String conditionID, String attributeName, String attributeValue) throws ServiceException {
		checkUpdatePermission();

		final String method = "update(String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			ConditionVo conditionVo = findByPrimaryKey(conditionID);
			conditionVo.setAttributeName(attributeName);
			conditionVo.setAttributeValue(attributeValue);
			getJDODataAccessService().store(conditionVo);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			//reset Cache
			try {
				CAFPermissionCache.invalidRuleCache();
				CAFPermissionCache.invalidPermissionCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
				throw new ServiceException("AUTH_RESET_CACHE");
			}
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	public String create(String businessRuleID) throws ServiceException {
		String conditionID = null;

		final String method = "create(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
		
		ConditionVo conditionVo = null;
		checkUpdatePermission();

		if (businessRuleID == null) {
			ServiceException ruleException = new ServiceException("AUTH_CREATE_CONDITION");
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, ruleException.getMessage(), ruleException);
			throw ruleException;
		}

		try {
			conditionVo = new ConditionVo();
			conditionVo.setBusinessRuleID(businessRuleID);
			try {
				conditionID = PrimaryKeyFactory.getInstance().getPrimaryKey();
			} catch (DataAccessException e) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
				throw new ServiceException(e);
			}
			conditionVo.setKey(conditionID);
			getJDODataAccessService().create(conditionVo);
			return conditionID;
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			//reset Cache
			try {
				CAFPermissionCache.invalidRuleCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
				throw new ServiceException("AUTH_RESET_CACHE");
			}
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}
	
	public void delete(String conditionID) throws ServiceException {
		checkUpdatePermission();
		
		final String method = "delete(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		if (conditionID == null) {
			ServiceException ruleException = new ServiceException("AUTH_DELETE_CONDITION");
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, ruleException.getMessage(), ruleException);
			throw ruleException;
		}
		ConditionVo conditionVo = null;
		try {
			conditionVo = findByPrimaryKey(conditionID);
			getJDODataAccessService().remove(conditionVo);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			//reset Cache
			try {
				CAFPermissionCache.invalidRuleCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
				throw new ServiceException("AUTH_RESET_CACHE");
			}
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}


	public ConditionVo findByPrimaryKey(String conditionID) throws ServiceException {
		ConditionVo conditionVo = null;

		final String method = "findByPrimaryKey(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			String param = "String key";
			String filter = "this.key == key";
			HashMap hm = new HashMap();
			hm.put("key", conditionID);
			//transaction already handled using CMT
			Collection resultSet = 
				getJDODataAccessService().query(ConditionVo.class, filter, param, hm, null);
			conditionVo = (ConditionVo)resultSet.iterator().next();
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return conditionVo;
	}

	public Collection findByBusinessRule(String businessRuleID) throws ServiceException {
		final String method = "findByBusinessRule(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
		
		//now get the attribute set for this object type
		String param = "String businessRuleID";
		String filter = "this.businessRuleID == businessRuleID";
		HashMap hm = new HashMap();
		hm.put("businessRuleID", businessRuleID);

		try {
			//transaction handled using CMT
			Collection resultSet = getJDODataAccessService().query(ConditionVo.class, filter, param, hm, null);
			return resultSet;
		} catch (DataAccessException dae) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, dae.getMessage(), dae);
			throw new ServiceException(dae);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

}
