package com.sap.caf.rt.security.acl.impl.businessrule;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;

import javax.ejb.SessionBean;

import com.sap.caf.rt.bol.IDependentObject;
import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.security.acl.impl.CAFPermission;
import com.sap.caf.rt.security.acl.impl.SecurityServiceBase;
import com.sap.caf.rt.security.acl.impl.condition.ConditionLocal;
import com.sap.caf.rt.security.acl.impl.condition.ConditionVo;
import com.sap.caf.rt.security.acl.impl.objectattribute.ObjectAttributeLocal;
import com.sap.caf.rt.security.acl.impl.objectattribute.ObjectAttributeVo;
import com.sap.caf.rt.security.acl.impl.objecttype.ObjectTypeLocal;
import com.sap.caf.rt.security.util.CAFBusinessRuleList;
import com.sap.caf.rt.security.util.CAFPermissionCache;
import com.sap.caf.rt.security.util.CAFPermissionName;
import com.sap.caf.rt.security.util.CAFPermissionUtil;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.caf.rt.bol.pk.PrimaryKeyFactory;
import com.sap.security.api.IPrincipal;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * @ejbLocal <{com.sap.caf.rt.security.acl.impl.businessrule.BusinessRuleLocal}>
 * @ejbLocalHome <{com.sap.caf.rt.security.acl.impl.businessrule.BusinessRuleLocalHome}>
 * @stateless 
 * @transactionAttribute Required
 */

public class BusinessRuleBean extends SecurityServiceBase implements SessionBean {

	private static final String APPLICATION = BusinessRuleBean.class.getName();
	private static final Location LOGGER = Location.getLocation(APPLICATION);
	private static final String JARMREQPREFIX = "CAF:RT:security:";
	private static final String JARMREQUEST = JARMREQPREFIX+APPLICATION;

	/** 
	 * This API creates a new business rule which includes rule name and associated conditions. <p>
	 * 
	 * @param  objectType selected object(eg, xPD/sap.com/Concept) 
	 * @param  ruleName as entered by the user 
	 * @param conditions hashMap of (attributeName, condition). Please see the note above.
	 * @return ruleHeaderId is the id linking the business rule to the ACL (access control list). ACL is stored in UM database
	 * @exception ServiceException 
	 * @transactionAttribute Required
	 */

	public String create(String objectTypeID, String ruleName, HashMap conditions) throws ServiceException {
		checkUpdatePermission();

		final String method = "create(String, String, HashMap)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
		
		String businessRuleID = null;
		ObjectTypeLocal objectTypeLocal = null;
		try {
			businessRuleID = PrimaryKeyFactory.getInstance().getPrimaryKey();

			String ruleIdForAcl = buildRuleIdForAcl(objectTypeID, conditions);

			objectTypeLocal = CAFPermissionUtil.getObjectTypeLocal();
			String objectType = objectTypeLocal.findByPrimaryKey(objectTypeID).getObjectType();			

			BusinessRuleVo businessRuleVo =
				new BusinessRuleVo(businessRuleID, ruleIdForAcl, objectTypeID, objectType, ruleName);

			getJDODataAccessService().create(businessRuleVo);
			CAFPermission.createOwnerPermission(getCurrentUserID(), businessRuleID);
			
			//assigned owner for a new acl must be principal with IUSER type.
			//workaround: create acl with current user from CAFAdmin role and remove it from the acl.       
			CAFPermission.removeOwnerPermission(getCurrentUserID(), businessRuleID) ;
				
			//create fullcontrol permission for current user
			CAFPermission.createPermission(getCurrentUserID(), businessRuleID, CAFPermissionName.fullcontrol, getCurrentUserID()) ;				

		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			try {
				CAFPermissionCache.invalidParamCache();
				CAFPermissionCache.invalidRuleCache();
				CAFPermissionCache.invalidPermissionCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
				throw new ServiceException("AUTH_RESET_CACHE");
			}
			release(objectTypeLocal);
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return businessRuleID;
	}

	
	public void update(String businessRuleID, String businessRuleName) throws ServiceException {
		checkUpdatePermission();
		
		final String method = "update(String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);
		
		BusinessRuleVo businessRuleVo = null;

		try {
			businessRuleVo = findByPrimaryKey(businessRuleID);
			businessRuleVo.setBusinessRuleName(businessRuleName);
			getJDODataAccessService().store(businessRuleVo);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	/**
	 * This API deletes the business rule (and associated conditions)
	 * based on the specified businessRuleID. <p>  
	 * @param ruleId  id of the rule selected by the user 
	 * @return void
	 * @exception ServiceException 
	 * @transactionAttribute Required 
	 */
	public void delete(String businessRuleID) throws ServiceException {
		checkUpdatePermission();
		
		final String method = "delete(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		BusinessRuleVo businessRuleVo = null;
		ConditionVo conditionVo = null;
		ConditionLocal conditionLocal = null;
		try {
			businessRuleVo = findByPrimaryKey(businessRuleID);
			
			conditionLocal = CAFPermissionUtil.getConditionLocal();
			Collection conditions = conditionLocal.findByBusinessRule(businessRuleVo.getKey());
			for (Iterator it = conditions.iterator(); it.hasNext();) {
				conditionVo = (ConditionVo)it.next();
				conditionLocal.delete(conditionVo.getKey());
			}
			
			getJDODataAccessService().remove(businessRuleVo);
			CAFPermission.removePermission(getCurrentUserID(), businessRuleID, null, (IPrincipal) null);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			try {
				CAFPermissionCache.invalidParamCache();
				CAFPermissionCache.invalidRuleCache();
				CAFPermissionCache.invalidPermissionCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
				throw new ServiceException("AUTH_RESET_CACHE");
			}
			release(conditionLocal);
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	public BusinessRuleVo findByPrimaryKey(String businessRuleID) throws ServiceException {
		BusinessRuleVo businessRuleVo = null;

		final String method = "findByPrimaryKey(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			String param = "String key";
			String filter = "this.key == key";
			HashMap hm = new HashMap();
			hm.put("key", businessRuleID);
			//transaction already handled using CMT
			Collection resultSet = 
				getJDODataAccessService().query(BusinessRuleVo.class, filter, param, hm, null);
			businessRuleVo = (BusinessRuleVo)resultSet.iterator().next();
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return businessRuleVo;
	}
	

	/*
	 * builds the id that is relevant for Acl
	 */
	private String buildRuleIdForAcl(String objectTypeID, HashMap conditions) throws ServiceException {
		final String method = "buildRuleIdForAcl(String, HashMap)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		String ruleIdForAcl = null;
		StringBuffer buf = new StringBuffer();
		Object obj = null;
		String attributeName = null;
		Object[] attributes = null;


		//get Attribute Set
		try {
			ObjectAttributeLocal objectAttributeLocal = CAFPermissionUtil.getObjectAttributeLocal();
			attributes = objectAttributeLocal.findByObjectTypeID(objectTypeID).toArray();
		} catch(Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		}

		//now retrieve conditions from hashMap. sequence of conditions in hahsMap may not correspond to 
		//the sequence in which they were entered by the user. Attributes are stores in a certain order in the 
		//attributes set. the user is forced to enter conditions in the same order. we therefore use the sequence 
		//in attriute set to extract the conditions from the hashMap
		for (int i = 0; i < attributes.length; i++) {

			if (attributes[i] != null) {
				if (attributes[i] instanceof ObjectAttributeVo) {
					attributeName = ((ObjectAttributeVo) attributes[i]).getObjectAttributeName();
					obj = conditions.get(attributeName);
				}
			}

			if ((obj != null) && (i > 0)) {
				buf.append(":");
			}

			if (obj != null) {
				buf.append(obj);
			}
		}
		
		if (buf.length() > 0) {
			ruleIdForAcl = buf.toString();
		}
			

		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		return ruleIdForAcl;
	}

	/*
	 * retrieve all rules based on a attribute set Id
	 */
	public Collection findByObjectTypeID(String objectTypeID) throws ServiceException {
		Collection resultSet = new ArrayList(1);

		final String method = "findByObjectTypeID(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		//now get the attribute set for this object type
		String param = "String objectTypeID";
		String filter = "this.objectTypeID == objectTypeID";
		HashMap hm = new HashMap();
		hm.put("objectTypeID", objectTypeID);

		try {
			//transaction already handled using CMT
			resultSet = getJDODataAccessService().query(BusinessRuleVo.class, filter, param, hm, null);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}

		return resultSet;
	}

	public Collection findByObjectType(String objectType) throws ServiceException {
		Collection resultSet = new ArrayList(1);

		final String method = "findByObjectType(String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		//now get the attribute set for this object type
		String param = "String objectType";
		String filter = "this.objectType == objectType";
		HashMap hm = new HashMap();
		hm.put("objectType", objectType);

		try {
			//transaction already handled using CMT
			resultSet = getJDODataAccessService().query(BusinessRuleVo.class, filter, param, hm, null);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}

		return resultSet;
	}

	public Collection findByObjectID(String objectID, String objectTypeID) throws ServiceException {
		Collection result = new ArrayList();

		final String method = "findByObjectID(String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IDependentObject businessObject = CAFPermissionUtil.getDependentObject(objectID,objectTypeID);

			if (businessObject!=null) {
				Collection rules = retrieveBusinessRulesByObject(businessObject, objectTypeID);
				for (Iterator iter = rules.iterator(); iter.hasNext();) 
				{
					BusinessRuleVo businessRuleVo = (BusinessRuleVo)iter.next();
					businessRuleVo.setObjectType(objectTypeID);

					result.add(businessRuleVo);
				}
			}
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}

		return result;
	}

	private Collection retrieveBusinessRulesByObject(IDependentObject object, String objectTypeID) throws ServiceException {
		Collection resultSet = null;

		final String method = "retrieveBusinessRulesByObject(IDependentObject, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		if (object == null)	{
			ServiceException serviceException =new ServiceException("AUTH_NO_RULE");
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, serviceException.getMessage(), serviceException);
			throw serviceException;
		}

		try {
			CAFBusinessRuleList ruleList = 
				(CAFBusinessRuleList)CAFPermissionCache.getBusinessRuleFromCache(objectTypeID);	
			
			if (ruleList == null) {
				return Collections.EMPTY_LIST;
			}
							
			final int rulesCount = ruleList.size();
			Object[] attrCache = ruleList.getEmptyAttributeCache();
			Collection ids = new HashSet(rulesCount);
			
			for (int i=0; i< rulesCount; i++) {
				if (ruleList.objectFulfillBusinessRuleCondition(object,i,attrCache)) {
					ids.add(ruleList.getBusinessRule(i).getRuleID());
				} 
			}
			resultSet = retrieveRules(ids); 
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return resultSet;
	}

	private Collection retrieveRules(Collection ids) throws ServiceException {
		final String method = "retrieveRules(Collection)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		Collection resultSet = null;
		
		try {
			if (ids==null || ids.isEmpty()) {
				return Collections.EMPTY_LIST;
			}
			
			StringBuffer params = new StringBuffer(); //"String attributeSetId";
			StringBuffer filter = new StringBuffer(); 
			HashMap hm = new HashMap();
			int index = 0;
			for (Iterator iter = ids.iterator(); iter.hasNext();) {
				
				String id = (String) iter.next();
				String parameter = "ruleID"+index;
				if (index>0) {
					params.append(",");
					filter.append(" || ");
				}
				params.append("String ").append(parameter);				
				filter.append("this.key==").append(parameter);
				hm.put(parameter,id);
				index++;
			}
			//transaction already handled using CMT
			resultSet = getJDODataAccessService().query(BusinessRuleVo.class, filter.toString(), params.toString(), hm, null);
		} catch (Exception e) {
			getSessionContext().setRollbackOnly();
			CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, e.getMessage(), e);
			throw new ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
		return resultSet;
	}

}
