/*
 * Created on Aug 13, 2003
 *
 * To change the template for this generated file go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
package com.sap.caf.rt.security.acl.impl;

/**
 * @author i006085
 *
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import com.sap.caf.rt.bol.IDependentObject;
import com.sap.caf.rt.bol.context.CAFProperties;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.caf.rt.exception.CAFPermissionException;
import com.sap.caf.rt.security.srv.ServicePermission;
import com.sap.caf.rt.security.util.CAFBusinessRuleList;
import com.sap.caf.rt.security.util.CAFPermissionCache;
import com.sap.caf.rt.security.util.CAFPermissionName;
import com.sap.security.api.IPrincipal;
import com.sap.security.api.IPrincipalFactory;
import com.sap.security.api.IRole;
import com.sap.security.api.IRoleFactory;
import com.sap.security.api.IUser;
import com.sap.security.api.IUserFactory;
import com.sap.security.api.NoSuchRoleException;
import com.sap.security.api.UMException;
import com.sap.security.api.UMFactory;
import com.sap.security.api.acl.IAcl;
import com.sap.security.api.acl.IAclEntry;
import com.sap.security.api.acl.IAclManager;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * Programming Guide for ACL Permission Check and Service Permission Check. ACL Permission Check allows 
 * performing permission check based on individual business object instance. Service permission check 
 * allows checking whether a user is allowed to access the service method that is provided by service object.
 * <p>
 * ACL Permission Check allows checking whether the login user has been granted the authorization to access 
 * the object instance. Two levels of permission check are applied. The first one applies to the instance. 
 * During the creation of any instance, The owner of the instance is always been assigned. It is either the 
 * login user or the designate user. It is application specific so it is up to the application to decide 
 * which user to be the instance owner. Besides the owner permission been granted to the instance, the 
 * owner or CAF Admin can also grant other permission to other users. Here are the lists of the permissions 
 * used by CAF; "read", "create", "update", "delete", "fullcontrol", "owner".
 * Whoever has the "read" permission can retrieve the instance. Whoever has the "create" permission can create 
 * a new instance. Whoever has the "update" permission can retrieve and modify the instance. Whoever has 
 * the "delete" permission can retrieve the instance as well as delete the instance.
 * Whoever is "owner" or "fullcontrol" permission can retrieve, create, modify and delete the instance. 
 * The "owner" and "fullcontrol" has the same permission authorization. In CAF, a new role "CAFAdmin" 
 * is created as the administrator for CAF. Any login user who has the "CAFAdmin" role is allowed to 
 * create, retrieve, delete or modify the instance.
 * <p>
 * The second permission check is based on business rule. While CAF provides rule maintenance screen, 
 * here are some notes to high light.
 * The operators for condition of the rule as follows:<br>
 * "EQ" - equal to the low attribute value<br>
 * "LT" - less than the low attribute value<br>
 * "LE" - less equal to the low attribute value<br>
 * "GT" - greater than the low attribute value<br>
 * "GE" - greater equal to the low attribute value<br>
 * "BT" - greater equal to the low attribute value and less equal to the high attribute value<br>
 * <P>
 * The supported data types of the attribute in condition are defined as follows:<br>
 * "int" - the attribute of the object is "int" data type<br>
 * "String" - the attribute of the object is "String" data type<br>
 * "Date" - the attribute of the object is "Date" data type. The "Date" class is from java.util package. While it parses to the "String"
 * 			from "Date" object, FormatDate.getInstance().format(Date) has to be used.<br>
 * "Collection" - the attribute of the object is a "Collection" data type. A "Collection" data type in CAF means that the instance has the
 * 			cardinality 1:n. In other word, it is a java util.Set class.  However, we only support one level of Collection despite the possibility
 * 			of n-level Collection. In other word, the objects store in the "Collection" must be "int", "String, or "Date" data type.<br>
 * <P>
 * All business rules and attribute list are cache during the run time to improve the efficiency. The cache scope is per VM and its life span of the
 * cache is per server. Hence, it is important to reload the cache after any business rule or attribute list are modified during the run time. 
 * To reload the cache, one must call the API CAFPermission.resetCache().<br>
 * <P>
 * If users wish to have their own customerized permission check, CAF provides an exit to fulfill the need. The customerized implementation class 
 * must implement an Interface class "ICAFCheckPermission" under package "com.sap.caf.rt.security.acl.api". CAF provides an
 * implementation template "CAFAuthorityExitTemplate" under package "com.sap.caf.rt.security.acl.impl" for reference.
 * Besides the method "checkAclPermission" in the customerization, there is another one "isStandardAclCheckRequired" that the customer must implement. 
 * If the method "iStandardAclCheckRequired" returns true, the CAF standard permission check is performed regardless the result of the customerization
 * "checkAclPermission". However, the CAF standard permission check will be performed if the customerization "checkAclPermission" returns false.
 * During the configuration time, the configurator must define where is the customerization implementation class. Here is the processes;
 * In the SAP Admin Console, locate the configuration properties.<br>
 * 1) Cluster--> server 0 --> Services --> Configuration Adaptor --> (in the right panel) apps --> sap.com --> caf~sap.com.caf.runtime --> 
 * appcfg --> propertysheet application.global.properties<br>
 * 2) Fill the customerized class name and its path in the "Value" part of the property "CAF_AUTHORITY_EXIT".<br>
 * 3) One can refer to the CAF template class as "com.sap.caf.rt.security.acl.impl.CAFAuthorityExitTemplate".<br>
 * 4) Assemble and deploy the application server to take effect.<br>
 * <p>
 * Service permission check requires quite a bit of configuration during the installation. Please refer to "CAF Permission User Configuration Guide" 
 * for details. In the design time, the designer assigns a permission to service method during the service object creation. There are six permissions.
 * "read", "write", "modify", "delete", "owner" and "fullcontrol". "read" is a member of "modify" and "delete". "write" is a member of "modify".
 * "modify" and "delete" is a member of "fullcontrol" and "owner". There are five UME roles that are created during the configuration. They are<br>
 * "CAFServiceReadRole"<br>
 * "CAFServiceWriteRole"<br>
 * "CAFServiceDeleteRole"<br>
 * "CAFServiceModifyRole"<br>
 * "CAFServiceFullcontrolRole"<br>
 *  <p>
 * If the login user is intended to access a service method that is assign to a particual permission, 
 * the login user must have the proper UME role in order to access the service method. Please refer to 
 * the name of the CAF role that is corresponding to the permission level.
 * 
 * For example,the role "CAFServiceWriteRole" is assigned to "write" permission that is 
 * allowed to "read" and "write" permission.
 */
public class CAFPermission {  
	protected static final String ADMIN_ROLE = "CAFAdmin";
	
	private static final String APPLICATION_ID = "CAF";
	private static final String ACL_CUSTOMIZED_CLASS = "CAF_AUTHORITY_EXIT";
	private static final String CHECK_PERMISSION_METHOD = "checkAclPermission";
	private static final String STANDARD_CHECK_REQUIRED_METHOD = "isStandardAclCheckRequired";

	private static final String APPLICATION = CAFPermission.class.getName();
	private static final Location LOGGER = Location.getLocation(APPLICATION);
	private static final String JARMREQPREFIX = "CAF:RT:security:";
	private static final String JARMREQUEST = JARMREQPREFIX+APPLICATION;

	private static IAclManager s_aclManager;
	private static IUserFactory s_userFactory;
	private static IPrincipalFactory s_principalFactory;
	private static String s_AclCustomizedClass = "-1";
	private static boolean s_StandardCheckPermissionRequired = false;
	private static Class s_CutomizedPermissionClass = null;
	private static IRole s_adminRole;

	static {
		try {
			createDefaultPermissionHierarchy();
		} catch (Exception e) {
			String method = "Static block";
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,"AUTH_PERMISSION_ERROR",new Object[]{method,e.getMessage()},e);
		};
	}
	
	/**
	 * <p>
	 * Check permission for the userId. Check instance permission based on the user id and permission name
	 * first. If the user id has the permission to access the instance, simply returns true, otherwise,
	 * check permission based on business rules. 
	 * @param object instance of the target object.
	 * @param userID user Id of the IUser that is subjected for permission check
	 * @param permissionName is the permission such as "read" or 'write' etc that the user is attempted to 
	 * the object
	 * @param objectType identifies the object class
	 * @return boolean 'true'indicatess the permission is granted, 'false' indicates the permission is denied  
	 * @exception PermissionException
	 * <p>
	 */
	
	public static boolean checkAclPermission(Object object, String principalID, String permissionName, 
		String objectType) throws CAFPermissionException {
			final String method = "checkAclPermission(Object, String, String, String)" ;
			CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

			IPrincipal principal = null;
			try{	
				principal = getPrincipal(principalID) ;
			}catch(UMException e){
				CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,"AUTH_PERMISSION_ERROR",new Object[]{method,e.getMessage()},e);
				throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[]{method,e.getMessage()});

			}
			return checkAclPermission(object, principal, permissionName, objectType, null, false) ;
	}
	
	/**
	 * <p>
	 * Check permission for the userId. Check instance permission based on the principal and permission name
	 * first. If the principal  has the permission to access the instance, simply returns true, otherwise,
	 * check permission based on business rules. 
	 * @param object instance of the target object.
	 * @param principal principal that is subjected for permission check
	 * @param permissionName is the permission such as "read" or 'write' etc that the user is attempted to 
	 * the object
	 * @param objectType identifies the object class
	 * @param boRuleID business rule id - check permission based on this rule only  
	 * @param checkAclPermissions - 'true' alows to check instance permission only, 'false' alows to check 
	 * permission for both - instance and rule based   
	 * @return boolean 'true'indicatess the permission is granted, 'false' indicates the permission is denied  
	 * @exception PermissionException
	 * <p>
	 */
	public static boolean checkAclPermission(Object object, IPrincipal principal, String permissionName, 
		String objectType, String boRuleID, boolean checkAclPermissions) throws CAFPermissionException {
		
		final String method = "checkAclPermission(Object, String, String,  String, String, boolean)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 0);

		try {        
			String principalID = principal.getUniqueID() ;

			// check whether a customerized permission check implementation defined
			
			if (s_AclCustomizedClass != null && s_AclCustomizedClass.equals("-1")) {
				s_CutomizedPermissionClass = getAclCustomerizedClass();
			}
			

			boolean bCustomerizedPermission = false;
			try {
				
				String principalType = UMFactory.getPrincipalFactory().getPrincipalType(principalID) ;

				if ((s_CutomizedPermissionClass != null) && (principalType.equals(IPrincipalFactory.IUSER)))
				{
					bCustomerizedPermission = 
								checkCustomerizedAclPermission(
									s_CutomizedPermissionClass, 
									object, 
									principalID, 
									permissionName, 
									objectType
								);
				}
			} catch (UMException e) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			}

			// if the permission check from customerized implementation class is true and
			// standardCheckPermission is not required, them simply returns true 
			if (bCustomerizedPermission && !s_StandardCheckPermissionRequired) {
				return true;
			}
		
			// check instance permission based on ACL
			if (boRuleID == null){
				if (object!=null) {
					String key = ((IDependentObject) object).getKey(); 
					if (hasAclPermission(principal, (key != null) ? key : "", permissionName)) {
						return true;
					}
				}
			}
			// check permission based on business rules
			// build parameter name, type and value array list	
			// Check each business rule for the object type
			if (!checkAclPermissions){
				
				CAFBusinessRuleList ruleList = (CAFBusinessRuleList)CAFPermissionCache.getBusinessRuleFromCache(objectType);
				if (ruleList == null || ruleList.isEmpty()) {
					return false;	
				}
				
				Object[] attrCache = ruleList.getEmptyAttributeCache();
				for (int i=0, rulesCount = ruleList.size(); i< rulesCount; i++) {
					
					boolean isObjectFulfillBusinessRuleCondition = false;
					
					if (CAFPermissionName.create.equals(permissionName) && object==null){
						isObjectFulfillBusinessRuleCondition = true;
					} else {
						isObjectFulfillBusinessRuleCondition = ruleList.objectFulfillBusinessRuleCondition(object,i,attrCache); 
					}
					  
					if (isObjectFulfillBusinessRuleCondition) {
						
						String ruleID = ruleList.getBusinessRule(i).getRuleID();

						//check permissions for boRuleID only
						if ((boRuleID != null)&&(!boRuleID.equals(ruleID))){
							continue ;
						}
					
						String accessCacheKey = ruleID + principalID + permissionName;
						Boolean cachedResult = CAFPermissionCache.getAllowed( accessCacheKey);
						if (cachedResult == null) {
							// If no cached value - evaluate one
							boolean result = hasAclPermission(principal, ruleID, permissionName);
							// cache result in any case 
							CAFPermissionCache.cacheAllowedAccess(accessCacheKey, result);
							// we may need to check next rule
							if (result) {
								return true;
							}
						} else {
							// we may need to check next rule
							if (cachedResult.booleanValue()) {
								return true;
							}
						}
					}
				}
			}
		} catch (ClassCastException ce) {
			try {
				CAFPermission.resetCache();
			} catch (Exception pex) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR, LOGGER, method, pex.getMessage(), pex);
			}
		} catch (CAFPermissionException e) {
			throw e;
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			Object[] param = { method, e.getMessage()};
			CAFPermissionException pe = new CAFPermissionException("AUTH_PERMISSION_ERROR", param);
			throw pe;
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 0);
		}
		return false;
	}
	
	/**
	 * <p>
	 * Check service method access permission for the  login user. If the login user has the role that defines the permission then
	 * it simply returns true, otherwise, false is returned.
	 * @param usr is the login user string that is subjected for permission check. It is passes from the generated code when the
	 * service method is created. Here is one of the example how it is gneerated. i.e. "new ServicePermission("read")"
	 * @param ServicePermission is the ServicePermission class that defines permission name and its value.
	 * @return boolean 'true' indicates the permission is granted, 'false' indicates the permission is denied  
	 * @exception PermissionException
	 * <p>
	 */
	public static boolean checkServicePermission(String usr, ServicePermission sp) throws CAFPermissionException {
		final String method = "checkServicePermission(String, ServicePermission)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser iusr = CAFPermission.getUser(usr);
			return iusr.hasPermission(sp);
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[]{method,e.getMessage()});
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1); 
		}
	}

	/** 
	 * <p>
	 * Invalid cache so the parameter set and business rule condition can be reloaded. This API is intended to call when
	 * business rule, condition or parameter set has been modified during the run time.
	 * @param null - apply the entire cache for CAF
	 * @return void
	 * @exception PermissionException
	 * <p>
	 */
	public static void resetCache() throws CAFPermissionException {
		final String method = "resetCache()";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		CAFPermissionCache.invalidCache();

		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}
	
	/** 
	 * <p>
	 * Add owner permission for user id. The ACL must not exist for the object id. The user id will be assigned as the owner.
	 * A CAFAdmin role also be assigned as the co-owner of the object.
	 * @param userID is assigned with the owner permission of the instance. The permission for the owner is always "owner" permission
	 * @param objectID is an unique id to identify the object instance
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void createOwnerPermission(String userId, String objectId) throws CAFPermissionException {
		final String method = "createOwnerPermission(String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser iusr = getUser(userId);
			IAclManager aclManager = getAclManagerInstance(); 
			try {
				if (aclManager.getAcl(objectId) != null){
					throw new CAFPermissionException("Owner has existed for object Id = " + objectId, new String [] {}); 			
				}
			} catch (UMException e) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			}

			IAcl iAcl = aclManager.createAcl(iusr, objectId);
			IRoleFactory rf = UMFactory.getRoleFactory();
			// create CAFAdmin role as co-owner
			try {
				IRole ir = rf.getRoleByUniqueName(ADMIN_ROLE);
				iAcl.addOwner(iusr, ir);
			} catch (NoSuchRoleException e) { 
				CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
				throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[]{ADMIN_ROLE});
			}
			
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[]{ method,e.getMessage()});
		}

		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}
	
	/** 
	 * <p>
	 * Remove owner permission for user id
	 * @param userID is assigned with the owner permission of the instance. The permission for the owner is always "owner" permission
	 * @param objectID is an unique id to identify the object instance
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void removeOwnerPermission(String userID, String objectID) throws CAFPermissionException {
		final String method = "removeOwnerPermission(String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser iusr = getUser(userID);
			IAcl acl = getACL(objectID, true);
			if (checkAuthorizePermission(iusr, acl)){
				acl.removeOwner(iusr, iusr);
			}
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[]{method, e.getMessage()});
		}
		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}

	/** 
	 * <p>
	 * Add a permission to a principal 
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param permissionName is the permission grants to the principal
	 * @param iPrincipal is the principal object. It can be a user, group or a role,
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void createPermission(String loginUserID, String objectID, String permissionName, IPrincipal iPrincipal) throws CAFPermissionException {
		final String method = "createPermission(String, String, String, IPrincipal)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser iusr = getUser(loginUserID);
			IAcl acl = getACL(objectID, true);
			if (checkAuthorizePermission(iusr, acl)) {
				acl.createAclEntry(iusr, iPrincipal, permissionName, false);
			}
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[]{ method, e.getMessage()});
		}
		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}
	
	/** 
	 * <p>
	 * Add a permission to a user 
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param permissionName is the permission grants to the principal
	 * @param assignedUserID is the user id that is assigned to the permission.
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void createPermission(String loginUserID, String objectID, String permissionName, String assignedUserID) throws CAFPermissionException {
		final String method = "createPermission(String, String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser assignedUsr = getUser(assignedUserID);
			createPermission(loginUserID, objectID, permissionName, assignedUsr);
		} catch (UMException e) {
			Object[] param = { method, e.getMessage()};
			CAFPermissionException pe = new CAFPermissionException("AUTH_PERMISSION_ERROR", param);
			throw pe;
		}
		finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}
	
	/** 
	 * <p>
	 * Add permissions to principals. A principal can be assigned with one or more permissions. For example, user_xx can have both "read"
	 * and "delete" permission. This API allows to assign multiple permissions to multiple principals.
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param principalList is an arrayList that store HashMap objects. The HashMap is a collection of
	 * principals and permission name pair. The key of the HashMap is a principal and the value of the HashMap is a permission name.
	 * For example, if the user "user_xx" and "user_yy" have been assigned with permission name such as "read",
	 * then the HashMap will have two entries as user_xx/read and user_yy/read pairs. Or two HashMap objects are created. One for
	 * user_xx/read and the other is for user_yy/read. 
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void createPermissionAll(String loginUserID, String objectID,List principalList) throws CAFPermissionException {
		final String method = "createPermissionAll(String, String, List)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		final int size = principalList.size();
		for (int i=0; i<size; i++) {
			HashMap hm = (HashMap) principalList.get(i);
			Iterator iter = hm.keySet().iterator();
			while (iter.hasNext()) {
				IPrincipal ipr = (IPrincipal) iter.next();
				String permissionName = (String) hm.get(ipr);
				createPermission(loginUserID, objectID, permissionName, ipr);
			}
		}
		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}
	
	/** 
	 * <p>
	 * Modify permission for a principal 
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param oldPermissionName is the permission granted to the principal. This permission will be modified after this API.
	 * @param permissionName is the new permission that is intended to grant to the principal
	 * @param iPrincipal is the principal object. It can be a user, group or a role,
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void modifyPermission(String loginUserID, String objectID, String oldPermissionName, String newPermissionName, IPrincipal iPrincipal) throws CAFPermissionException {
		final String method = "modifyPermission(String, String, String, String, IPrincipal)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser iusr = getUser(loginUserID);
			IAcl acl = getACL(objectID, true);
			List ar1 = acl.getAclEntries(iPrincipal);
			modifyACEntries (acl, iusr, iPrincipal, ar1, oldPermissionName, newPermissionName);
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			Object[] param = { method, e.getMessage()};
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR", param);
		}
		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}
	
	/** 
	 * <p>
	 * Modify permission for a user 
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param oldPermissionName is the permission granted to the principal. This permission will be modified after this API.
	 * @param permissionName is the new permission that is intended to grant to the principal
	 * @param assignedUserID is the user id that is assigned to the permission.
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void modifyPermission(String loginUserID, String objectID, String oldPermissionName, String newPermissionName, String assignedUserID) throws CAFPermissionException {
		final String method = "modifyPermission(String, String, String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser assignedUser = getUser(assignedUserID);
			modifyPermission(loginUserID, objectID, oldPermissionName, newPermissionName, assignedUser);
		} catch (UMException e) {
			Object[] param = { method, e.getMessage()};
			CAFPermissionException pe = new CAFPermissionException("AUTH_PERMISSION_ERROR", param);
			throw pe;
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}
	
	/** 
	 * <p>
	 * Remove permissions from principals. If permissionName and prinicpal object are provided, only permissions with the
	 * same permissionName and principal are removed. If permissionName is space or null, 
	 * all permissions for the specific principal are removed. If the principal is a a null object,
	 * all permissions for all principals with the permissionName are removed. If the permissionName and principal are both null object,
	 * then all permissions (regardless permission name or principal) are removed. 
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param permissionName is the permission intended to be removed. It has to be a legal permission or space or null. 
	 * @param iPrincipal is the principal object. It can be a user, group or a role. It also can be a null object.
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void removePermission(String loginUserID, String objectID, String permissionName, IPrincipal iPrincipal) throws CAFPermissionException {
		final String method = "removePermission(String, String, String, IPrincipal)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser iusr = getUser(loginUserID);
			IAcl iAcl = getACL(objectID, true);
			List ar1;
			if (iPrincipal == null){
				ar1 =iAcl.getAclEntries();
			} else {
				ar1 = iAcl.getAclEntries(iPrincipal);
			}
			removeACEntries (iAcl, iusr, ar1, permissionName);
//			if (iPrincipal == null && (permissionName == null || permissionName.length() == 0)) {
//				getAclManagerInstance().removeAcl(_iusr, iAcl);
//			}
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR", new Object[] {method, e.getMessage()} );
		}
		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}
	
	/** 
	 * <p>
	 * Remove permissions from a user. If permissionName and assigned user id are provided, only permissions with the
	 * same permissionName and assigned user id are removed. If permissionName is space or null, 
	 * all permissions for the specific assigned user id are removed. If the assigned user id is a a null object or empty string,
	 * all permissions for all principals with the permissionName are removed. If the permissionName and assigned user id are both null object,
	 * then all permissions (regardless permission name or principal) are removed. 
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param permissionName is the permission intended to be removed. It has to be a legal permission or space or null. 
	 * @param assignedUserID is the user id that is assigned to the permission.. It also can be a null object or empty string.
	 * @return void
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static void removePermission(String loginUserID, String objectID, String permissionName, String assignedUserID) throws CAFPermissionException {
		final String method = "removePermission(String, String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser assignedUser = null;
			if (assignedUserID != null && assignedUserID.length() != 0) {
				assignedUser = getUser(assignedUserID);
			}
				
			removePermission(loginUserID, objectID, permissionName, assignedUser);
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[] { method, e.getMessage()});
		}
		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}
	
	/** 
	 * <p>
	 * Retrieve permissions of principals. If permissionName and prinicpal object are provided, only permissions with the
	 * same permissionName and principal are retrieved. If permissionName is space or null, 
	 * all permissions for the specific principal are retrieved. If the principal is a a null object,
	 * all permissions for all principals with the permissionName are retrieved. If the permissionName and principal are both null object,
	 * then all permissions (regardless permission name or principal) are retrieved. 
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param permissionName is the permission intended to be removed. It has to be a legal permission or space or null. 
	 * @param iPrincipal is the principal object. It can be a user, group or a role. It also can be a null object.
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static ArrayList retrievePermission(String loginUserID, String objectID, String permissionName, IPrincipal iPrincipal) throws CAFPermissionException {
		final String method = "retrievePermission(String, String, String , IPrincipal)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser iusr = getUser(loginUserID);
			List ar1 = new ArrayList();
			IAcl acl = getACL(objectID, true);
			if (hasAuthorizePermission(iusr,acl))  {
				if (iPrincipal == null){
					ar1 = acl.getAclEntries();
				}				
				else {
					ar1 = acl.getAclEntries(iPrincipal);
				}
					
			}
			return retrieveACEntries (ar1, permissionName);
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);			
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[]{method, e.getMessage()});
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}
	
	/** 
	 * <p>
	 * Retrieve permissions of assignedUserID. If permissionName and assignedUserID string are provided, only permissions with the
	 * same permissionName and assignedUserID are retrieved. If permissionName is space or null, 
	 * all permissions for the specific assignedUserID are retrieved. If the assignedUserID is a a null object or zero length,
	 * all permissions for all principal with the permissionName are retrieved. If the permissionName and assignedUserID are both null object,
	 * then all permissions (regardless permission name or assignedUserID) are retrieved. 
	 * @param loginUserID is the login user. The permission for the login user must be the owner of the instance or CAFAdmin role
	 * @param objectID is an unique id to identify the object instance
	 * @param permissionName is the permission intended to be removed. It has to be a legal permission or space or null. 
	 * @param assignedUserID is the user id string. It can be a null object or zero string.
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static ArrayList retrievePermission(String loginUserID, String objectID, String permissionName, String assignedUserID) throws CAFPermissionException {
		final String method = "retrievePermission(String, String, String, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try {
			IUser assignedUser = null;
			if (assignedUserID != null && assignedUserID.length() != 0) {
				assignedUser = getUser(assignedUserID);
			}
				
			return retrievePermission(loginUserID, objectID, permissionName, assignedUser);
		} catch (UMException e) {
			Object[] param = { method, e.getMessage()};
			CAFPermissionException pe = new CAFPermissionException("AUTH_PERMISSION_ERROR", param);
			throw pe;
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}
	
	/** 
	 * <p>
	 * Create Default Permission Hierarchy only call once for the lifetime
	 * The permission hierarchy is top node of (owner and fullcontrol) and the mid node of (readwrite and delete) 
	 * and the bottom node of (read and write). If a user would like to read a data and modify it back, the user should
	 * have (readwrite). The onwer and fullcontrol has the same priviledge.
	 * @exception CAFPermissionException
	 * <p>
	 */
	public static synchronized void createDefaultPermissionHierarchy() throws CAFPermissionException {
		final String method = "createDefaultPermissionHierarchy()";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try { 
			IAclManager aclManager = getAclManagerInstance();
			
			List permissions = aclManager.getAllPermissions();
			if (!permissions.contains(CAFPermissionName.read)) {
				aclManager.addPermission(CAFPermissionName.fullcontrol, null);
				aclManager.addPermission(CAFPermissionName.owner, null);
				aclManager.addPermission(CAFPermissionName.create, null);
				aclManager.addPermission(CAFPermissionName.update, null);
				aclManager.addPermission(CAFPermissionName.delete, null);
				//aclManager.addPermission(CAFPermissionName.authorize,null);
				aclManager.addPermission(CAFPermissionName.read, null);				
				aclManager.addPermission(CAFPermissionName.write, null);
				aclManager.addPermission(CAFPermissionName.remove, null);

				aclManager.addPermissionMember(CAFPermissionName.owner, CAFPermissionName.create);
				aclManager.addPermissionMember(CAFPermissionName.owner, CAFPermissionName.update);
				aclManager.addPermissionMember(CAFPermissionName.owner, CAFPermissionName.delete);				
				//aclManager.addPermissionMember(CAFPermissionName.owner, CAFPermissionName.authorize);

				aclManager.addPermissionMember(CAFPermissionName.update, CAFPermissionName.read);
				aclManager.addPermissionMember(CAFPermissionName.update, CAFPermissionName.write);
				
				aclManager.addPermissionMember(CAFPermissionName.delete, CAFPermissionName.read);
				aclManager.addPermissionMember(CAFPermissionName.delete, CAFPermissionName.remove);
			}
		} catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR",new Object[] {method,e.getMessage()});
		}	
		CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	}
	
	private static void removeACEntries(IAcl iAcl, IPrincipal loggedInUser,List ar, String permissionName) throws UMException {
		final int size = ar.size();
		final boolean removeAll = permissionName == null || permissionName.length() == 0;
		for (int i=0; i<size; i++) { 
			IAclEntry entry = (IAclEntry) ar.get(i);
			if (removeAll || entry.getPermission().equals(permissionName)) {
				iAcl.removeAclEntry(loggedInUser, entry);			
			}
		}
	} 

	private static ArrayList retrieveACEntries(List ar, String permissionName) throws CAFPermissionException {
		ArrayList resultList = new ArrayList();
		final int size = ar.size();
		for (int i=0; i<size; i++) {
			IAclEntry ace = (IAclEntry) ar.get(i);
			if (permissionName == null ||
				permissionName.length() == 0 ||
				ace.getPermission().equals(permissionName))	{
					HashMap hm = new HashMap();
					hm.put(ace.getPrincipal(), ace.getPermission());		
				resultList.add(hm);	
			}
		}
		return resultList;
	}
	
	private static void modifyACEntries(IAcl iAcl, IPrincipal ipr, IPrincipal iPrincipal, List ar, String oldPermissionName, String newPermissionName) throws UMException {
	   final int size = ar.size();
	   for	(int i=0; i<size; i++) {
		   IAclEntry ace = (IAclEntry) ar.get(i);
		   if (ace.getPermission().equals(oldPermissionName)) {			
			   iAcl.removeAclEntry(ipr, ace);	
			   iAcl.createAclEntry(ipr, iPrincipal, newPermissionName, false);
		   }
	   }
	}
	private static synchronized IAclManager getAclManagerInstance() {
		if (s_aclManager == null) {
			s_aclManager = UMFactory.getAclManager(APPLICATION_ID); 	
		}
		return s_aclManager;
	}
	
	private static synchronized IUser getUser (String userID) throws UMException {
		String method = "getUser (String)";
		IUser usr = null;

		if (s_userFactory == null) {
			s_userFactory = UMFactory.getUserFactory(); 
		}
		
		if (!userID.startsWith("USER.")){
			try{
				usr = s_userFactory.getUserByUniqueName(userID);
				if (usr != null){
					return usr;
				}
			}catch(UMException e){
				CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			}
		}

		try {
			usr = s_userFactory.getUser(userID);	
		} catch (UMException e) {
			usr = s_userFactory.getUserByUniqueName(userID);
		}
		return usr;
	}

	private static synchronized IPrincipal getPrincipal (String principalId) throws UMException {
		String method = "getPrincipal (String)";
		
		IPrincipal prcl = null;
		if (s_userFactory == null) {
			s_userFactory = UMFactory.getUserFactory(); 
		}

		if (s_principalFactory == null) {
			s_principalFactory = UMFactory.getPrincipalFactory(); 
		}
		if (!principalId.startsWith("USER.")){
			try {
				prcl = s_userFactory.getUserByUniqueName(principalId);
				if (prcl != null){
					return prcl;
				}
			} catch (UMException e) {
				CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
			}
		}

		try {
			prcl = s_principalFactory.getPrincipal(principalId);
		} catch (UMException _e) {
   			prcl = s_userFactory.getUserByUniqueName(principalId);	
		}
		
		return prcl;
	}

	private static synchronized Class getAclCustomerizedClass () {
		Class lClass = null;
		if (s_AclCustomizedClass.equals("-1")) { 
			CAFProperties cPro = new CAFProperties();
			try {
				cPro.loadApplicationProperties();
				s_AclCustomizedClass = (String) cPro.get(ACL_CUSTOMIZED_CLASS);
			} catch (Exception e) {
				s_AclCustomizedClass = null;
			}
		}
//		cAclCustomerizedClass = "com.sap.caf.rt.security.acl.impl.MyClass";
		if (s_AclCustomizedClass == null || s_AclCustomizedClass.length() == 0) {
			return null;
		}
			
		try {
			lClass = Class.forName(s_AclCustomizedClass);
			if (lClass != null) {
				s_StandardCheckPermissionRequired = isStandardCheckPermissionRequired(lClass);
			}
		} catch (ClassNotFoundException _e) {
			lClass = null;
		} catch (CAFPermissionException _e) {
			lClass = null;
		}
		return lClass;
	}

	private static boolean hasAclPermission(IPrincipal principal, String objectID, String permissionName) throws CAFPermissionException {
		boolean isAllowed;
		try {
			//IPrincipal _iusr = getPrincipal(principalID);
			IAclManager aclManager = getAclManagerInstance(); 
			IAcl iAcl = aclManager.getAcl(objectID);
			if (iAcl == null) {
				return false;
			}
			isAllowed =  iAcl.isAllowed(principal, permissionName);
		} catch (Exception e) {
			throw new CAFPermissionException("AUTH_ACL_ACCESS", new Object[] {objectID, principal, permissionName},e);
		}
		return isAllowed;
	}
	
	private static boolean checkCustomerizedAclPermission(Class cClass, Object obj, String userID, String permissionName, String objectType) throws CAFPermissionException {
		boolean bPermission;
		try {
			Class[] parameterDataType = {Object.class, userID.getClass(), permissionName.getClass(), objectType.getClass() };
			Object[] parameterValueObject = {obj, userID, permissionName, objectType};
			Method method = cClass.getMethod(CHECK_PERMISSION_METHOD, parameterDataType);
			bPermission = ((Boolean)method.invoke(cClass.newInstance(), parameterValueObject)).booleanValue();
		} catch (Exception _e) {
			throw new CAFPermissionException(_e);
		}
		return bPermission;
	}
	
	private static boolean isStandardCheckPermissionRequired(Class cClass) throws CAFPermissionException {
		boolean bRequired;
		try {
			Object[] nullObject = {};
			Method method = cClass.getMethod(STANDARD_CHECK_REQUIRED_METHOD, new Class[0]);
			bRequired = ((Boolean)method.invoke(cClass.newInstance(), nullObject)).booleanValue();
		} catch (Exception _e) {
			throw new CAFPermissionException(_e);
		}
		return bRequired;
	}

	public static IAcl getACL(String objectID) throws UMException, CAFPermissionException {
		IAclManager aclManager = getAclManagerInstance(); 
		IAcl acl = aclManager.getAcl(objectID);
		return acl;
	}	
	
	private static IAcl getACL(String objectID, boolean mustExist) throws UMException, CAFPermissionException {
		IAclManager aclManager = getAclManagerInstance(); 
		IAcl acl = aclManager.getAcl(objectID);
		if (acl==null && mustExist) {
			throw new CAFPermissionException("AUTH_NO_ACL",new Object[] { objectID });
		}
		return acl;
	}	

	public static Collection getAllowedPermission(String principalID,Object object,String objectType) throws CAFPermissionException {
		final String method = "getAllowedPermission(String, Object, String)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		try { 
			Iterator iter = CAFPermissionName.getAllNames();
			Collection result  = new ArrayList();

			while (iter.hasNext()) 
			{
				CAFPermissionName permission = (CAFPermissionName)iter.next();
				String name = permission.getName();
				
				if (checkAclPermission(object,principalID,name,objectType)) 
				{
					result.add(name);					
				}
			}
			return result;
		} catch (Exception e) {
			throw new CAFPermissionException(e);
		} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
		}
	}

	private static boolean checkAuthorizePermission(IUser user,IAcl acl) throws CAFPermissionException {
			if (!hasAuthorizePermission(user,acl)) {
					throw new CAFPermissionException("AUTH_NO_ACL_MODIFY_ACCESS",new Object[]{user.getDisplayName()});
			} else {
					return true;
			}
	}

	private static boolean hasAuthorizePermission(IUser user,IAcl acl) throws CAFPermissionException {
		final String method = "hasAuthorizePermission(IUser, IAcl)";
		CAFPublicLogger.entering(null, JARMREQUEST, method, LOGGER, 1);

		if (user==null || acl==null) { 
				return false;
		} 

		try {
			if (acl.isAllowed(user,CAFPermissionName.owner)) {                              
					return true;
			} else {
					if (s_adminRole==null) {
							try {
									s_adminRole = UMFactory.getRoleFactory().getRoleByUniqueName(ADMIN_ROLE);       
							} catch (UMException e) {
									CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(),e);
									throw new CAFPermissionException("AUTH_NO_ROLE", new Object[]{ADMIN_ROLE});
							}                                       
					}
					return s_adminRole.isUserMember(user.getUniqueID(),true);
			}
	    } catch (UMException e) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR,LOGGER,method,e.getMessage(), e);
			Object[] param = { method, e.getMessage()};
			throw new CAFPermissionException("AUTH_PERMISSION_ERROR", param);
	  	} finally {
			CAFPublicLogger.exiting(null, JARMREQUEST, method, LOGGER, 1);
	  	}
	}

}