/*
 * Created on Jun 24, 2003
 */
package com.sap.caf.rt.ui.cool.utils;

import java.util.Hashtable;

import javax.naming.Context;

import com.sap.caf.rt.ui.cool.generic.ServiceFacade;
import com.sap.tc.col.cds.ConnectionData;
import com.sap.tc.col.cds.ConnectionDataCommon;
import com.sap.tc.col.client.generic.api.IAspect;
import com.sap.tc.col.client.generic.api.IKey;
import com.sap.tc.col.client.generic.api.IMessage;
import com.sap.tc.col.client.generic.api.IMessageList;
import com.sap.tc.col.client.generic.api.IQuery;
import com.sap.tc.col.client.generic.api.IServiceFacade;
import com.sap.tc.col.client.generic.api.IServiceModule;
import com.sap.tc.col.client.generic.api.ServiceFacadeFactory;
import com.sap.tc.col.client.generic.core.GCPRuntimeException;
import com.sap.tc.col.client.metadata.api.IAspectDescriptor;
import com.sap.tc.col.client.metadata.api.IFieldDescriptor;
import com.sap.tc.col.client.metadata.api.IServiceModuleDescriptor;
import com.sap.tc.col.client.metadata.api.IStructureDescriptor;
import com.sap.tc.webdynpro.progmodel.api.IWDMessageManager;
import com.sap.tc.webdynpro.progmodel.api.IWDNode;
import com.sap.tc.webdynpro.progmodel.api.IWDNodeElement;
import com.sap.tc.webdynpro.progmodel.api.IWDViewController;
import com.sap.tc.webdynpro.services.sal.localization.api.WDResourceHandler;
import com.sap.tc.webdynpro.services.session.IScope;
import com.sap.tc.webdynpro.services.session.Utils;

/**
 * WebDynpro application Cool Services utilities</p>
 *
 * This class contains some useful utility-methods you can use  
 * coding for WebDynpro environment using <i>Cool</i> Services.</p>
 * 
 * These methods may help to do many things, allowing to reduce 
 * your time efforts coding various <i>Cool</i>-aware stuff.
 * 
 * @author Victor Kulaga
 * @version 1.0
 */
public class CoolUtils {	
	
	public static final String SERVICE_FACADE_KEY = "SERVICE_FACADE";
	
	/**
	 * Returns {@link com.sap.tc.col.client.generic.api.IServiceFacade} to use <i>Cool</i> 
	 * services with. Connection parameters from specified deployable object's properties file are used.
	 * @deprecated This method will be removed in next version of library. Use {@link getServiceFacade(CoolConnectionProperties cp)} instead.
	 * @return {@link com.sap.tc.col.client.generic.api.IServiceFacade}
	 * @throws {@link CoolUtilsException} if any error occurs.  
	 * */
	public static IServiceFacade getServiceFacade(String deployableObjectName) throws CoolUtilsException {		
	  try 
	  {
		CoolConnectionProperties cp = 
			CoolConnectionProperties.getCoolConnectionProperties(deployableObjectName);
		return getServiceFacade(cp);
	  } catch (CoolConnectionPropertiesException e) {
		throw new CoolUtilsException("Failed to read Cool connection parameters: \n"+e.getMessage());
	  } catch (GCPRuntimeException e) {
	  	throw new CoolUtilsException("Failed to connect to Cool server: \n"+e.getMessage());
	  }
	}
	
	public static IServiceFacade getTestServiceFacade(String url, String user, String password) throws CoolUtilsException {
		IServiceFacade sf = (IServiceFacade)Utils.getScope(IScope.APPLICATION_SCOPE).get(SERVICE_FACADE_KEY);
		if (sf==null){
			sf = new ServiceFacade(url, user, password);
			Utils.getScope(IScope.APPLICATION_SCOPE).put(SERVICE_FACADE_KEY, sf);
		} 
		return sf;
	}	
	
	/**
	 * Returns {@link com.sap.tc.col.client.generic.api.IServiceFacade} to use <i>Cool</i> 
	 * services with.
	 * @param cp {@link CoolConnectionProperties} object used to connect to <i>Cool</i>
	 * @return {@link com.sap.tc.col.client.generic.api.IServiceFacade}
	 * @throws {@link CoolUtilsException} if any error occurs.  
	 * */
	public static IServiceFacade getServiceFacade(CoolConnectionProperties cp) throws CoolUtilsException {		
	  try 
	  {
	  	IServiceFacade sf = (IServiceFacade)Utils.getScope(IScope.APPLICATION_SCOPE).get(SERVICE_FACADE_KEY);
	  	if (sf==null){
			if (cp.getCoolHost().equalsIgnoreCase("caf")) {
				Hashtable ht = initHashtable(cp);				
				sf = new ServiceFacade(ht, WDResourceHandler.getCurrentSessionLocale()); 
			} else {
				ConnectionData cd = new ConnectionDataCommon();
				cd.setProperty(ConnectionData.propertyNameClient, cp.coolClient);
				cd.setProperty(ConnectionData.propertyNameUser, cp.coolUserName);
				cd.setProperty(ConnectionData.propertyNamePassword, cp.coolUserPassword);
				cd.setProperty(ConnectionData.propertyNameLanguage, cp.coolLocaleStr);
				cd.setProperty(ConnectionData.propertyNameSystemNumber, cp.coolSysNumber);
				cd.setProperty(ConnectionData.propertyNameSystemName, cp.coolHost);
				sf = ServiceFacadeFactory.createServiceFacade(cd);
			}
			Utils.getScope(IScope.APPLICATION_SCOPE).put(SERVICE_FACADE_KEY, sf);			
	  	}	  	
		return sf;
	  } catch (GCPRuntimeException e) {
		throw new CoolUtilsException("Failed to connect to Cool server: \n"+e.getMessage());
	  }	  
	}
	
	public static Hashtable initHashtable(CoolConnectionProperties cp) {
		Hashtable ht = new Hashtable();
		ht.put(Context.PROVIDER_URL, cp.coolURL);
		ht.put(Context.SECURITY_PRINCIPAL, cp.coolUserName);
		ht.put(Context.SECURITY_CREDENTIALS, cp.coolUserPassword);
		ht.put(Context.INITIAL_CONTEXT_FACTORY, "com.sap.engine.services.jndi.InitialContextFactoryImpl");
		return ht;			
	}
	
	public static IServiceModule getServiceModule(IServiceFacade serviceFacade, String serviceModuleName) throws CoolUtilsException {
		
		if (serviceFacade == null) {
			return null; 
		}
		//	obtaining Service Module
		serviceModuleName = (serviceModuleName!=null ? serviceModuleName : "");
		try {
			IServiceModule serviceModule = serviceFacade.getServiceModule(serviceModuleName);
			serviceModule.associatedModelInfo();	// check the Service Module
			return serviceModule;
			
		} catch (Exception e) {
			throw new CoolUtilsException("Invalid or wrong backend Service Module specified: \n" + e.getMessage());
		}
	}	
	/** 
	 * This method connects to SAP back-end via <i>Cool</i> services and reads the list of 
	 * <i>Aspect</i> fields using the connection parameters read from the project's 
	 * properties file and <i>Service Module</i> and <i>Aspect</i> input field values 
	 * specified in the form.<br> 
	 * The appropriate context elements of the node <i>HeaderAttributes</i> are created on runtime 
	 * when data is obtained from SAP back-end.</p>  
	 * declared method 
	 * */
	public static IAspectDescriptor  readAspectMetaData(
			IServiceModule serviceModule, String aspectName, 
			String[] fieldAttrs, IWDNode nodeTo, String typePrefix) 
			throws CoolUtilsException {
		
		if (serviceModule == null) {
			return null; 
		}
		//	obtaining Service Module meta data
		IServiceModuleDescriptor serviceModuleDesc = serviceModule.getDescriptor();
		// obtaining Aspect meta data
		aspectName = (aspectName!=null ? aspectName : "");
		IAspectDescriptor aspectDesc = serviceModuleDesc.getAspectDescriptor(aspectName);
		if (aspectDesc == null) {
			throw new CoolUtilsException("Invalid or wrong backend Aspect specified: \"" +aspectName+"\"");
		}
		IStructureDescriptor strDesc = aspectDesc.getStructure();	
		int structSize = strDesc.size();	// get the quantity of Aspect fields
		
		// delete all elements from WDNode
		for (int i = 0; i < nodeTo.size(); i++) {
			nodeTo.removeElement(nodeTo.getElementAt(i));
		}
	
		for (int i = 0; i < structSize; i++) {
			IFieldDescriptor  fieldDesc = strDesc.getFieldDescriptor(i);
			// add element to context node on runtime using Aspect field from SAP back-end
			IWDNodeElement  newEl = nodeTo.createElement();
			try {
				newEl.setAttributeValue(fieldAttrs[0], fieldDesc.getName());	// set name
			} catch (Exception e) {
			}
			try {
				newEl.setAttributeValue(fieldAttrs[1], fieldDesc.getText());	// set description
			} catch (Exception e) {
			}
			try {
				// temporary workaround as data in backend doesn't contains proper types
				String type = typePrefix + fieldDesc.getType();
				
				newEl.setAttributeValue(fieldAttrs[2], type);							// set type
			} catch (Exception e) {
			}
			nodeTo.addElement(newEl);
		}
		return aspectDesc;
	}
	
	/**
	  * Remove aspect row and corresponding current element of the context node
	  * @param aspect the Aspect to delete row from
	  * @param node the context node that corresponds to specified aspect
	  */
	public static void removeCurrentAspectRow(IAspect aspect, IWDNode node)
			throws CoolUtilsException {
		
		try {
			IWDNodeElement el = node.getCurrentElement();
			aspect.remove(node.getLeadSelection());
			aspect.sendChanges();
			node.removeElement(el);
		} catch (Exception e) {
			throw new CoolUtilsException("Could not remove the aspect row and corresponding current element of the context node : \n" + e.getMessage());
		}
	}

	/* this method converts key to string */
	public static String getKeyAsString(IKey key) throws CoolUtilsException {
		try {
			String strKey = key.toString();
			return strKey.substring(1, strKey.length() - 1);
		} catch (Exception ex) {
			throw new CoolUtilsException("Could not convert Key to string : \n" + ex.getMessage());
		}
	}


	public static boolean reportCoolMessages(IServiceFacade sf, IWDViewController viewController, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(sf.getAllMessages(), viewController.getComponent().getMessageManager(), stopOnFirstFail, cancelNavigation, false);
	}

	public static boolean reportCoolMessages(IAspect aspect, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(aspect.getMessages(), manager, stopOnFirstFail, cancelNavigation, false);
	}

	public static boolean reportCoolMessages(IQuery query, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(getQueryResultAspect(query).getMessages(), manager, stopOnFirstFail, cancelNavigation, false);
	}

	private static IAspect getQueryResultAspect(IQuery query) {
		IAspect aspect = query.getResultAspect();
		if (aspect == null) {
			throw new RuntimeException("Cannot get list of messages from query, because result aspect is null. Query: " + query);
		}
		return aspect;
	}

	public static boolean reportCoolMessages(IServiceFacade sf, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(sf.getAllMessages(), manager, stopOnFirstFail, cancelNavigation, false);
	}
	
	public static boolean reportCoolMessages(IMessageList list, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(list, manager, stopOnFirstFail, cancelNavigation, false);
	}

	public static boolean raiseCoolMessages(IServiceFacade sf, IWDViewController viewController, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(sf.getAllMessages(), viewController.getComponent().getMessageManager(), stopOnFirstFail, cancelNavigation, true);
	}

	public static boolean reportCoolMessages(IAspect aspect, IWDViewController viewController, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(aspect.getMessages(), viewController.getComponent().getMessageManager(), stopOnFirstFail, cancelNavigation, false);
	}

	public static boolean reportCoolMessages(IQuery query, IWDViewController viewController, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(getQueryResultAspect(query).getMessages(), viewController.getComponent().getMessageManager(), stopOnFirstFail, cancelNavigation, false);
	}

	public static boolean raiseCoolMessages(IAspect aspect, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(aspect.getMessages(), manager, stopOnFirstFail, cancelNavigation, true);
	}

	public static boolean raiseCoolMessages(IQuery query, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(getQueryResultAspect(query).getMessages(), manager, stopOnFirstFail, cancelNavigation, true);
	}

	public static boolean raiseCoolMessages(IServiceFacade sf, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(sf.getAllMessages(), manager, stopOnFirstFail, cancelNavigation, true);
	}
	
	public static boolean raiseCoolMessages(IMessageList list, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation) {
		return CoolUtils.reportOrRaiseCoolMessages(list, manager, stopOnFirstFail, cancelNavigation, true);
	}

	/**
	 * Method <code>reportOrRaiseCoolMessages</code> is a hepler method that should be used in UI components for reporting errors, warnings or just useful information to the user.
	 * It takes list of the messages generated by cool calls during processing such actions as execute() of IAction or IQuery, sendChanges() of IAspect or IServiceModule.
	 * @param list list of messages generated by cool during processing a call. The list is cleared after execution of the method.
	 * @param manager UI message manager to report or raise messages in.
	 * @param stopOnFirstFail if true, processing of messages is stoped just after first message with <code>Failed<code> flag set is encountered in the list. The rest of the messages in the list will not be processed. This flag does not have sence if <code>raise</code> flag is true. 
	 * @param cancelNavigation see IWDMessageManager for more details on the meaning of this flag.
	 * @param raise if the value is <code>true</code>, first error met in the list causes an WDNonFatalException to raise as described in IWDMessageManager methods <code>raise...</code> 
	 * @return status of the latest call. If equals <code>true</code> then the call is failed and the list contains fatal errors. See {@link #isFatal(IMessageList)} for details.
	 */
	public static boolean reportOrRaiseCoolMessages(IMessageList list, IWDMessageManager manager, boolean stopOnFirstFail, boolean cancelNavigation, boolean raise) {
		boolean result = false;
		int size = list.size();
		try {
			for (int i = 0; i < size; i++) {
				IMessage message = list.getMessage(i);
				switch (message.getType()) { 
					case IMessage.ERROR:
						if (raise) {
							manager.raiseException(message.getText(), cancelNavigation);
						} else {
							manager.reportException(message.getText(), cancelNavigation);
						}
						break;
					case IMessage.WARNING:
						manager.reportWarning(message.getText());
						break;
					case IMessage.INFO:
						manager.reportSuccess(message.getText());
						break;
					default:
						if (raise) {
							manager.raiseException("Unknown type of message has been recieved from <cool>. Message type: " + message.getType()
									+ ", message text: " + message.getText(), true);
						} else {
							manager.reportException("Unknown type of message has been recieved from <cool>. Message type: " + message.getType()
									+ ", message text: " + message.getText(), true);
						}
				}
				if (message.isFailed()) {
					result = true;
					if (stopOnFirstFail) {
						break;
					}
				}
			} 
		} finally {
			list.clear();
		}
		return result;
	}
	
	public static boolean isFatal(IAspect aspect) {
		return isFatal(aspect.getMessages());
	}

	public static boolean isFatal(IQuery query) {
		return isFatal(getQueryResultAspect(query).getMessages());
	}

	public static boolean isFatal(IServiceFacade sf) {
		return isFatal(sf.getAllMessages());
	}

	/**
	 * The method isFatal returns <code>true</code>, if the list contains one or more fatal errors.
	 * The list is not cleared during the method execution. To clear the messages list you should call list.clear() method or execute one of report/raise methods.
	 * @param list list of mesages generated by cool at the latest call(s).
	 * @return <code>true</code>, if one or more fatal errors are present in the list.
	 */
	public static boolean isFatal(IMessageList list) {
		int size = list.size();
		for (int i = 0; i < size; i++) {
			IMessage msg = list.getMessage(i);
			if (msg.isFailed()) {
				return true;
			}
		}
		return false;
	}
	
	/** 
	 * This method reads the SAP back-end data (via <i>Cool</i> services) responding to selected 
	 * Aspect fields retrieved by method {@link readAspectFields()} and displayed in the 
	 * <i>Header Configurator</i> form. User selects the necessary <i>Aspect</i> fields,
	 * clicks the appropriate button and then this method is called to create and 
	 * initialize the context attributes of node <i>Results</i>(used by <i>Header UI</i> 
	 * Layout) on runtime.</p>
	 * declared method */
/*	public static void readAspectData(IServiceFacade serviceFacade) throws CoolUtilsException {
	  if (serviceFacade == null) {
		return; 
	  }
	  IPublicCustomHeaderConfigurator.IContextElement contextEle = 
		  wdContext.currentContextElement() ;  
	 
	  IServiceModule serviceModule = serviceFacade.getServiceModule(contextEle.getServiceModule());
	  ICMIModelInfo modelInfo = serviceModule.associatedModelInfo();
    
	  // create a new key list for a specified aspect:
		
	  String paramKeyList = ResourceUtils.getProperty(
		  PROPNAME_SERVICEMODULE_KEYLIST, 
		  DEPLOYABLE_OBJECT_NAME_PRJ,
		  wdThis.wdGetAPI().getComponent().getMessageManager());		
	  IKeyList keyList = serviceModule.createKeyList(paramKeyList);
	
	  // add the known keys to the key list:    
	  String parameter1 = wdThis.wdGetHeaderConfiguratorInterfaceController().wdGetContext().currentContextElement().getParameter1();
	  String [] strKey = (parameter1!=null ? parseKey(parameter1) : new String[3]);	    
	  IKey key1 = serviceModule.createKey(paramKeyList, strKey); 
	  keyList.add(key1);
	
	  // now select the data for the keys:
	  IAspect aspectDetails = serviceModule.getAspect(contextEle.getAspect(), keyList);
	  ICMIModelClassInfo aspectinfo = modelInfo.getModelClassInfo("Aspect." + contextEle.getAspect());
	  IPublicCustomHeaderConfigurator.IResultsNode aspectNode = wdContext.nodeResults();
	  aspectNode.getNodeInfo().setModelClassInfo(aspectinfo);
	  aspectNode.getNodeInfo().addAttributesFromModelClassInfo();
	  if(aspectDetails.size()>0)
		  aspectNode.addElement(aspectNode.createElement(aspectDetails.getAspectRow(0)));
	  else 
	  {
		  IPublicCustomHeaderConfigurator.IResultsElement el = aspectNode.createResultsElement(aspectDetails.createAspectRow());
		  aspectNode.addElement(el);	    
	  }    
	}*/	
}
