package com.sap.caf.rt.ui.cool.generic;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.sap.caf.rt.srv.IDataContainerBean;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.cmi.metadata.ICMIModelInfo;
import com.sap.tc.cmi.model.ICMIModelClass;
import com.sap.tc.col.client.generic.api.IAction;
import com.sap.tc.col.client.generic.api.IAspect;
import com.sap.tc.col.client.generic.api.IKey;
import com.sap.tc.col.client.generic.api.IKeyList;
import com.sap.tc.col.client.generic.api.IQuery;
import com.sap.tc.col.client.generic.api.IServiceFacade;
import com.sap.tc.col.client.generic.api.IServiceModule;
import com.sap.tc.col.client.generic.api.IStructure;
import com.sap.tc.col.client.generic.api.LockStrategy;
import com.sap.tc.col.client.metadata.api.IAspectActionDescriptor;
import com.sap.tc.col.client.metadata.api.IAspectDescriptor;
import com.sap.tc.col.client.metadata.api.IKeyAspectDescriptor;
import com.sap.tc.col.client.metadata.api.IQueryDescriptor;
import com.sap.tc.col.client.metadata.api.IServiceModuleDescriptor;
import com.sap.tc.col.client.metadata.api.IStructureDescriptor;
import com.sap.tc.col.edo.IEdoStructure;
import com.sap.tc.col.servicemanager.api.ISrvMgrServiceManager;
import com.sap.tc.logging.Location;

/**
 * realizes a class for a <code>ServiceModule</code>.<p>
 * 
 * The <cool> backend architecture requires structuring of application logic in 
 * smaller units, the <code>ServiceModule</code>s.<p>
 * Examples for such service modules are business partners, employees, sales order,..<p>
 * The main tasks of <code>ServiceModule</code> are:
 * <ul>
 * <li> serves as a factory for <code>Aspect</code>, <code>Query</code>, <code>Structure</code>
 *      and <code>Action</code> instances
 * <li> holds association to the different<code>AspectCaches</code>
 * <li> manages call objects used as API to ServiceManager
 * </ul>
 * 
 * Additionally <code>ServiceModule</code> implements <code>ICMIGenericModel</code>.<p>
 * This is necessary to use it as Model from Web Dynpro. <p>
 *
 * Subclasses can extend or override the following methods to modify the factory:<p>
 * <ul>
 * <li>{@link #createAspectInternal(AspectDescriptor, ISrvMgrCall)}
 * <li>{@link #createKeysAspectInternal(KeyDescriptor, IKeyList)}
 * <li>{@link #createQueryInternal(QueryDescriptor)}
 * <li>{@link #createActionInternal(AspectActionDescriptor, Aspect)}
 * <li>{@link #createStructureInternal(StructureDescriptor, ISrvMgrStructure)}
 * </ul>
 * 
 * @author Helmut Mueller
 */
public class ServiceModule implements IServiceModule {

	/** Logging properites for this class */
	private static final String APPLICATION	= ServiceModule.class.getName();
	private static final String jARMRequest = AbstractModelClass.jARMReqPrefix+APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);

  /**
   * ServiceFacade, this service module belongs to.
   */
  private final ServiceFacade serviceFacade;
  private CAFServiceManager serviceManager;
  private IServiceModuleDescriptor serviceModuleDescriptor;
  private AspectServiceAccess proxy;

  private String shortName;
  
  /**
   * constructor
   * @param srvMgrServiceModule ServiceModule instance of the ServiceManager layer
   */
	protected ServiceModule(IServiceFacade serviceFacade,
							CAFServiceManager serviceManager,
							AspectServiceAccess proxy) {
	  if ((proxy) == null || (proxy.getServiceModuleDescriptor() == null)) {
	  	throw new NullPointerException("Service Module Description cannot be null.");
	  }
	  this.serviceFacade = (ServiceFacade) serviceFacade;
	  this.serviceModuleDescriptor = proxy.getServiceModuleDescriptor();
	  this.serviceManager = serviceManager;
	  this.proxy = proxy;
	  String name = serviceModuleDescriptor.getName();
	  if (name.indexOf(".")!=-1)
		  shortName = name.substring(name.lastIndexOf(".")+1);
	  else
		  shortName = name;
	}
	
	/**
	 * constructor
	 * @param srvMgrServiceModule ServiceModule instance of the ServiceManager layer
	 */
	  protected ServiceModule(IServiceFacade serviceFacade,
							  CAFServiceManager serviceManager,
							  IServiceModuleDescriptor desc) {
		if (serviceModuleDescriptor == null) {
		  throw new NullPointerException("Service Module Description cannot be null.");
		}
		this.serviceFacade = (ServiceFacade) serviceFacade;
		this.serviceModuleDescriptor = desc;
		this.serviceManager = serviceManager;
		this.proxy = serviceManager.getAspectServiceAccess(desc.getName());
		String name = serviceModuleDescriptor.getName();
		if (name.indexOf(".")!=-1)
			shortName = name.substring(name.lastIndexOf(".")+1);
		else
			shortName = name;
	  }
	

  public String getShortName() { return this.shortName;}
  
  /**
   * Return  instance of AspectServiceAccess
   */
  public AspectServiceAccess getAspectServiceAccess(){
	return proxy;
  }

  public ISrvMgrServiceManager getServiceManager() { return serviceManager;}  
  /**
   * Returns the global message list. Currently this is a ServiceFacade list
   */  
  MessageList getAllMessagesInternal() {
    return serviceFacade.getAllMessagesInternal();
  }
  
  protected ServiceFacade getServiceFacade() { return serviceFacade;}
  /**
   * Creates a new <code>Query</code> by getting meta data from <code>ServiceModuleDescriptor</code><p>
   * the method throws an <code>IllegalArgumentException</code>, if queryName is no valid name for a Query.<p>
   * @param queryName the name of the <code>Query</code>
   * @return IQuery the new created <code>Query</code>
   * @exception <code>IllegalArgumentException</code>, if queryName is no valid name for a Query.
   */
  public IQuery createQuery(String queryName) {
		final String method = jARMRequest + ":createQuery(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    logger.debugT("queryName={0}", new Object[] {queryName});
    try {
      IQueryDescriptor queryDescriptor = getDescriptor().getQueryDescriptor(queryName);
      if ( queryDescriptor == null ) {
        logger.fatalT("Query with name {0} doesn't exist!", new Object[] {queryName});
        throw new IllegalArgumentException("Query with name "+queryName+ " doesn't exist!");
      }

      // create query via factory method      
      // @todo avoid cast
      Query newQuery = (Query) createQueryInternal(queryDescriptor);

      // create aspect cache for this Query and bind it to Query
      AspectCache aspectCache = new AspectCache( (ICacheRootObject)newQuery );
      cacheRootObjects.add(newQuery);
      newQuery.bindAspectCache(aspectCache);
      return newQuery;
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
 	}
  
  /**
   * Creates an <code>Aspect</code> by getting meta data from <code>ServiceModuleDescriptor</code>.
   * If an aspectCache is provied, the newly created aspect is added to that cache. 
   * If a srvMgrCall is given, the created aspect is initialized with the result of that call,
   * otherwise the aspect is empty.
   * @param aspectCache the <code>AspectCache</code> to add the created aspect to or <code>null</code>
   * @param aspectName the name of the <code>Aspect</code> to create
   * @param srvMgrCall the service manager call that provides the data for the new aspect or <code>null</code>
   * @return IAspect the new created <code>Aspect</code>
   */
  protected Aspect createAspectInCache(AspectCache cache,
  									 String aspectName) {
		final String method = jARMRequest + ":createAspectInCache(AspectCache, String)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
      IAspectDescriptor aspectDescriptor = getDescriptor().getAspectDescriptor(aspectName);
      if ( aspectDescriptor == null ) {
        // try to find it in KeyDescriptors
        aspectDescriptor = getDescriptor().getKeyAspectDescriptor( aspectName );
        if ( aspectDescriptor == null ) {
          logger.fatalT("Aspect with name {0} doesn't exist!", new Object[] {aspectName});
          throw new IllegalArgumentException("Aspect with name "+aspectName+ " doesn't exist!");
        }
      }

      // create aspect via factory method      
      // @todo avoid cast
      Aspect aspect = (Aspect) createAspectInternal(aspectDescriptor);
      // registerAspect(aspect);
      
      // add created Aspect to cache
      if ( cache != null ) {
        cache.add(aspect);
      }
      return aspect;
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * Creates an <code>Aspect</code> by getting meta data from <code>ServiceModuleDescriptor</code>.<p>
   * 
   * This new created <code>Aspect</code> is root object of a new <code>Aspect</code> hierarchy, which
   * is cached in GCP (GenericClientProxy) and can be released by an explicit <code>invalidate()</code> or
   * implicitly by GCP, when changes on Aspects in cache are executed in backend.<p>
   * 
   * The method throws an <code>IllegalArgumentException</code>, if aspectName is no valid name for a Aspect.<p>
   * 
   * @param aspectName the name of the <code>Aspect</code> to create
   * @return IAspect the new created <code>Aspect</code>
   * @exception <code>IllegalArgumentException</code>, if aspectName is no valid name for an Aspect.
   */
  public IAspect createAspect(String aspectName) {
		final String method = jARMRequest + ":createAspect(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		logger.debugT("aspectName={0}", new Object[] {aspectName});
    try {
      Aspect aspect = createAspectInCache(null, aspectName);
      createCacheForAspect(aspect);
      registerAspect( aspect );
      return aspect;
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * returns the meta data of this <code>ServiceModule</code>
   * @return IServiceModuleDescriptor the meta data of <code>ServiceModule</code>
   */
  public IServiceModuleDescriptor getDescriptor() {
    return serviceModuleDescriptor;
  }
  
  /**
   * sends all pending changes of all <code>Aspects</code> to service layer.<p>
   * There are no special services for insert, update, delete, cause of performance reasons, we bundle and
   * manage all local changes until this method is called. Then all changes are sent to service layer in one 
   * step.
   * Example:
   * <pre>
   *   aspect.removeAspectRow( myRow );
   *   myUpdatedRow = aspect1.getAspectRow( myKey );
   *   myUpdatedRow.setAttributeValue("NAME1", "BLABLA");
   *   ...
   *   serviceModule.sendChanges();
   *   ..
   * </pre>
   */
  public void sendChanges() {
		final String method = jARMRequest + ":sendChanges()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
      ArrayList objs = new ArrayList(cacheRootObjects);
      Iterator it = objs.iterator();
      while ( it.hasNext() ) {
        AspectCache aspectCache = ((ICacheRootObject) it.next()).getAspectCache();
        Iterator aspectIt = aspectCache.iterator();
        while ( aspectIt.hasNext() ) {
          Aspect aspect = (Aspect) aspectIt.next();
          aspect.sendChanges();
        }
      }
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * flushes the service queue, which is associated with the <code<ServiceManager</code> instance of this
   * <code>ServiceModule</code>.
   * The events resulting from this call, are caught by the <code>ServiceFacade</code>, which itself propagates them
   * to their <code>ServiceModules</code>.
   */
  public void flush() {
		final String method = jARMRequest + ":flush()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {    	
      serviceManager.flush();
		}
		finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
	  }
  }
  
  /**
   * EventHandler wihch is called just before the service queue will be flushed.<p>
   * 
   * The ServiceModule propagates this event to all aspects in its caches
   * in order to give them a chance to put any pending changes into the queue.</p>
   * 
   * The order in which changes are put into the queue is solely defined by the 
   * cache order (which currently is ambiguous). If a client needs a specific ordering, 
   * it can call <code>sendChanges()</code> on the aspects accordingly.
   */
  public void onServiceQueueWillBeFlushed() {
		final String method = jARMRequest + ":onServiceQueueWillBeFlushed()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
           
      Iterator it = cacheRootObjects.iterator();
      while ( it.hasNext() ) {
        ICacheRootObject rootObject = (ICacheRootObject) it.next();
        AspectCache aspectCache = rootObject.getAspectCache();
        Iterator aspectIt = aspectCache.iterator();
        while ( aspectIt.hasNext() ) {
          Aspect aspect = (Aspect) aspectIt.next();
          aspect.onServiceQueueWillBeFlushed();
        }
      }
      
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * Event handler called after the service queue has been flushed.<p>
   * 
   * The ServiceModule propagates this event to all aspects in its caches.
   * If one of the caches is dirty after this event is processed, the complete cache is
   * invalidated by calling invalidate() on the cache root object
   */
  public void onServiceQueueFlushed() {
		final String method = jARMRequest + ":onServiceQueueFlushed()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
           
      // inform callers about processed calls      
      /**Iterator it = srvMgrCallObjects.entrySet().iterator();
      while ( it.hasNext() ) {
        Map.Entry entry = (Map.Entry) it.next();
        if ( entry.getValue() != null ) {
          List callList = (List) entry.getValue();
          ISrvMgrCall[] calls = (ISrvMgrCall[]) callList.toArray(new ISrvMgrCall[callList.size()]);
          ((ISrvMgrCaller)entry.getKey()).onCallsProcessed(calls);
        }
        else {
          ((ISrvMgrCaller)entry.getKey()).onCallsProcessed(null);
        }
        it.remove();
      }**/
      
      // assert:
      // if ( !srvMgrCallObjects.isEmpty() )
      //  throw new AssertionError("their should no registered calls be left");
    
      Iterator it = cacheRootObjects.iterator();
      while ( it.hasNext() ) {
        ICacheRootObject rootObject = (ICacheRootObject) it.next();
        if ( rootObject.getAspectCache().isDirty() ) {
          logger.infoT("onServiceQueueFlushed", "Cache of Root Object is invalidated" );
          rootObject.invalidate(); 
          if ( rootObject instanceof Aspect ) {
            it.remove();
          }
          else {
            rootObject.getAspectCache().setDirty(false);
          }
        }
      }
    }
    finally { 
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * returns the meta data of this model class instance
   * @return ICMIModelInfo the meta data of this model class instance
   */
  public ICMIModelInfo associatedModelInfo() {
    if ( modelInfo == null ) {
      modelInfo = (ColModelInfo)serviceManager.getRepositoryManager().getCMIModel(getDescriptor().getName());

//new ColModelInfo( getDescriptor() );
    }
    return modelInfo;
  }

  /**
   * @throws <code>UnsupportedOperationException</code>
   */
  public ICMIModelClass createModelObject(String modelClassName) {
    throw new UnsupportedOperationException(); 
  }

  /**
   * @throws <code>UnsupportedOperationException</code>
   */
  public ICMIModelClass createModelObject(Class modelClassClazz) {
    throw new UnsupportedOperationException(); 
  }

  /**
   * returns the model instance name, which corresponds to the service module name
   * @return String the model name, which corresponds to the service module name
   */
  public String getModelName() {
    return associatedModelInfo().getName();
  }
  
  /**
   * meta data of model
   */
  private ColModelInfo modelInfo;

  /**
   * @link aggregation
   * @associates <{ICacheRootObject}>
   * @clientCardinality 1
   * @supplierCardinality 0..*
   */
  private ArrayList cacheRootObjects = new ArrayList();
  
  /**
   * @see com.sap.tc.cmi.model.ICMIModel#getModelInstanceId()
   */
  public String getModelInstanceId() {
    return null;
  }

  /**
   * Creates an empty <code>KeyList</code> by getting meta data from <code>ServiceModuleDescriptor</code>.<p>
   * The method throws an <code>IllegalArgumentException</code>, if keyName is no valid name for a Key.<p>
   * 
   * @param keyName the name of the <code>KeyList</code> to create
   * @return IKeyList the new created <code>KeyList</code>
   * @exception <code>IllegalArgumentException</code>, if keyName is no valid name for a Key.
   */
  public IKeyList createKeyList(String keyName) {
    IKeyAspectDescriptor keyDescriptor = getDescriptor().getKeyAspectDescriptor(keyName);
    if ( keyDescriptor == null ) {
      logger.fatalT(" Key with name "+keyName+ " doesn't exist");
      throw new IllegalArgumentException(" Key with name "+keyName+ " doesn't exist");   
    }
    return new KeyList( keyDescriptor );
  }
  
  /**
   * Creates a <code>Key</code> by getting meta data from <code>ServiceModuleDescriptor</code>.<p>
   * The method throws an <code>IllegalArgumentException</code>, if keyName is no valid name for a Key.<p>
   * 
   * @param keyName the name of the <code>Key</code> to create
   * @param keyFields the key fields of <code>Key</code> to create
   * @return IKey the new created <code>Key</code>
   * @exception <code>IllegalArgumentException</code>, if keyName is no valid name for a Key or key fielda array is <code>null</code>.
   */
  public IKey createKey(String keyName, String[] keyFields ) {
    IKeyAspectDescriptor keyDescriptor = getDescriptor().getKeyAspectDescriptor(keyName);
    if ( keyDescriptor == null ) {
      logger.fatalT(" Key with name {0} doesn't exist", new Object[] {keyName});
      throw new IllegalArgumentException(" Key with name "+keyName+ " doesn't exist");   
    }
    if ( keyFields == null ) {
      logger.fatalT(" Array with key fields must not be null!");
      throw new IllegalArgumentException(" Array with key fields must not be null!");   
    }
    return new Key( keyDescriptor, keyFields );
  }

  /**
   * Creates an <code>Aspect</code> by navigating from the given <code>KeyList</code> to the targetAspect with given name.<p>
   * 
   * This new created <code>Aspect</code> is root object of a new <code>Aspect</code> hierarchy, which
   * is cached in GCP (GenericClientProxy) and can be released by an explicit <code>invalidate()</code> or
   * implicitly by GCP, when changes on Aspects in cache are executed in backend.<p>
   * 
   * The method throws an <code>IllegalArgumentException</code>, if aspectName is no valid name for a Aspect.<p>
   *
   * @param aspectName the name of the <code>Aspect</code> to create
   * @param keyList the input Key list
   * @return IAspect the new created <code>Aspect</code>
   * @exception <code>IllegalArgumentException</code>, if aspectName is no valid name for an Aspect.
   */
  public IAspect getAspect(String aspectName, IKeyList keyList) {
    // providing null as cache forces a new cache to be created for the returned aspect 
    return getAspect(null, aspectName, keyList, LockStrategy.SHARED_READ_ONLY);
  }
  
  /**
   * Creates an <code>Aspect</code> by navigating from the given <code>KeyList</code> to the targetAspect with given name.<p>
   * 
   * This new created <code>Aspect</code> is root object of a new <code>Aspect</code> hierarchy, which
   * is cached in GCP (GenericClientProxy) and can be released by an explicit <code>invalidate()</code> or
   * implicitly by GCP, when changes on Aspects in cache are executed in backend.<p>
   * 
   * The method throws an <code>IllegalArgumentException</code>, if aspectName is no valid name for a Aspect.<p>
   *
   * @param aspectName the name of the <code>Aspect</code> to create
   * @param keyList the input Key list
   * @param lock lock flag
   * @return IAspect the new created <code>Aspect</code>
   * @exception <code>IllegalArgumentException</code>, if aspectName is no valid name for an Aspect.
   */
  public IAspect getAspect(String aspectName, IKeyList keyList, LockStrategy lockMode) {
    // providing null as cache forces a new cache to be created for the returned aspect 
    return getAspect(null, aspectName, keyList, lockMode);
  }
  
  /**
   * Creates an <code>Aspect</code> by navigating from the given <code>KeyList</code> to the targetAspect with given name.<p>
   * 
   * This new created <code>Aspect</code> is registered in the given cache if given 
   * or becomes a root object of a new <code>Aspect</code> hierarchy if no cache is given (<code>cache == null</code>).
   * The cached is maintained by the GCP (GenericClientProxy) and can be released by an explicit 
   * <code>invalidate()</code> or implicitly by GCP, when changes on Aspects in cache are executed in backend.<p>
   * 
   * The method throws an <code>IllegalArgumentException</code>, if aspectName is no valid name for a Aspect.<p>
   *
   * @param sourceAspect, where the navigation starts or <code>null</code>, if a new root Aspect should be created
   * @param aspectName the name of the <code>Aspect</code> to create
   * @param keyList the input Key list
   * @param lock lock flag
   * @return IAspect the new created <code>Aspect</code>
   * @exception <code>IllegalArgumentException</code>, if aspectName is no valid name for an Aspect.
   */
  IAspect getAspect(Aspect sourceAspect, String aspectName, IKeyList keyList, LockStrategy lockMode) {
		final String method = jARMRequest + ":getAspect(Aspect, String, IKeyList, LockStrategy)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    Aspect targetAspect;
	    
	    AspectCache cache = ( sourceAspect == null ) ? null : sourceAspect.getAspectCache();
	    
	    // minmize roundtrips: KeyAspects can be created locally
	    if ( keyList.getDescriptor().getName().equals( aspectName ) ) {
		    // create aspect via factory method
		    // @todo avoid cast
		    targetAspect = (Aspect) createKeysAspectInternal(keyList.getDescriptor(), keyList);
		  	// registerAspect(targetAspect);
		
		  	if ( cache != null ) {
		   	 cache.add(targetAspect);
		  	}
	    }
	    else {
	    	//TODO figure out if this is right - They had switch for HOLLOW
	       
	    	targetAspect = (Aspect)createAspect(aspectName);
	 			
			 try {
			 	//TODO move to central location for loading aspects
				Object rawResult = proxy.readAspectObjects(targetAspect.getName(),keyList/*, lockMode*/);
				((Aspect)targetAspect).setState(Aspect.PROCESS_QUEUE_WAITING);
				 if (rawResult instanceof List) {
					Iterator it = ((List)rawResult).iterator();
					while(it.hasNext())
						createRow(targetAspect, (IDataContainerBean )it.next());		
				 }
				 else
					createRow(targetAspect, (IDataContainerBean )rawResult);
				((Aspect)targetAspect).setState(Aspect.CLEAN);
			 } catch (Exception e) {
				 MessageFactory.createAndRegisterMessageFromException(e, getServiceFacade());
			 }
	    }
	    
	    // if no cache was given, create a new one
	    if ( cache == null ) {
		createCacheForAspect(targetAspect);
	    }
			else { // now add to cache target Aspect
				cache.add(targetAspect);
			}
	    return targetAspect; 
		}
	  finally {							
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
	  }
  }
  
  
//TODO move to util class
  private void createRow(IAspect result, IDataContainerBean  bean) {
		final String method = jARMRequest + ":createRow(IAspect, IDataContainerBean)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
			AspectRow row = (AspectRow)((Aspect)result).createAspectRow();
			row.supplyRowWithData(bean);
		}
		finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }
  
  /**
   * helper method to create a cache with the needed associations
   */
  private void createCacheForAspect(Aspect targetAspect) {
    AspectCache aspectCache = new AspectCache( targetAspect );
    cacheRootObjects.add(targetAspect);
    targetAspect.setCacheRootObject(true);
    aspectCache.add( targetAspect );
    targetAspect.bindAspectCache(aspectCache);
  }
  
  
  
   

  /**
   * Creates an <code>Aspect</code> with the given descriptor and initial data.
   * @param aspectDescriptor the descriptor of the new <code>Aspect</code>
   * @param initialCall initial call that provides data for the new aspect or null.
   * @return IAspect the new created <code>Aspect</code>
   */
  protected IAspect createAspectInternal(IAspectDescriptor aspectDescriptor) {
		final String method = jARMRequest + ":createAspectInternal(IAspectDescriptor)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
      return Aspect.createEmptyInstance(this, aspectDescriptor);
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * Creates a <code>KeysAspect</code> with the given descriptor and keyList.
   * @param keyDescriptor the descriptor of the new <code>Aspect</code>
   * @param keyList a list of keys
   * @return IAspect the new created <code>Aspect</code>
   */
  protected IAspect createKeysAspectInternal(IKeyAspectDescriptor keyDescriptor, IKeyList keyList) {
		final String method = jARMRequest + ":createKeysAspectInternal(IKeyDescriptor, IKeyList)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
      return Aspect.createPopulatedInstance(this, keyDescriptor, keyList);
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * Creates a <code>Query</code> with the given descriptor and keyList.
   * @param queryDescriptor the descriptor of the new <code>Query</code>
   * @return IQuery the new created <code>Query</code>
   */
  protected IQuery createQueryInternal(IQueryDescriptor queryDescriptor) {
		final String method = jARMRequest + ":createQueryInternal(IQueryDescriptor)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
      return new Query(queryDescriptor, this);
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * Creates a <code>Action</code> for the given aspect with the given descriptor.
   * @param actionDescriptor the descriptor of the new <code>Action</code>
   * @param owner aspect that will own the new <code>Action</code>
   * @return IAction the new created <code>Action</code>
   */
  protected IAction createActionInternal(IAspectActionDescriptor actionDescriptor, Aspect owner) {
		final String method = jARMRequest + ":createActionInternal(IAspectActionDescriptor, Aspect)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
      return new Action(actionDescriptor, owner);
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * Creates a <code>Structure</code> with the given descriptor.
   * @param structureDescriptor the descriptor of the new <code>Structure</code>
   * @param srvMgrStructure service manager structure that contains the data
   * @return IAspect the new created <code>Aspect</code>
   */
  protected IStructure createStructureInternal(
  								IStructureDescriptor structureDescriptor, 
  								IEdoStructure edoStructure) {
		final String method = jARMRequest + ":createStructureInternal(IStructureDescriptor,IEdoStructure)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
      return new Structure(this, edoStructure);
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  /**
   * set backend debugging mode
   */  
  public void setDebug(boolean enabled, String guiHost)
  {
    serviceManager.setDebug(enabled, serviceModuleDescriptor.getName(), guiHost);
  }

//  private final Map aspectsByType = new HashMap();
    
  protected void registerAspect(Aspect aspect) {
//	  Set aspectsOfType = (Set) aspectsByType.get(aspect.getName());
//    if ( aspectsOfType == null ) {
//      aspectsOfType = new HashSet();
//      aspectsByType.put(aspect.getName(), aspectsOfType);
//    }
//    aspectsOfType.add(aspect);
  }

  void unregisterAspect(Aspect aspect) {
//    Set aspectsOfType = (Set) aspectsByType.get(aspect.getName());
//    if ( aspectsOfType != null ) {
//      aspectsOfType.remove(aspect);
//    }
//    unregisterAspectRowKeys(aspect, aspect.getKeyList());
  }

  void unregisterAspectRowKey(Aspect aspect, Key key) {
//    Set aspectsOfType = (Set) aspectsByType.get(aspect.getName());
//    if ( aspectsOfType != null ) {
//      Iterator it = aspectsOfType.iterator();
//      while (it.hasNext()) {
//        Aspect registeredAspect = (Aspect) it.next();
//        if ( registeredAspect != aspect 
//             && registeredAspect.getAspectRow(key) != null )
//          return;
//      } 
//      // no registered aspects contains the key, so this was the last row
//      // notify repository manager about key dispose
//    }
  }
  
//  private void unregisterAspectRowKeys(Aspect aspect, IKeyList keys) {
//    Set aspectsOfType = (Set) aspectsByType.get(aspect.getName());
//    if ( aspectsOfType != null ) {
//      Iterator it = aspectsOfType.iterator();
//      while (it.hasNext()) {
//        Aspect registeredAspect = (Aspect) it.next();
//        if ( registeredAspect != aspect ) {
//          keys.removeAll(aspect.getKeyList());
//        }
//      } 
//      // any key remaining in keys doesn't exist any longer
//    }
//  }
  
   
  
	/* (non-Javadoc)
	 * @see com.sap.tc.col.client.generic.api.IServiceModule#getAspect(java.lang.String, java.lang.String[])
	 */
	public IAspect getAspect(String arg0, String[] arg1) {
		// TODO Auto-generated method stub
		return null;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.col.client.generic.api.IServiceModule#getAspect(java.lang.String, java.lang.String[], com.sap.tc.col.client.generic.api.LockStrategy)
	 */
	public IAspect getAspect(String arg0, String[] arg1, LockStrategy arg2) {
		// TODO Auto-generated method stub
		return null;
	}

	public boolean isDirty() {
		final String method = jARMRequest + ":isDirty()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
			Iterator it = cacheRootObjects.iterator();
			while (it.hasNext()) {
				AspectCache aspectCache = ((ICacheRootObject) it.next()).getAspectCache();
				Iterator aspectIt = aspectCache.iterator();
				while (aspectIt.hasNext()) {
					Aspect aspect = (Aspect) aspectIt.next();
					if (aspect.isDirty()) {
						return true;
					}
				}
			}
			return false;
		} finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
	}


}
