package com.sap.caf.rt.ui.cool.generic;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import com.sap.tc.col.client.generic.api.IKey;
import com.sap.tc.col.client.generic.api.IKeyList;
import com.sap.tc.col.client.metadata.api.IKeyAspectDescriptor;
import com.sap.tc.logging.Location;

/**
 * represents a List of Keys.<p>
 * 
 * As the name states, a <code>KeyList</code> represents a list of <code>Key</code> objects and is a class for
 * value objects in the same sense as for <code>Key</code> objects.
 * 
 * The list is in no way dependent on backend resources (e.g. not backed by an RpeTable or ..)
 * Therefore a <code>KeyList</code> can be referenced as long as needed independently from any existing/non existing
 * connection. KeyLists have a life cycle which can outlast caches, transactions .. just like the <code>Key</code> objects have.<p>
 *  
 * A <code>KeyList</code> contains only <code>Key</code> objects of the same type ( same KeyDescriptor ). The methods, which insert Keys to
 * the KeyList check this type consistence.<p>
 * 
 * @author Helmut Mueller
 */
public class KeyList extends ArrayList implements IKeyList, Cloneable {

	/** Logging properites for this class */
	private static final String APPLICATION	= KeyList.class.getName();
	private static final String jARMRequest = AbstractModelClass.jARMReqPrefix+APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);
  
  /**
   * constructor
   */
  protected KeyList( IKeyAspectDescriptor keyDescriptor ) {
    if ( keyDescriptor == null )
    // descriptor must not be null
      throw new IllegalArgumentException("KeyDescriptor must not be null!");
    this.descriptor = keyDescriptor;
  }
  
  /**
   * the meta data of the <code>Key</code>s, this list contains.<p>
   */
  private transient IKeyAspectDescriptor descriptor;

  /**
   * returns the meta data of the <code>Key</code>s, this list contains.<p>
   * @return IKeyAspectDescriptor the meta data of the <code>Key</code>s, this list contains.<p>
   */
  public IKeyAspectDescriptor getDescriptor() {
    return descriptor;
  }

  /**
   * returns the <code>Key</code> at the specified position 
   * @param index the position of the <code>Key</code> to get
   * @return IKey the <code>Key</code> at the specified position 
   */
  public IKey getKey(int index) {
    return (IKey) this.get(index);
  }

  /**
   * Inserts the specified Key at the specified position in this <code>KeyList</code>.
   * Shifts the element currently at that position (if any) and any subsequent elements to the right (adds one to their indices).
   * The method throws an IllegalArgumentException, if element is no Key or the meta data are not consistent.
   * @param index the position in the KeyList
   * @param key the Key element to insert
   */
  public void add(int index, Object key) {
    if ( !(key instanceof Key) )
      // throw an exception
      throw new IllegalArgumentException("element is no Key instance!");
    // check KeyDescriptor
    if ( descriptor != ((Key)key).getKeyDescriptor() )
      // throw an exception
      throw new IllegalArgumentException("Key "+key+" is no Key of type "+descriptor.getName());
    super.add(index, key);
  }

  /**
   * Appends the specified Key to the end of this <code>KeyList</code>.
   * The method throws an IllegalArgumentException, if element is no Key or the meta data are not consistent.
   * @param key the Key element to insert
   */
  public boolean add(Object key) {
    if ( !(key instanceof Key) )
      // throw an exception
      throw new IllegalArgumentException("element is no Key instance!");
    // check KeyDescriptor
    if ( descriptor != ((Key)key).getKeyDescriptor() )
      // throw an exception
      throw new IllegalArgumentException("Key "+key+" is no Key of type "+descriptor.getName());
    return super.add(key);
  }

  /**
   * Appends all of the elements in the specified collection to the end of this <code>KeyList</code>. 
   * The method throws an IllegalArgumentException, if the Collection is no KeyList or the meta data are not consistent.
   * @throws IllegalArgumentException , if the Collection is no KeyList or the meta data are not consistent.
   */
  public boolean addAll(Collection keyList) {
    if ( !(keyList instanceof KeyList) )
      // throw an exception
      throw new IllegalArgumentException("Collection is no KeyList !");
    // check KeyDescriptor
    if ( descriptor != ((KeyList)keyList).getDescriptor() )
      // throw an exception
      throw new IllegalArgumentException("KeyList "+keyList+" is not of type "+descriptor.getName());
    return super.addAll(keyList);
  }

  /**
   * Inserts all of the elements in the specified collection into this <code>KeyList</code> at the specified position (optional operation). 
   * The method throws an IllegalArgumentException, if the Collection is no KeyList or the meta data are not consistent.
   * @throws IllegalArgumentException , if the Collection is no KeyList or the meta data are not consistent.
   */
  public boolean addAll(int index, Collection keyList) {
    if ( !(keyList instanceof KeyList) )
      // throw an exception
      throw new IllegalArgumentException("Collection is no KeyList !");
    // check KeyDescriptor
    if ( descriptor != ((KeyList)keyList).getDescriptor() )
      // throw an exception
      throw new IllegalArgumentException("KeyList "+keyList+" is not of type "+descriptor.getName());
    return super.addAll(index, keyList);
  }

  /**
   * Replaces the Key at the specified position in this <code>KeyList</code> with the specified Key.
   * The method throws an IllegalArgumentException, if element is no Key or the meta data are not consistent.
   * @param key the Key element to insert
   */
  public Object set(int index, Object key) {
    if ( !(key instanceof Key) )
      // throw an exception
      throw new IllegalArgumentException("element is no Key instance!");
    // check KeyDescriptor
    if ( descriptor != ((Key)key).getKeyDescriptor() )
      // throw an exception
      throw new IllegalArgumentException("Key "+key+" is no Key of type "+descriptor.getName());
    return super.set(index, key);
  }

	/* (non-Javadoc)
	 * @see java.lang.Object#clone()
	 */
	public Object clone() {
		KeyList keyList = new KeyList(getDescriptor());
		for (Iterator i = iterator(); i.hasNext(); ) {
			Object o = i.next();
			if (o instanceof Key) {
				try {
					Key clonnedKey = (Key) ((Key) o).clone();
					keyList.add(clonnedKey);
				} catch (CloneNotSupportedException e) {
					// Clonning is not supported by Key, just copy the reference
					keyList.add(o);
				}
			} else {
				// Just add reference
				keyList.add(o);
			}
		}
		return keyList;
	}

}
