package com.sap.caf.rt.ui.cool.generic;

import java.util.HashMap;
import java.util.Iterator;

import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.col.edo.IEdoIndexTable;
import com.sap.tc.logging.Location;

/**
 * Common Cache for dependant aspects. Used by Aspect and AspectRow to store 
 * sibling aspects and related aspects.</p>
 * 
 * @todo make Relation a private inner class of this cache?
 * 
 * @author Frank Weigel, Helmut Mueller
 */
class DependantAspectCache {

	/** Logging properites for this class */
	private static final String APPLICATION	= DependantAspectCache.class.getName();
	private static final String jARMRequest = AbstractModelClass.jARMReqPrefix+APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);
	
  /**
   * @link aggregation
   * @associates <{Relation}>
   * @supplierCardinality 0..*
   * @clientCardinality 1
   */
  private final HashMap relations = new HashMap();
  
  public void addRelatedAspect(Aspect sourceAspect, String relationName, Aspect targetAspect, IEdoIndexTable indexTable) {
    // cache the relation
    Relation relation = new Relation( relationName, targetAspect, indexTable );
    targetAspect.setSourceAspectOfCreatingRelation(sourceAspect);
    relations.put( relationName, relation );
  }
  
  public void addRelatedAspect(AspectRow sourceAspectRow, String relationName, Aspect targetAspect, IEdoIndexTable indexTable) {
	// cache the relation
	 Relation relation = new Relation( relationName, targetAspect, indexTable );
	 targetAspect.setSourceAspectOfCreatingRelation((Aspect)sourceAspectRow.getAspect());
	 relations.put(relationName, relation);
	 
	 targetAspect.setSourceAspectRowOfCreatingRelation(sourceAspectRow);
  }
  
  /**
   * Returns the relation with the given name or <code>null</code> if no 
   * such relation is cached.
   */
  protected Relation getRelation( String relationName ) {
    return (Relation) relations.get( relationName );
  }

  /**
   * Get the target aspect for the relation with the given name or <code>null</code>
   * if no such relation is cached.
   */
  public Aspect getRelatedAspect( String relationName ) {
    Relation relation = (Relation) relations.get( relationName );
    if ( relation != null )
      return relation.getTargetAspect();
    return null;
  }
  
  /** 
   * @associates <{Aspect}>
   * @clientCardinality 1
   * @supplierCardinality 0..*
   * @supplierRole siblingAspects
   * @undirected*/
  private final HashMap siblingAspects = new HashMap();
  
  /**
   * Add sibling Aspect.
   */
  protected void addSiblingAspect(String aspectName, Aspect aspect) {
    siblingAspects.put(aspectName, aspect);
  }
  
  /**
   * get sibling Aspect
   */
  protected Aspect getSiblingAspect(String aspectName) {
    return (Aspect)siblingAspects.get(aspectName);
  }

  /**
   * Invalidate whole cache. Propagates invalidation to all cached aspects and then releases 
   * all references to them.
   */
  public void invalidate() {
		final String method = jARMRequest + ":invalidate()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    // invalidate related aspects and remove all relations 
	    Iterator it = relations.values().iterator();
	    while ( it.hasNext() ) {
	      Relation relation = (Relation) it.next();
	      relation.getTargetAspect().invalidateInternal();
	    }
	    relations.clear();
	    
	    // invalidate and remove all sibling aspects
	    it = siblingAspects.values().iterator();
	    while ( it.hasNext() ) {
	      Aspect siblingAspect = (Aspect) it.next();
	      siblingAspect.invalidateInternal();
	    }
	    siblingAspects.clear();
		}
		finally {	 
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }

}
