package com.sap.caf.rt.ui.cool.generic;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.resource.NotSupportedException;

import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.dictionary.runtime.IDataProvider;
import com.sap.dictionary.runtime.IField;
import com.sap.dictionary.runtime.IStructure;
import com.sap.dictionary.types.services.IBackendConversionRule;
import com.sap.tc.cmi.metadata.CMICardinality;
import com.sap.tc.cmi.metadata.CMISetting;
import com.sap.tc.cmi.metadata.CMISettingDefinition;
import com.sap.tc.cmi.metadata.ICMIModelClassInfo;
import com.sap.tc.cmi.metadata.ICMIModelClassPropertyInfo;
import com.sap.tc.cmi.metadata.ICMIModelInfo;
import com.sap.tc.cmi.metadata.ICMIRelationRoleInfo;
import com.sap.tc.col.client.metadata.api.IAspectActionDescriptor;
import com.sap.tc.col.client.metadata.api.IAspectDescriptor;
import com.sap.tc.col.client.metadata.api.IFieldDescriptor;
import com.sap.tc.col.client.metadata.api.IQueryDescriptor;
import com.sap.tc.col.client.metadata.api.IRelationDescriptor;
import com.sap.tc.col.client.metadata.api.IStructureDescriptor;
import com.sap.tc.col.edo.IEdoStructure;
import com.sap.tc.logging.Location;

/**
 * Realizes meta data of <COL> model classes.</p>
 * there are four types of <COL> model classes, which we can differ from name prefix of meta data. 
 * There are: AspectRow ( prefix "Aspect." ), Query ( prefix "Query." ),  Action ( prefix "Action." ) and Structure ( prefix "Structure." )
 * 
 * @author Helmut Mueller, Stephan Ritter
 */
public class ColModelClassInfo implements ICMIModelClassInfo {

	/** Logging properites for this class */
	private static final String APPLICATION	= ColModelClassInfo.class.getName();
	private static final String jARMRequest = AbstractModelClass.jARMReqPrefix+APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);
  
  // constants for prefix definitions
  /**
   * constant string prefix for AspectRow model classes
   */
  public final static String ASPECT = "$Aspect$";
  /**
   * constant string prefix for Query model classes
   */
  public final static String QUERY = "$Query$";
  /**
   * constant string prefix for Action model classes
   */
  public final static String ACTION = "$Action$";
  /**
   * constant string prefix for Structure model classes
   */
  public final static String STRUCTURE = "$Structure$";
  
  /**
   * the name of the <COL> model class, which is derived from <COL> name and
   * prefixed with the constant strings defined in this class, where the prefix
   * corresponds to the type of <COL> class
   */
  private String name;
  
  /**
   * map of all meta data of the properties
   */
  private HashMap propertyInfos = new HashMap();
  /**
	* list of all meta data of the properties
	*/
  private List propertyInfosList = new ArrayList();
  
  /**
   * map of all meta data of the relation roles
   */
  private HashMap sourceRoleInfos;
  /**
   * the meta data of the associated <COL> model
   * @supplierCardinality 1
   * @clientCardinality 0..*
   * @undirected
   */
  private ColModelInfo modelInfo;
  
  /**
   * the wrapped <COL> meta data object for <code>Structure</code>, if this class
   * has a name prefixed with "Structure." or <code>null</code> otherwise
   */
  private IStructureDescriptor structureDescriptor;
  /**
   * the wrapped <COL> meta data object for <code>Aspect</code>, if this class
   * has a name prefixed with "Aspect." or <code>null</code> otherwise
   */
  private IAspectDescriptor aspectDescriptor;
  /**
   * the wrapped <COL> meta data object for <code>Query</code>, if this class
   * has a name prefixed with "Query." or <code>null</code> otherwise
   */
  private IQueryDescriptor queryDescriptor;
  /**
   * the wrapped <COL> meta data object for <code>Action</code>, if this class
   * has a name prefixed with "Action." or <code>null</code> otherwise
   */
  private IAspectActionDescriptor actionDescriptor;

  /**
   * constructor of ColModelClassInfo for StructureDescriptor
   * @param structureDescriptor the meta data of a <COL> <code>Structure</code>
   * @param modelInfo meta data of the associated <COL> model
   */
  protected ColModelClassInfo(IStructureDescriptor structureDescriptor, ColModelInfo modelInfo ) {
    this.structureDescriptor = structureDescriptor;
    this.name = new String(STRUCTURE + structureDescriptor.getName());
    this.modelInfo = (ColModelInfo) modelInfo;
    int size = structureDescriptor.size();
    for (int i = 0; i< size; i++) {
      IFieldDescriptor current = structureDescriptor.getFieldDescriptor(i);
	  ColModelClassPropertyInfo currentInfo = new ColModelClassPropertyInfo(current, this);
      propertyInfos.put(current.getName(), currentInfo);
	  propertyInfosList.add(currentInfo);
    }  
  }
  /**
   * constructor of ColModelClassInfo for AspectDescriptor
   * @param aspectDescriptor the meta data of a <COL> <code>Aspect</code>
   * @param modelInfo meta data of the associated <COL> model
   */
  protected ColModelClassInfo(IAspectDescriptor aspectDescriptor, ColModelInfo modelInfo) {
    this.aspectDescriptor = aspectDescriptor;
    IStructureDescriptor structureDescriptor = aspectDescriptor.getStructure();
    this.name = new String(ASPECT + aspectDescriptor.getName());
    this.modelInfo = (ColModelInfo) modelInfo;
    int size = structureDescriptor.size();
    for (int i = 0; i< size; i++) {
      IFieldDescriptor currentFD = structureDescriptor.getFieldDescriptor(i);
	  ColModelClassPropertyInfo currentInfo = new ColModelClassPropertyInfo(currentFD, this);
      propertyInfos.put(currentFD.getName(), currentInfo);
	  propertyInfosList.add(currentInfo);
    }  
  }

  /**
   * constructor ColModelClassInfo for QueryDescriptor
   * @param queryDescriptor the meta data of a <COL> <code>Query</code>
   * @param modelInfo meta data of the associated <COL> model
   */
  protected ColModelClassInfo(IQueryDescriptor queryDescriptor, ICMIModelInfo modelInfos ) {
    this.queryDescriptor = queryDescriptor;
    this.name = new String(QUERY + queryDescriptor.getName());
    this.modelInfo = (ColModelInfo) modelInfos;
  }

  /**
   * constructor ColModelClassInfo for AspectActionDescriptor
   * @param aspectActionDescriptor the meta data of a <COL> <code>Action</code>
   * @param modelInfo meta data of the associated <COL> model
   */
  protected ColModelClassInfo(IAspectActionDescriptor aspectActionDescriptor, ICMIModelInfo modelInfos) {
    this.actionDescriptor = aspectActionDescriptor;
    this.name = new String(ACTION + aspectActionDescriptor.getName());
    
    this.modelInfo = (ColModelInfo) modelInfos;
  }

  /**
   * returns the meta data of the associated COL model
   * @return ICMIModelInfo the associated model meta data
   */
  public ICMIModelInfo getModelInfo() {
    return modelInfo;
  }
  
  /**
   * returns a collection of all properties meta data
   * @return Collection a collection of all properties meta data
   */
  public Collection getPropertyInfos() {
    return propertyInfosList;//propertyInfos.values();
  }

  /**
   * returns an iterator over all properties meta data
   * @return an iterator over all properties meta data
   */
  public Iterator iteratePropertyInfos() {
    return getPropertyInfos().iterator();
  }

  /**
   * returns meta data of property with given name
   * @param name the name of the property
   * @return ICMIModelClassPropertyInfo the property object with given name
   */
  public ICMIModelClassPropertyInfo getPropertyInfo(String name) {
    if ( !propertyInfos.containsKey(name) )
      throw new IllegalArgumentException("Property with name '"+name+"' doesn't exist!");
    return (ICMIModelClassPropertyInfo) propertyInfos.get(name);
  }
  
  /**
   * returns meta data of property with given index
   * @param index the index of the property
   * @return ICMIModelClassPropertyInfo the property object with given index
   */
  ICMIModelClassPropertyInfo getPropertyInfo(int index) {
    String name;
    if ( structureDescriptor == null ) {
      name = aspectDescriptor.getStructure().getFieldDescriptor(index).getName();
    }
    else {
      name = structureDescriptor.getFieldDescriptor(index).getName();
    }
    return (ICMIModelClassPropertyInfo) propertyInfos.get(name);
  }

  /**
   * returns a collection of all relation roles meta data
   * @return Collection a collection of all relation roles meta data
   */
  public Collection getSourceRoleInfos() {
		final String method = jARMRequest + ":getSourceRoleInfos()";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		try {
	    // create source role infos on demand to avoid recursion in constructor
	    // todo: find a better name for the sourceRole?
	    if ( sourceRoleInfos == null ) {
	      sourceRoleInfos = new HashMap();
	      if ( aspectDescriptor != null ) {
	        IRelationDescriptor[] relationDescriptors = aspectDescriptor.getRelationDescriptors();
	        for (int i = 0; i< relationDescriptors.length; i++) {
	          IRelationDescriptor currentRD = relationDescriptors[i];
		  IAspectDescriptor targetAspectDescriptor = currentRD.getTargetAspectDescriptor();
		  ColModelInfo relModelInfo = (ColModelInfo)modelInfo.repositoryManager.getCMIModel(targetAspectDescriptor.getServiceModuleDescriptor().getName());

	          ColModelClassInfo targetModelClassInfo = (ColModelClassInfo) relModelInfo.getOrCreateModelClassInfo(targetAspectDescriptor);
	          ColModelRelationRoleInfo sourceRoleInfo = new ColModelRelationRoleInfo(currentRD.getName(),
	              this, targetModelClassInfo, currentRD.getAttributeTargetCardinality() );
	          sourceRoleInfos.put(currentRD.getName(), sourceRoleInfo);
	        }
	      }
	      else if ( queryDescriptor != null ) {
	         // first input parameters
	        ColModelClassInfo targetModelClass;
	        ColModelRelationRoleInfo sourceRoleInfo;
	        if ( queryDescriptor.hasInputParameters() ) {
	          targetModelClass = (ColModelClassInfo) modelInfo.getOrCreateModelClassInfo(queryDescriptor.getInputParameters());
	          sourceRoleInfo = new ColModelRelationRoleInfo(Query.INPUT_PARAMETER, this, targetModelClass, CMICardinality.CARDINALITY_ZERO_TO_ONE);
	          sourceRoleInfos.put(sourceRoleInfo.getName(), sourceRoleInfo);
	        }
	      
	        // next result	        
	        targetModelClass = (ColModelClassInfo) modelInfo.getOrCreateModelClassInfo(queryDescriptor.getResultAspectDescriptor() );
	        sourceRoleInfo = new ColModelRelationRoleInfo(Query.RESULT,
	        this, targetModelClass, CMICardinality.CARDINALITY_MANY);
	        sourceRoleInfos.put(sourceRoleInfo.getName(), sourceRoleInfo);
	      }
	      else if ( actionDescriptor != null && actionDescriptor.hasInputParameters() ) {
	        // only input parameters
	        ColModelClassInfo targetModelClass = (ColModelClassInfo) modelInfo.getOrCreateModelClassInfo(actionDescriptor.getInputParameters());
	        ColModelRelationRoleInfo sourceRoleInfo = new ColModelRelationRoleInfo(Query.INPUT_PARAMETER,
	            this, targetModelClass, CMICardinality.CARDINALITY_ZERO_TO_ONE);
	        sourceRoleInfos.put(sourceRoleInfo.getName(), sourceRoleInfo);
	      
	      }
	    }  
	    return sourceRoleInfos.values();
		}
		finally {	 
			CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		}
  }

  /**
   * returns an iterator for all relation roles meta data
   * @return an iterator for all relation roles meta data
   */
  public Iterator iterateSourceRoleInfos() {
    return getSourceRoleInfos().iterator();
  }

  /**
   * returns meta data of relation role with given name
   * @return ICMIModelRelationRoleInfo meta data of relation role with given name
   * @deprecated
   */
  public ICMIRelationRoleInfo getSourceRoleInfo(String name) {
    if (sourceRoleInfos == null) {
      getSourceRoleInfos(); 
    }
    return (ICMIRelationRoleInfo) sourceRoleInfos.get(name);
  }

  /**
   * @see com.sap.tc.cmi.metadata.ICMIModelClassInfo#getTargetRoleInfo(String)
   */
  public ICMIRelationRoleInfo getTargetRoleInfo(String targetRoleName) {
    ICMIRelationRoleInfo targetRoleInfo;
    for (Iterator it = iterateSourceRoleInfos(); it.hasNext(); ) {
      targetRoleInfo = ((ICMIRelationRoleInfo) it.next()).getOtherRoleInfo();
      if (targetRoleInfo.getName().equals(targetRoleName))
        return targetRoleInfo;
    }
    return null;
  }

  /**
   * convenience method which returns the metadata for the model class to which this model class
   * is related via a relation, where the relation is determined through the
   * name of its source role at this model class.
   */
  public ICMIModelClassInfo getRelatedModelClassInfo(String targetRoleName) {
    if (getTargetRoleInfo(targetRoleName) != null)
      return getTargetRoleInfo(targetRoleName).getModelClassInfo();
    return null;  
  }

  /**
   * returns always <code>true</code>, cause COL model classes are generic
   * @return boolean always <code>true</code>, cause COL model classes are generic
   */
  public boolean isGeneric() {
    return true;
  }

  /**
   * returns the name of the <COL> model class, which is derived from <COL> name and
   * prefixed with the constant strings defined in this class, where the prefix
   * corresponds to the type of <COL> class
   * @return String the name of the <COL> model class, which is derived from <COL> name
   */
  public String getName() {
    return name;
  }

  
  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#getSetting(CMISettingDefinition)
   */
  public CMISetting getSetting(CMISettingDefinition settingDef) {
    throw new java.lang.UnsupportedOperationException();
  }

  
  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#setSetting(CMISettingDefinition, String)
   */
  public void setSetting(CMISettingDefinition settingDef, String value) {
    throw new java.lang.UnsupportedOperationException();
  }

  
  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#getSettings()
   */
  public Map getSettings() {
    throw new java.lang.UnsupportedOperationException();
  }

 
  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#supports(String)
   */
  public boolean supports(String option) {
    throw new java.lang.UnsupportedOperationException();
  }

  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#supportedOptions()
   */
  public Collection supportedOptions() {
    throw new java.lang.UnsupportedOperationException();
  }

 
  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#addSupportedOption(String)
   */
  public void addSupportedOption(String option) {
    throw new java.lang.UnsupportedOperationException();
  }
  
  public IStructure getStructureType() {
		return null;//throw new RuntimeException("Boooom!");
  }
  
}
