package com.sap.caf.rt.ui.cool.generic;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.sap.tc.col.client.generic.api.IAspectRow;
import com.sap.tc.col.client.generic.api.IKey;
import com.sap.tc.col.client.generic.api.IKeyList;
import com.sap.tc.col.client.metadata.api.IStructureDescriptor;
import com.sap.tc.logging.Location;

/**
 * Manages local changes of <code>AspectRow</code>s.<p>
 * 
 * Every change (update, insert, delete, Action ) of the associated <code>Aspect</code> is registered in this class. Each <code>Aspect</code>,
 * which needs management of local changes aggregates this class. The concerned <code>AspectRow</code>s are not hold as copy or reference, but in form of
 * indices.<p>
 * The class offers methods to
 * <ul>
 * <li> register inserts, updates, deletes, Actions
 * <li> getting registered changes
 * <li> resetting changes
 * </ul> 
 * When a change is registered it is checked, whether a change for this <code>AspectRow</code> exists,
 * or a change makes another change unnecessary.
 * 
 * @author Helmut Mueller
 */
class AspectChanges {

	AspectChanges(Aspect aspect) {
		this.structureDescriptor = aspect.getDescriptor().getStructure();
		this.aspect = aspect;
	}

	private IStructureDescriptor structureDescriptor;

	private Aspect aspect;

	/**
	 * logging object
	 */
	private final static Location logger = Location.getLocation(AspectChanges.class);

	/**
	 * registers a changing <code>Action</code> 
	 */
	protected void registerAction() {
		logger.entering("registerAction");
		changingAction = true;
		logger.exiting();
	}

	/**
	 * resets a changing <code>Action</code> 
	 */
	protected void resetAction() {
		logger.entering("resetAction");
		collectActionChanges();
		changingAction = false;
		logger.exiting();
	}

	private void collectActionChanges() {
		if (changingAction) {
			changingAction2 = changingAction;
		}
	}

	/**
	 * registers an <code>AspectRow</code> as inserted, the <code>AspectRow</code> is given as Key.<p>
	 * 
	 * @param key the key of the <code>AspectRow</code>, which should be registered as inserted
	 * @exception <code>IllegalArgumentException</code>, if key is null
	 */
	protected void registerInsert(IKey key) {
		logger.entering("registerInsert");
		try {
			if (key == null) {
				// throw an exception
				throw new IllegalArgumentException("Key of AspectRow '" + aspect.getName() + "' is null!");
			}
			insertedRows.add(key);
		} finally {
			logger.exiting();
		}
	}

	/**
	 * registers for an inserted <code>AspectRow</code> the key of the related source AspectRow,
	 * the <code>AspectRow</code>s are given as Keys.<p>
	 * 
	 * @param key the key of the <code>AspectRow</code>, which should be registered as inserted
	 * @param sourceKey the key of the source<code>AspectRow</code> of the relation to insert
	 * @exception <code>IllegalArgumentException</code>, if key or sourceKey is null
	 */
	protected void registerInsertRelation(IKey key, IKey sourceKey) {
		logger.entering("registerInsertRelation");
		try {
			if (key == null || sourceKey == null) {
				// throw an exception
				throw new IllegalArgumentException(
					"Key of AspectRow '" + aspect.getName() + "' is null or Key of sourceAspect!");
			}
			insertedRelatedRows.put(key, sourceKey);
		} finally {
			logger.exiting();
		}
	}

	/**
	 * registers an <code>AspectRow</code> as updated, the <code>AspectRow</code> is given as Key.<p>
	 * This update could be a real update or an update of a new inserted <code>AspectRow</code>, so the name of the
	 * changed field is hold with the inserted <code>AspectRow</code>.
	 * @param key the key of the <code>AspectRow</code>, which should be registered as updated
	 * @param fieldName the name of the updated field (attribute) of the <code>AspectRow</code>
	 * @exception <code>IllegalArgumentException</code>, if key is null
	 */
	protected void registerUpdate(IKey key, String fieldName) {
		logger.entering("registerUpdate");
		try {
			if (key == null) {
				// throw an exception
				throw new IllegalArgumentException("Key of AspectRow '" + aspect.getName() + "' is null!");
			}

			// first look whether this row is an inserted row
			if (insertedRows.contains(key))
				return;
			int index = structureDescriptor.getFieldIndex(fieldName);
			if (index < 0) {
				throw new IllegalArgumentException("Illegal field name '" + fieldName + "'");
			}

			// .. then whether this row is marked for delete
			if (deletedKeyList != null && deletedKeyList.contains(key)) {
				if (updatedRows.containsKey(key)) {
					updatedRows.remove(key);
				}
				return;
			}

			// .. otherwise it is an updated row
			BitSet changedFields = (BitSet) updatedRows.get(key);
			if (changedFields == null) {
				// no changes exist for this AspectRow
				// create a new entry in updateRows Map and a new ArrayList with 
				// field names
				changedFields = new BitSet(structureDescriptor.size());
				updatedRows.put(key, changedFields);
			}
			changedFields.set(index);
		} finally {
			logger.exiting();
		}
	}

	/**
	 * resets all registered updates
	 */
	protected void resetAllUpdates() {
		logger.entering("resetAllUpdates");
		try {
			collectUpdateChanges();
			updatedRows.clear();
		} finally {
			logger.exiting();
		}
	}

	private void collectUpdateChanges() {
		for (Iterator i = updatedRows.entrySet().iterator(); i.hasNext(); ) {
			Map.Entry entry = (Map.Entry) i.next();
			Object key = entry.getKey();
			if (insertedRows2.indexOf(key) < 0 && (deletedKeyList2 == null || (deletedKeyList2 != null && deletedKeyList2.indexOf(key) < 0))) {
				// Actually, we are not interested in exact fields changed, but once it is a point the following code must be changed.
				updatedRows2.put(key, entry.getValue());
			}
		}
	}

	/**
	 * resets registered update of given AspectRows
	 */
	protected void resetUpdates(int[] rows) {
		logger.entering("resetUpdates");
		try {
			for (int i = 0; i < rows.length; i++) {
				IAspectRow row = aspect.getAspectRow(rows[i]);
				IKey key = row.getKey();
				boolean flag = true;
				flag &= insertedRows2 == null ? true : (insertedRows2.indexOf(key) < 0) ;
				flag &= deletedKeyList2 == null ? true : (deletedKeyList2.indexOf(key) < 0) ;
				if (flag) {
						updatedRows2.put(key, updatedRows.get(key));
				}
				updatedRows.remove(key);
			}
		} finally {
			logger.exiting();
		}
	}

	/**
	 * resets an insert of AspectRow with given Key
	 */
	protected void resetInsert(IKey key) {
		logger.entering("resetInsert");
		try {
			collectInsertChange(key);
			insertedRows.remove(key);
			insertedRelatedRows.remove(key);
		} finally {
			logger.exiting();
		}
	}

	private void collectInsertChange(IKey key) {
		if (insertedRows2.indexOf(key) < 0) {
			insertedRows2.add(key);
			insertedRelatedRows2.put(key, insertedRelatedRows.get(key));
			updatedRows2.remove(key);
			if (deletedKeyList2 != null) { 
				deletedKeyList2.remove(key);
			}
		}
	}

	/**
	 * resets all inserts registered
	 */
	protected void resetAllInserts() {
		logger.entering("resetAllInserts");
		try {
			for (Iterator i = insertedRows.iterator(); i.hasNext(); ) {
				collectInsertChange((IKey) i.next());
			}
			insertedRows.clear();
			insertedRelatedRows.clear();
		} finally {
			logger.exiting();
		}
	}

	/**
	 * resets all deletes of AspectRows, but these in KeyList. If KeyList is null or empty
	 * all deletes are reset.
	 */
	protected void resetDeleteButKeyList(IKeyList keyList) {
		logger.entering("resetDeleteButKeyList");
		try {
			if (deletedKeyList != null) {
				collectDeleteChanges(keyList);
				if (keyList != null) {
					deletedKeyList.retainAll(keyList);
				} else {
					deletedKeyList.clear();
				}
			}
		} finally {
			logger.exiting();
		}
	}

	private void collectDeleteChanges(IKeyList keyList) {
		for (Iterator i = deletedKeyList.iterator(); i.hasNext(); ) {
			IKey key = (IKey) i.next(); 
			if ((keyList == null) || (keyList != null && keyList.indexOf(key) < 0)) {
				deletedKeyList2.add(key);
				updatedRows2.remove(key);
				insertedRows2.remove(key);
				insertedRelatedRows2.remove(key);
			}
		}
	}

	/**
	 * registers an <code>AspectRow</code> as deleted, the <code>AspectRow</code> is given as key.<p>
	 * The method checks, whether the given key exists as inserted or updated row and if this is the case these
	 * registered changes are removed.
	 * @param key the key of the <code>AspectRow</code>, which should be registered as deleted
	 * @exception <code>IllegalArgumentException</code>, if rowIndex is null or negative
	 */
	protected void registerDelete(AspectRow row) {
		logger.entering("registerDelete");
		try {
			Key key = (Key) row.getKey();
			// first check, whether this row is a inserted one
			if (insertedRows.contains(key)) {
				// was locally inserted
				insertedRows.remove(key);
				return;
			}
			// .. then whether an update was registered for this row
			BitSet fields = (BitSet) updatedRows.get(key);
			if (fields != null) {
				// remove from updated rows ..
				updatedRows.remove(key);
			}
			if (deletedKeyList == null) {
				deletedKeyList = new KeyList(key.getKeyDescriptor());
				deletedKeyList2 = new KeyList(key.getKeyDescriptor());
			}
			if (!deletedKeyList.contains(key)) {
				deletedKeyList.add(key);
			}
		} finally {
			logger.exiting();
		}
	}

	/**
	 * returns an array of all inserted <code>AspectRow</code>s as <code>Array</code> of <code>int</code>. If no
	 * inserted <code>AspectRow</code> exists, an empty <code>Array</code> is returned.
	 * @return int[] an array of all inserted <code>AspectRow</code>s as <code>Array</code> of <code>int</code> or 
	 * an empty <code>Array</code>, if no inserted <code>AspectRow</code> exists
	 */
	protected int[] getInsertedAspectRows() {
		Iterator it = insertedRows.iterator();
		int[] indexArray = new int[insertedRows.size()];
		int i = 0;
		while (it.hasNext()) {
			AspectRow row = (AspectRow) aspect.getAspectRow((IKey) it.next());
			indexArray[i++] = row.getIndex();
		}
		return indexArray;
	}

	protected IKey getRelatedInsertedKey(IKey key) {
		return (IKey) insertedRelatedRows.get(key);
	}

	/**
	 * returns the key of the source AspectRow, which is related to the inserted AspectRow with
	 * given index
	 */
	protected IKey getRelatedInsertedKey(int index) {
		if (insertedRelatedRows.isEmpty())
			return null;
		IKey key = (IKey) insertedRows.get(index);

		return getRelatedInsertedKey(key);
	}

	/**
	 * returns an array of all updated <code>AspectRow</code>s as <code>Array</code> of <code>int</code>. If no
	 * updated <code>AspectRow</code> exists, an empty <code>Array</code> is returned.
	 * @return int[] an array of all updated <code>AspectRow</code>s as <code>Array</code> of <code>int</code> or 
	 * an empty <code>Array</code>, if no updated <code>AspectRow</code> exists
	 */
	protected int[] getChangedAspectRows() {
		Iterator it = updatedRows.keySet().iterator();
		int[] indexArray = new int[updatedRows.keySet().size()];
		int i = 0;
		while (it.hasNext()) {
			AspectRow row = (AspectRow) aspect.getAspectRow((IKey) it.next());
			indexArray[i++] = row.getIndex();
		}
		return indexArray;
	}

	/**
	 * returns a KeyList of all removed <code>AspectRow</code>s. If no
	 * removed <code>AspectRow</code> exists, <code>null</code> is returned.
	 */
	protected KeyList getDeletedKeys() {
		return deletedKeyList;
	}

	/**
	 * returns <code>true</code> if there are no changes (insert, update, delete), otherwise <code>false</code>
	 * @return boolean <code>true</code> if there are no changes (insert, update, delete), otherwise <code>false</code>
	 */
	protected boolean isEmpty() {
		return ((deletedKeyList == null || deletedKeyList.isEmpty()) && updatedRows.isEmpty() && insertedRows.isEmpty());
	}
	/**
	 * resets all registered changes (insert, update, delete)
	 */
	protected void reset() {
		resetDeleteButKeyList(null);
		resetAllUpdates();
		resetAllInserts();
		resetAction();
	}

	/**
	 * method returns the information about the different types of registered
	 * changes, for each registered change type a bit is set
	 */
	protected BitSet getChangesState() {
		BitSet changeState = new BitSet(Aspect.NUMBER_OF_HINTS);
		if (insertedRows.size() > 0) {
			changeState.set(Aspect.INVALIDATE_ON_INSERT);
		}
		if (updatedRows.size() > 0) {
			changeState.set(Aspect.INVALIDATE_ON_UPDATE);
		}
		if (deletedKeyList != null && !deletedKeyList.isEmpty()) {
			changeState.set(Aspect.INVALIDATE_ON_DELETE);
		}
		if (changingAction) {
			changeState.set(Aspect.INVALIDATE_ON_ACTION);
		}
		return changeState;
	}

	/**
	 * returns the Map with the changed AspectRows
	 */
	protected Map getUpdatedRows() {
		return updatedRows;
	}

	protected void commitChanges() {
		resetCollectedChanges();
	}

	private void resetCollectedChanges() {
		if (deletedKeyList2 != null) {
			//deletedKeyList2 = null;
			deletedKeyList2.clear();
		}
		updatedRows2.clear();
		insertedRows2.clear();
		insertedRelatedRows2.clear();
		changingAction2 = false;
	}

	protected void rollbackChanges() {
		// The clean up code are repeated here to avoid unnecessary calls of the collectXXXChanges() methods
		// that can improve performance of the operation dramatically when number of changes is big enough.
		updatedRows.clear();
		insertedRows.clear();
		insertedRelatedRows.clear();
		changingAction = false;
		if (deletedKeyList != null) {
			deletedKeyList.clear();
			deletedKeyList.addAll(deletedKeyList2);
		}
		updatedRows.putAll(updatedRows2);
		insertedRows.addAll(insertedRows2);
		insertedRelatedRows.putAll(insertedRelatedRows2);
		changingAction = changingAction2;
		resetCollectedChanges();
	}

	/**
	 * KeyList with all deleted rows
	 */
	private KeyList deletedKeyList;

	/**
	 * all updated AspecRows are managed in a HashMap with the index as Key 
	 * and an ArrayList of all updated Fields
	 */
	private Map updatedRows = new HashMap();

	/**
	 * all inserted AspecRows are managed in a HashMap with the index as Key 
	 * and an ArrayList of all inserted Fields
	 */
	private ArrayList insertedRows = new ArrayList();

	private Map insertedRelatedRows = new HashMap();

	private boolean changingAction = false;

	/* --- The following duobles are used for storing changes of aspects that are waiting for transaction to be commited. --- */
	/**
	 * KeyList with all deleted rows
	 */
	private KeyList deletedKeyList2;

	/**
	 * all updated AspecRows are managed in a HashMap with the index as Key 
	 * and an ArrayList of all updated Fields
	 */
	private Map updatedRows2 = new HashMap();

	/**
	 * all inserted AspecRows are managed in a HashMap with the index as Key 
	 * and an ArrayList of all inserted Fields
	 */
	private ArrayList insertedRows2 = new ArrayList();

	private Map insertedRelatedRows2 = new HashMap();

	private boolean changingAction2 = false;

}
