/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/model/api/WDModelFactory.java#3 $
 */
package com.sap.tc.webdynpro.progmodel.model.api;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.sap.tc.logging.Category;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;
import com.sap.tc.webdynpro.basesrvc.util.LoggingFormatter;
import com.sap.tc.webdynpro.progmodel.model.core.AbstractModelMaintainer;
import com.sap.tc.webdynpro.progmodel.model.core.IMaintainableModel;
import com.sap.tc.webdynpro.progmodel.model.core.Model;
import com.sap.tc.webdynpro.progmodel.model.core.AbstractModelMaintainer.IInstanceIdMaintainable;
import com.sap.tc.webdynpro.progmodel.model.core.AbstractModelMaintainer.IScopeTypeMaintainable;
import com.sap.tc.webdynpro.services.exceptions.WDRuntimeException;
import com.sap.tc.webdynpro.services.session.IMaintainScope;
import com.sap.tc.webdynpro.services.session.IScope;
import com.sap.tc.webdynpro.services.session.Utils;

/**
 * Factory for Web Dynpro Models.
 *
 * Model instances are referenced by their implementing class and can be assigned
 * a specific scope (see {@link WDModelScopeType}) and instance id.
 *
 * @author SAP
 * @SAPWebDynproPart 2
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/model/api/WDModelFactory.java#3 $
 * 
 */
public class WDModelFactory
{

  // Logging
  private static final Location logger = Location.getLocation (WDModelFactory.class.getName()) ;
  static {
    Location.getLocation("ID.com.sap.tc.webdynpro.progmodel.api").infoT("$Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/model/api/WDModelFactory.java#3 $");
  }

  /**
   * factory method, which returns an instance within the default
   * scope representing a specific model type (param modelClazz)
   *
   * If no instance of type modelClazz exists, a new instance is created
   * and added within the default scope. This is the default instance having no specific modelInstanceId.
   * 
   * Only one instance of a model can exist within the default scope having no specific modelInstanceId
   *
   * e.g. getModelInstance(MySpecialRFCAdapterModel.class)
   * @param modelClazz the specific model type class
   * @return the created model instance
   */
  static public IWDModel getModelInstance(Class modelClazz)
  {
    return getModelInstance(modelClazz, null, null);
  }

  /**
   * factory method, which returns an instance within the default
   * scope representing a specific model type (param modelClazz)
   * having the specific name or id (param modelInstanceId)
   *
   * If no instance with the given parameters exists, a new instance is created
   * and added within the default scope keyed by the modelInstanceId
   * 
   * Only one instance of a model can exist within a single scope using the same modelInstanceId
   *
   *  e.g. getModelInstance(MySpecialRFCAdapterModel.class)
   * @param modelClazz the specific model type class
   * @param modelInstanceId if more than one instance of a model is used in a single scope, this represents the name or id of the model instance.
   * @return the created model instance
   */
  static public IWDModel getModelInstance(Class modelClazz, String modelInstanceId)
  {
    return getModelInstance(modelClazz, null, modelInstanceId);
  }

  /**
   * factory method, which returns an instance within the
   * scope of the current thread's application instance representing
   * a specific model type (param modelClazz)
   *
   * If no instance of the Class modelClazz within the given scope exists, a new instance is created
   * and added within the given scope. This is the default instance having no specific modelInstanceId.
   * 
   * Only one instance of a model can exist within a single scope
   *
   * e.g. getModelInstance(MySpecialRFCAdapterModel.class)
   * @param modelClazz the specific model type class
   * @param scope the ModelScope, which defines the maximum lifespan of the model
   * @return the created model instance
   */
  static public IWDModel getModelInstance(Class modelClazz, WDModelScopeType scope)
  {
    return getModelInstance(modelClazz, scope, null);
  }

  /**
   * factory method, which returns an instance representing:
   * 1) a specific model type (param modelClazz)
   * 2) within the scope as defined by the ModelScope parameter (param scope)
   * 3) having the specific name or id (param modelInstanceId)
   * 
   * If no instance with the given parameters exists, a new instance is created
   * and added within the given scope keyed by the modelInstanceId
   * 
   * Only one instance of a model can exist within a single scope using the same modelInstanceId
   * 
   * e.g. getModel(RFCAdapterModel.class, ModelScope.APPLICATION)
   * @param modelClazz the specific model type class
   * @param scope the ModelScope, which defines the maximum lifespan of the model
   * @param modelInstanceId if more than one instance of a model is used in a single scope, this represents the name or id of the model instance.
   * @return the model instance (created if not already previously instantiated)
   */
  static public IWDModel getModelInstance(Class modelClazz, WDModelScopeType scope, String modelInstanceId)
  {
    String modelkey = getModelKeyForStorageInScope(modelClazz, modelInstanceId);
		//First check if the scope is null and if yes, determine if a defaultScope was registered
		if (scope==null)
			scope = lookupDefaultScopeForModel(modelClazz);        
		// If no defaultScope was registered, then use the Default => NO_SCOPE
		if (scope==null)
			scope = WDModelScopeType.NO_SCOPE;        
			
    IWDModel model = null;
    //Depending on ModelScope, get/place reference to Model instance in appropriate context
    if (scope==WDModelScopeType.NO_SCOPE)
    {
      // TODO simplify using a NO_SCOPE scope maintainer
			try
			{
      	model = getNewModelInstance( modelClazz , modelInstanceId);
			}
      // FWE catch Exception only, as Throwables should not be caught in WD framework
			catch (Exception ex) 
			{
				String errorMsg = "failed to create instance of model '"+modelClazz.getName()+
													"' in scope '"+scope.getScopeType()+"' with instanceId '"+modelInstanceId+
													"': " + LoggingFormatter.extractCallstack(ex);
				Category.SYS_USER_INTERFACE.logThrowableT( Severity.ERROR, logger, errorMsg, ex);
				logger.errorT("WDModelFactory", errorMsg);
        throw new WDRuntimeException(errorMsg, ex);
			}
    }
    else
    {
      IMaintainScope scopeMaintainer = getScopeMaintainer(scope);
      model = getOrCreateModelInstanceFromScopeMaintainer(
        modelClazz, modelkey, scopeMaintainer, modelInstanceId);
    }
    return model;
  }

  /**
   * reassigns an existing model to a new scope and/or instanceId:
   * 
   * returns an instance with the new scope and instanceID
   * if a model instance with the new scope and instanceID already exists, a WDModelException is thrown
   * 
   * Only one instance of a model can exist within a single scope using the same modelInstanceId
   * 
   * @param model the model instance
   * @param newScope WDModelScopeType, which defines the maximum lifespan of the model, null will not change the scope!
   * @param newInstanceId if more than one instance of a model is used in a single scope, this represents the name or id of the model instance!
   * @throws WDModelException if the newScope is null or if a model instance already exists with the new scope and modelInstanceId
   */
  static public synchronized void reassignModel(IWDModel model, WDModelScopeType newScope, String newInstanceId)
    throws WDModelException
  {
    if (newScope==null)
    {
      throw new WDModelException(
        WDModelException.MODEL_SCOPE_CANT_BE_NULL);
    }
    String prevInstanceId = model.getModelInstanceId();
    WDModelScopeType prevScope = model.getModelScope();
    IScope prevScopeMap = null;
    if(prevScope != WDModelScopeType.NO_SCOPE)
    	prevScopeMap = getScopeMaintainer(prevScope).getScope();
    IScope newScopeMap = null;
    Class modelClazz = model.getClass();
    if (prevScope==newScope)       // scope remains the same
    {
      newScopeMap = prevScopeMap;

      if (prevInstanceId==null && newInstanceId==null)
      {
        throw new WDModelException(
          WDModelException.MODEL_ALREADY_EXIST_WITH_INSTANCEID,
            new Object[]{model.getClass().getName(), newScope, newInstanceId});
      }
      else if ((prevInstanceId==null) || (newInstanceId==null))
      {
				if(newScopeMap!=null)
				{
	          // change instanceID from null to new or from something to null
	        prevScopeMap.remove(getModelKeyForStorageInScope(modelClazz, prevInstanceId));
	        newScopeMap.put(getModelKeyForStorageInScope(modelClazz, newInstanceId), model);
				}
      }    
      else if (prevInstanceId.equals(newInstanceId))
      {
        throw new WDModelException(
          WDModelException.MODEL_ALREADY_EXIST_WITH_INSTANCEID,
            new Object[]{model.getClass().getName(), newScope, newInstanceId});
      }
      else 
      {
				if(newScopeMap!=null)
				{
	          // change instanceID from something to something else
	        prevScopeMap.remove(getModelKeyForStorageInScope(modelClazz, prevInstanceId));
	        newScopeMap.put(getModelKeyForStorageInScope(modelClazz, newInstanceId), model);
				}
      }    
			assignScopeAndInstanceIdToModel(model, newScope, newInstanceId);
    }
    else // scope must also be changed
    {
			newScopeMap = null;
			if(newScope != WDModelScopeType.NO_SCOPE)
	      newScopeMap = getScopeMaintainer(newScope).getScope();

      if (prevInstanceId==null && newInstanceId==null)
      {
          // change scope only
        String modelKey = getModelKeyForStorageInScope(modelClazz, prevInstanceId);
        if (prevScopeMap!=null) prevScopeMap.remove(modelKey);
				if (newScopeMap!=null)  newScopeMap.put(modelKey, model);
      }
      else if ((prevInstanceId==null) || (newInstanceId==null))
      {
          // change instanceID (from null to new OR from something to null) AND change scope
				if (prevScopeMap!=null) prevScopeMap.remove(getModelKeyForStorageInScope(modelClazz, prevInstanceId));
				if (newScopeMap!=null)  newScopeMap.put(getModelKeyForStorageInScope(modelClazz, newInstanceId), model);
      }    
      else if (prevInstanceId.equals(newInstanceId))
      {
          // change scope only
        String modelKey = getModelKeyForStorageInScope(modelClazz, prevInstanceId);
				if (prevScopeMap!=null) prevScopeMap.remove(modelKey);
				if (newScopeMap!=null)  newScopeMap.put(modelKey, model);
      }
      else 
      {
          // change instanceID from something to something else AND change scope
				if (prevScopeMap!=null) prevScopeMap.remove(getModelKeyForStorageInScope(modelClazz, prevInstanceId));
				if (newScopeMap!=null)  newScopeMap.put(getModelKeyForStorageInScope(modelClazz, newInstanceId), model);
      }
          
			assignScopeAndInstanceIdToModel(model, newScope, newInstanceId);
    }
  }

  /**
   * returns the ScopeMaintainer for the given scope
   * Essentially, the scopeMaintainer manages a HashMap which holds the model instances
   * as long as the scopeMaintainer lives
   * 
   * @param scope the WDModelScopeType defining which scope
   * @return the scopeMaintainer
   */
  static private IMaintainScope getScopeMaintainer(WDModelScopeType scopeType)
  {
		if (scopeType==WDModelScopeType.NO_SCOPE)
			return null; //NO_SCOPE => no scope maintainer!
		if (testMode==true)
			return vmScopeMaintainer;
		IMaintainScope scopeMaintainer = null; 
  	if (scopeType==WDModelScopeType.APPLICATION_SCOPE)
      scopeMaintainer = (IMaintainScope)Utils.getCurrentApplication();
    else if (scopeType==WDModelScopeType.TASK_SCOPE)
			scopeMaintainer = (IMaintainScope)Utils.getCurrentTask();
		else if (scopeType==WDModelScopeType.SERVERSESSION_AT_LEAST_ONE_APP_SCOPE)
			scopeMaintainer = (IMaintainScope)Utils.getScopeMaintainer(IScope.SERVERSESSION_AT_LEAST_ONE_APP_SCOPE);
    else if (scopeType==WDModelScopeType.SERVERSESSION_SCOPE)
			scopeMaintainer = (IMaintainScope)Utils.getCurrentServerSession();
		else if (scopeType==WDModelScopeType.VM_SCOPE)
			scopeMaintainer = (IMaintainScope)Utils.getScopeMaintainer(IScope.VM_SCOPE);

	  if (scopeMaintainer==null)
			throw new WDRuntimeException(
				"No Scope Maintainer was found for WDModelScopeType: '" + scopeType.toString() + "'.");
		return scopeMaintainer; 
  }

	/**
	 * This method sets the environment in the test mode state.
	 * The test mode is intended for internal testing. It enables them to
	 * run without the complete Web Dynpro enviroment. (In detail: The session services
	 * are not used and the scopeMaintainer is always the VM scope.)
	 */
	public static void setTestMode(boolean testing) {
		testMode = testing;
	}

	/**
	 * This method sets the environment in the test mode state.
	 * The test mode is intended for internal testing. It enables them to
	 * run without the complete Web Dynpro enviroment. (In detail: The session services
	 * are not used and the scopeMaintainer is always the VM scope.)
	 */
	public static boolean getTestMode() {
		return testMode ;
	}

	private static boolean testMode;

  /**
   *  This method can not be used if scopeType==NO_SCOPE (since no scopeMaintainer exists for NO_SCOPE)  
   */
  static private synchronized IWDModel getOrCreateModelInstanceFromScopeMaintainer(Class modelClazz, String modelName, IMaintainScope scopeMaintainer, String modelInstanceId)
  {
    IWDModel model = null;
    if (scopeMaintainer==null) return null;
    try
    {
      IScope theScope = scopeMaintainer.getScope();
      model = (IWDModel)theScope.get(modelName);
      if (model==null) //create new model instance
      {
        model = getNewModelInstance( modelClazz , modelInstanceId);
				assignScopeAndInstanceIdToModel(model, WDModelScopeType.getModelScopeType(scopeMaintainer.getScopeType()), modelInstanceId);
        theScope.put(modelName, model);
      }
    }
    // FWE catch Exception only, as Throwables should not be caught in WD framework
    catch (Exception ex)
    {
      if (logger.beLogged(Severity.ERROR))
        logger.fatalT("failed to create or init instance of model ''{0}'' in scope {1} with instanceId ''{2}'': {3}",
          new Object[] { modelClazz.getName(), scopeMaintainer.getScopeType(), modelInstanceId, LoggingFormatter.extractCallstack(ex) });

      throw new WDRuntimeException(
        "failed to create or init instance of model ''{0}'' in scope {1} with instanceId ''{2}''",
        new Object[] {modelClazz.getName(), scopeMaintainer.getScopeType(), modelInstanceId }, 
        ex);
    }
    return model;
  }

	/**
	 * registers a model instance in the appropriate scope. This instance will be used by all model classes of the current application instance
	 * running in the same scope and modelInstanceId as the model instance being registered via this method. 
	 * 
	 * @param modelInstance the IWDModel instance to be registered
	 * @param scope WDModelScopeType (e.g. TASK_SCOPE, APPLIATION_SCOPE)
	 * @return previousModelInstance previously registered Model Instance or null if none previously registered
	 */
	static public IWDModel registerModelInstanceInScope(IWDModel modelInstance, WDModelScopeType scope)
	{
		if (modelInstance==null) return null;
		//First check if the scope is null and if yes, determine if a defaultScope was registered
		if (scope==null)
			scope = lookupDefaultScopeForModel(modelInstance.getClass());        
		// If no defaultScope was registered, then use the Default => NO_SCOPE
		if (scope==null)
			scope = WDModelScopeType.NO_SCOPE;        
		
		IMaintainScope scopeMaintainer = getScopeMaintainer(scope) ;
		if (scopeMaintainer==null) return null;

		IScope theScope = scopeMaintainer.getScope();
		String modelName = getModelKeyForStorageInScope(modelInstance.getClass(), modelInstance.getModelInstanceId());
		Object returnValue = theScope.put(modelName, modelInstance);
		if(returnValue instanceof IWDModel)	return (IWDModel)returnValue;
		return null;
	}

  /** */
  static private IWDModel getNewModelInstance(Class modelClazz, String modelInstanceId)
  	throws WDModelException
  {
    IWDModel instance = null;
    if (IWDModel.class.isAssignableFrom(modelClazz))
    {
      try
      {
        instance = (IWDModel)modelClazz.newInstance();
      }
      catch(Exception ex)
      { // simplest form of exception handling
				if (logger.beLogged(Severity.ERROR))
					logger.fatalT("failed to create instance of model ''{0}'': {1}", new Object[] {modelClazz.getName(), LoggingFormatter.extractCallstack(ex)});
				throw new WDModelException("failed to create instance of model ''{0}''", new Object[] {modelClazz.getName()}, ex);
      }
    }
    else
			throw new WDModelException(
				WDModelException.MODEL_NOT_COMPATIBLE_FOR_CREATION_BY_WDMODELFACTORY,
					new Object[]{modelClazz.getName()});

    return instance;
  }

  /** */
  static private String getModelKeyForStorageInScope(Class modelClazz, String modelInstanceId)
  {
    String modelKey = modelClazz.getName();
    if ((modelInstanceId != null) && (modelInstanceId.length() > 0 ))
    {
      StringBuffer buf = new StringBuffer(modelKey.length() + modelInstanceId.length() + 1 );
      buf.append(modelKey).append("_").append(modelInstanceId);
      modelKey = buf.toString();
    }
    return modelKey;
  }
  
	/**
	 * if the scope parameter is null, then do not change the scope,
	 * this behaviour is required for default initialization 
	 */
	static private synchronized void assignScopeAndInstanceIdToModel(Object model, WDModelScopeType newScope, String newInstanceId)
	{
		if (model==null) return;
		WDModelScopeType oldScope;
		String oldInstanceId;
		if ( model instanceof IMaintainableModel)
		{
			AbstractModelMaintainer maintainer =
				((IMaintainableModel)model).modelMaintainer();
			if (maintainer instanceof IScopeTypeMaintainable && newScope!=null)
			{
				oldScope = ((IScopeTypeMaintainable)maintainer).getScope();
				if (newScope!=oldScope)
				{
					((IScopeTypeMaintainable)maintainer).setScope(newScope);
				}
			}
			if (maintainer instanceof IInstanceIdMaintainable)
			{
				oldInstanceId = ((IInstanceIdMaintainable)maintainer).getModelInstanceId();
				if ((newInstanceId!=oldInstanceId) ||
						((newInstanceId!=null) && !newInstanceId.equals(oldInstanceId)))
				{
					((IInstanceIdMaintainable)maintainer).setModelInstanceId(newInstanceId);
				}
			}
		}
    // TODO let Model implement IMaintainableModel and thereby avoid 'else' branch?
		else if ( model instanceof Model)
		{
			Model wdModel = (Model)model;
			oldScope=wdModel.getModelScope();
			oldInstanceId=wdModel.getModelInstanceId();
			if (newScope!=null && newScope!=oldScope)
			{
				wdModel.setModelScopeInternal(newScope);
			}
			if ((newInstanceId!=oldInstanceId) ||
			    ((newInstanceId!=null) && !newInstanceId.equals(oldInstanceId)))
			{
				wdModel.setModelInstanceIdInternal(newInstanceId);
			}
		}
	}
	
	
	private static final String KEY_WD_DEBUG_MAP 								 = "wd.tc.webdynpro.services.WDDebugMap";
	private static       String guiHostForGlobalBackendDebugging = null;
 /**
  * 
  * @return Map the map containing entries with application name as key, and guiHost as value
  */
	private static Map getWDBackendDebugMap()
	{
		IMaintainScope vmMaintainer = getScopeMaintainer(WDModelScopeType.VM_SCOPE);
		if(vmMaintainer==null) return null;
		IScope vmScope = vmMaintainer.getScope();
		if(vmScope==null) return null;

		Map appToGuiHostMap = (Map)vmScope.get(KEY_WD_DEBUG_MAP);
		if(appToGuiHostMap==null)
		{
			appToGuiHostMap = new HashMap(7);
			vmScope.put(KEY_WD_DEBUG_MAP, appToGuiHostMap);
		}
		return appToGuiHostMap;
	}
	
 /**
  * allows enabling or disabling backend debugging for a specific application on a specific host
  * if applicationName is null, then debugging is enabled for all applications in the vm
  * 
  * @param applicationName the Name of the Application for which to enable/disable Debugging 
  * @param guiHost the name of the Computer hosting the Client application for Debugging
  * @param turnDebugOn true if debugging should be enabled, false if debugging should be disabled
  * @return the previous guiHost associated with this application
  */
	public static String setBackendDebugModeForApplication(String applicationName, String guiHost, boolean turnDebugOn)
	{
		if (applicationName==null) 
		{
			String oldValue = guiHostForGlobalBackendDebugging; 
			if (turnDebugOn)
				guiHostForGlobalBackendDebugging = guiHost;
			else 
				guiHostForGlobalBackendDebugging = null;
			return oldValue;
		} 
		Map appToGuiHostMap = getWDBackendDebugMap();
		if(appToGuiHostMap==null) return null;
		if (turnDebugOn)
			return (String)appToGuiHostMap.put(applicationName, guiHost);
		return (String)appToGuiHostMap.remove(applicationName);
	}


	/**
	 * returns the name of the computer hosting the Debugger Client Process if debugging for the given application is enabled.
   * if applicationName is null, then the guiHost for global debugging of all applications in the vm is returned
	 * returns null if debugging is disabled for the given application
	 * 
	 * @param applicationName the Name of the Application for which to check if Debugging is enabled
	 * @return guiHost the name of the Computer hosting the Client application for Debugging
	 */
	public static String getGuiHostIfBackendDebuggingForApplicationIsEnabled(String applicationName)
	{
		if (applicationName==null) return guiHostForGlobalBackendDebugging;
		Map appToGuiHostMap = getWDBackendDebugMap();
		if(appToGuiHostMap==null) return null;
		String guiHost = (String)appToGuiHostMap.get(applicationName);
		if(guiHost==null || guiHost.length()==0) return null;
		return guiHost;
	}

	/**
	 * looksup the default scope of a model
	 * 
	 * @param modelClazz the class of the model, for which to look up the default scope
	 * @return scope the default scope to use for instances of the given model (specified by class of model)
	 */
	public static synchronized WDModelScopeType lookupDefaultScopeForModel(Class modelClazz)
	{
		return (WDModelScopeType)defaultScopeForModelClazz.get(modelClazz.getName());
	}
	
	/**
	 * registers the default scope of a model in the ModelFactory 
	 * 
	 * @param modelClazz the class of the model, for which to set the default scope
	 * @param scope the default scope to use for instances of the given model (specified by class of model)
	 */
	public static synchronized void registerDefaultScopeForModel(Class modelClazz, WDModelScopeType scope)
	{
		// use the Name of the class as key, not the class itself 
		// otherwise via redeployment, the class and its classloader are not released
		defaultScopeForModelClazz.put(modelClazz.getName(), scope);
	}
	
  /** Maintainer instance for managing models in VM scope */
  private static IMaintainScope vmScopeMaintainer = Utils.getScopeMaintainer(IScope.VM_SCOPE);
  private static Map defaultScopeForModelClazz = Collections.synchronizedMap(new HashMap());

}
