/* Title:        Web Dynpro Java Runtime: Message Enum
 * Description:  enumeration of message types
 * Copyright:    Copyright (c) 2002
 * Company:      SAP AG
 * @author SAP
 * @version      $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/WDMessage.java#2 $
 */
package com.sap.tc.webdynpro.progmodel.api;

import com.sap.tc.webdynpro.services.message.WDMessageType;

/**
 * Base class for all kinds of messages used in conjunction with the 
 * Web Dynpro {@link IWDMessageManager message manager}. </p>
 * 
 * The message manager exposes APIs for reporting exceptions, warnings and success messages. 
 * There are 2 categories of methods. First there exists simple String based methods, where one
 * can simply report messages by passing a parameter of type string. This string is used 
 * as the message text without any Web Dynpro services involved. 
 * Second there are methods that accept a WDMessage instance. This instance  
 * contains necessary information for additional Web Dynpro services like 
 * CATT, Help, etc. </p>
 * 
 * The typical usage of WDMessage is as enumeration type.<br>
 *  
 * <code>
 * 	
 * public class MyMessages extends WDMessage
 * {
 *  public static final IWDMessage WARNING_TABLE_CELL = new WDMessage( "com.sap.test.IMessageTest" , "WarningTableCell", WDMessageType.WARNING);
 *
 *  public static final IWDMessage DATE_IS_IN_FUTURE = new WDMessage( "com.sap.test.IMessageTest" , "DateIsInFuture", WDMessageType.ERROR);
 *
 *  public static final IWDMessage DESIRED_E_MAIL = new WDMessage( "com.sap.test.IMessageTest" , "DesiredEMail", WDMessageType.WARNING);
 *
 *  public static final IWDMessage MISSING_INPUT = new WDMessage( "com.sap.test.IMessageTest" , "MissingInput", WDMessageType.ERROR);
 *
 *  public static final IWDMessage SOME_SUCCESS_MESSAGE = new WDMessage( "com.sap.test.IMessageTest" , "SomeSuccessMessage", WDMessageType.STANDARD);
 *
 * </code>
 * 
 * In a later call to IWDMessageManager one may just use this enumeration value combined with optional parameters. 
 * Parameters are combined with the message text in the same way as for {@link java.text.MessageFormat}: </p> 
 * 
 * <code>
 * 
 * IWDMessageManager ex = wdThis.wdGetAPI().getComponent().getMessageManager();
 * ex.reportMessage(MyComponentMessages.Message1, new Object[] {"Some Parameter"} );
 * ex.reportMessage(MyComponentMessages.Message2, null );
 * 
 * </code>
 * 
 * @author SAP
 * @SAPWebDynproPart 1
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/WDMessage.java#2 $
 */
public class WDMessage implements IWDMessage {
		
	public WDMessage(String compName, String msgKey , WDMessageType  type)
	{
		this.msgKey = msgKey;
		this.type = type;
		this.compName=compName;
    this.msgText = null;
	}
  /**
   * Create an instance of a message item. This method is using parameter <code>message</code> as human readable text 
   * for display in the message area. The parameter <code>key</code> must be an language independent message key used
   * for eCATT recording or xml clients. <p>
   * NOTE: this methode does not use parameter <code>key</code> as a key in any resource bundle, but assume that the
   * user of this method guarantees that <code>key</code> is unique language neutral description for <code>message</code>
   * 
   * @param comp is the componenten this message belongs to. 
   * @param msgKey is the key of this message.
   * @param type is the type of this message.
   * @param Message is the human readable message text,
   */
  public WDMessage(IWDComponent comp, String msgKey , WDMessageType  type, String message)
  {
    this(comp.getQualifiedName(),msgKey, type);
    this.msgText = message;
  }
  
  
	
	/**
	 * Type of this message
	 */
	private WDMessageType type;
	
	/**
	 * locale independent resource key for this message 
	 */
	private String msgKey;

	/**
	 * locale independent component name owing the message declaration
	 */
	private String compName;
  
  /**
   * Text for this message. May be null. In this case msgKey is a key for the component's resource bundle
   * 
   */
  private String msgText;
		
	/**
	 * Method getMessageKey.
	 * @return the key of this item's message
	 */
    final public String getMessageKey ()
	{
   	  return msgKey;			
	}
  
  /**
   * Get the human readable display text for this message. This may be null. In  that case use  
   * @see WDMessage#getMessageKey() as key in the components resource bundle.
   * @return the message text or null
   */
  final public String getMessageText()
  {
    return msgText;
  }
		

	/**
	 * @return the component name this message belongs to.
	 */
	final public String getCompName() {
		return compName;
	}


	/**
	 * @return the type of this message.
	 */
	public WDMessageType getType() {
			return type;
		}


}
